/* Copyright (C) 2002-2005 RealVNC Ltd.  All Rights Reserved.
 * Copyright (C) 2011 D. R. Commander.  All Rights Reserved.
 * Copyright (C) 2015-2024 m-privacy GmbH.  All Rights Reserved.
 * 
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 */

//
// MPCompressor compresses RGB input into an MP image and stores it in
// an underlying MemOutStream
//

#ifndef __RFB_MPCOMPRESSOR_H__
#define __RFB_MPCOMPRESSOR_H__

#include <rdr/MemOutStream.h>
#include <rfb/PixelFormat.h>
#include <rfb/Rect.h>

#include <rfb/MP.h>

#include <wels/codec_api.h>
#include <zstd.h>

#if !defined(__APPLE__)
#include <jxl/encode.h>
#include <jxl/encode_cxx.h>
#ifndef WIN32
#include <jxl/thread_parallel_runner.h>
#endif
#endif

#define MP_MAX_ENCODERS 5
#define MP_HEADER_SIZE 2

struct jpeg_compress_struct;
struct MP_JPEG_ERROR_MGR;
struct MP_JPEG_DEST_MGR;

namespace rfb {

	class MPCompressor : public rdr::MemOutStream {

	public:

		MPCompressor(int bufferLen = 128*1024);
		virtual ~MPCompressor();

		bool compress(const rdr::U8 *buf, int stride, const Rect& r, const PixelFormat& pf, int mpLevel, int mpCompression);

		inline rdr::U8* getstart() { return memBuffer.data; }

		virtual inline void overrun(size_t needed, int key=0, QPrio prio=QPRIOMEDIUM) {
			MemOutStream::overrun(needed, key, prio);
		}

		static void setCpuPressure(rdr::U8 value);
		static void setSaveBandwidth(bool value);
		static void setSaveBandwidthHigh(bool value);
		static rdr::U8 getPressureLevel() {return pressureLevel;}
#if !defined(__APPLE__)
		static void setSupportsTightMPJpegXL(bool value);
#endif
		rdr::U64 totalBytes[MP_COMPRESSION_MAX+1];
		rdr::U64 totalCBytes[MP_COMPRESSION_MAX+1];
		int minCompressed[MP_COMPRESSION_MAX+1];
		int maxCompressed[MP_COMPRESSION_MAX+1];

	private:
		static rdr::U8 cpuPressure;
		static rdr::U8 pressureLevel;
		static bool saveBandwidth;
		static bool saveBandwidthHigh;
#if !defined(__APPLE__)
		static bool clientSupportsTightMPJpegXL;
#endif
		unsigned compressionCount[MP_COMPRESSION_MAX+1];
		unsigned optimizationCount[MP_COMPRESSION_MAX+1];

		SEncParamBase welsParam;
		int welsVideoFormat;
		int welsComplexity;
		SSourcePicture welsPic;

		struct jpeg_compress_struct *jpegCinfo;
		struct MP_JPEG_ERROR_MGR *jpegErr;
		struct MP_JPEG_DEST_MGR *jpegDest;

#if !defined(__APPLE__)
		JxlBasicInfo jpegxlBasicInfo;
		JxlEncoder* jpegxlEnc;
#ifndef WIN32
		void* jpegxlRunner;
#endif
		JxlColorEncoding jpegxlColorEncoding;
		JxlEncoderFrameSettings * jpegxlFrameSettings;
		JxlPixelFormat jpegxlPixelFormat;
		JxlEncoderStatus jpegxlStatus;
#endif

		ZSTD_CCtx *zstdCctx;
	};

} // end of namespace rfb

#endif
