/* Copyright 2017-2021 m-privacy GmbH.
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 */

#ifndef WIN32 /* This is only for the server, so we don't need to build it at all for Windows */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/inotify.h>
#include <tgpro_environment.h>

#include <rfb/LogWriter.h>

#include "SAutotransferDirWatcher.h"

#define INOTIFY_EVENT_SIZE (sizeof (struct inotify_event))
#define INOTIFY_BUFFER_LENGTH (1024 * (INOTIFY_EVENT_SIZE + 16))

static rfb::LogWriter vlog("SAutotransferDirWatcher");

// We use arrays for these static variables because every implementation should have new locks and condtions for its own queue/s.
MUTEX_TYPE SAutotransferDirWatcher::processingQueueLocks[MAX_THREADS];
TGVNC_CONDITION_TYPE SAutotransferDirWatcher::filesWaitingConditions[MAX_THREADS];
MUTEX_TYPE SAutotransferDirWatcher::filesWaitingConditionLocks[MAX_THREADS];

MUTEX_TYPE SAutotransferDirWatcher::threadCountLock;
unsigned int SAutotransferDirWatcher::threadCount = 0;

bool SAutotransferDirWatcher::clientSupportsAutotransfer = true;

SAutotransferDirWatcher::SAutotransferDirWatcher() {
	MUTEX_INIT(&threadCountLock);
}

SAutotransferDirWatcher::~SAutotransferDirWatcher()
{
	MUTEX_DESTROY(&threadCountLock);
	if (started) {
		TGVNC_CONDITION_DESTROY(&filesWaitingConditions[myThread]);
		MUTEX_DESTROY(&filesWaitingConditionLocks[myThread]);
		MUTEX_DESTROY(&processingQueueLocks[myThread]);
	}
}

void SAutotransferDirWatcher::start()
{
	dirToWatch = getDirToWatch();

	if (!file_exists(dirToWatch)) {
		vlog.error("%s: Directory '%s' doesn't exist. Can't watch it with inotify!", __func__, dirToWatch);
		return;
	}

	MUTEX_LOCK(&threadCountLock);
	myThread = threadCount++;
	MUTEX_UNLOCK(&threadCountLock);

	if (myThread >= MAX_THREADS) {
		vlog.error("%s: reached max. number of threads for DirWatchers. Not starting a new one.", __func__);
		return;
	}
	vlog.debug("%s: starting a new SAutotransferDirWatcher thread: %i", __func__, myThread);

	MUTEX_INIT(&filesWaitingConditionLocks[myThread]);
	TGVNC_CONDITION_INIT(&filesWaitingConditions[myThread]);
	MUTEX_INIT(&processingQueueLocks[myThread]);

	// The watches dirToWatch and adds new files to the queue.
	THREAD_ID inotifyWatchDirThreadId;
	THREAD_CREATE(inotifyWatchDirThread, inotifyWatchDirThreadId, (void *) this);
	THREAD_SET_NAME(inotifyWatchDirThreadId, "tg-inotifyWatchDir");

	// This thread watches a std::queue with a lock. The files that inotify finds are added to this queue sequentially. Runs processFile on every file in the queue
	THREAD_ID watchQueueThreadId;
	THREAD_CREATE(watchQueueThread, watchQueueThreadId, (void *) this);
	THREAD_SET_NAME(watchQueueThreadId, "tg-inotifyWatchQueue");

	started = true;
}

THREAD_FUNC SAutotransferDirWatcher::inotifyWatchDirThread(void* _thiss)
{
	SAutotransferDirWatcher* thiss = (SAutotransferDirWatcher*) _thiss;

	vlog.debug("%s: Starting to watch '%s' with inotify...", __func__, thiss->dirToWatch);
	int fd;
	int wd;

	fd = inotify_init();
	if (fd < 0) {
		vlog.error("inotify_init error %i: %s", errno, strerror(errno));

	}

	vlog.verbose("Adding watch");
	wd = inotify_add_watch(fd, thiss->dirToWatch, IN_CLOSE_WRITE | IN_MOVED_TO);
	if (wd < 0) {
		vlog.error("%s: Failed to add watch for %s. Error %i: %s. Stopping autotransfer functionality.", __func__, thiss->dirToWatch, errno, strerror(errno));
		THREAD_EXIT(THREAD_NULL);
	}

	char buffer[INOTIFY_BUFFER_LENGTH];
	while(42 == 42) {
		int length, i = 0;

		length = read(fd, buffer, INOTIFY_BUFFER_LENGTH);
		vlog.verbose("%s: Read %i bytes from inotify buffer (max. capacity: %ld)", __func__, length, INOTIFY_BUFFER_LENGTH);

		if (length < 0) {
			vlog.error("%s: read error %i on inotify fd. Stopping autotransfer functionality.", __func__, errno);
			THREAD_EXIT(THREAD_NULL);
		}

		while (i < length) {
			struct inotify_event* event = (struct inotify_event*) &buffer[i];
			if (event->len) {
				if (event->mask & IN_ISDIR) {
					vlog.error("%s: Directories not supported by the autotransfer feature. Ignoring '%s'", __func__, event->name);
				} else {
					if (event->mask & IN_CLOSE_WRITE) {
						vlog.verbose("%s: The file '%s' was 'close_written'.", __func__, event->name);
					} else if (event->mask & IN_MOVED_TO) {
						vlog.verbose("%s: The file '%s' was 'in_moved_to'.", __func__, event->name);
					}
					char* filename = (char*) malloc(4096);
					if (!filename) {
						vlog.error("%s: Failed to allocate memory for filename. Can't add to processing queue", __func__);
					} else {
						snprintf(filename, 4095, "%s/%s", thiss->dirToWatch, event->name);
						filename[4095] = '\0';
						if (!clientSupportsAutotransfer) {
							vlog.debug("%s: client SupportAutotransfer is set to false. Ignoring inotify event of file '%s'", __func__, filename);
						} else {
							thiss->addToProcessQueue(filename);
						}
					}
				}
			} else {
				vlog.error("%s: something strange is happening: the length of the inotify event is 0", __func__);
			}
			i += INOTIFY_EVENT_SIZE + event->len;
		}
	}

	vlog.verbose("Removing inotify watch");
	inotify_rm_watch(fd, wd);
	close(fd);

	vlog.debug("Leaving thread that watches autotransfer dir");
}

THREAD_FUNC SAutotransferDirWatcher::watchQueueThread(void* _sDirWatcher)
{
	vlog.debug("%s: Starting thread to watch the queue of files to autotransfer", __func__);
	SAutotransferDirWatcher* sDirWatcher = (SAutotransferDirWatcher*) _sDirWatcher;
	while (42 == 42) {
		bool goOn = true;
		while (goOn) {
			char* autotransferFile = NULL;
			MUTEX_LOCK(&processingQueueLocks[sDirWatcher->myThread]);
			if (!sDirWatcher->processingQueue.empty()) {
				autotransferFile = sDirWatcher->processingQueue.front();
				sDirWatcher->processingQueue.pop();
			}
			MUTEX_UNLOCK(&processingQueueLocks[sDirWatcher->myThread]);
			if (autotransferFile) {
				vlog.verbose("%s: Are there more files in processing queue? Yup! Processing '%s'", __func__, autotransferFile);
				sDirWatcher->processFile(autotransferFile);
			} else {
				// Keep going until there are no more files to autotransfer
				vlog.verbose("%s: Are there more files in processing queue? Nope!", __func__);
				goOn = false;
			}
			usleep(100 * 1000);
		}
		MUTEX_LOCK(&filesWaitingConditionLocks[sDirWatcher->myThread]);
		vlog.debug("%s: waiting for somebody to fill the processing queue...", __func__);
		TGVNC_CONDITION_WAIT(&filesWaitingConditions[sDirWatcher->myThread], &filesWaitingConditionLocks[sDirWatcher->myThread]);
		MUTEX_UNLOCK(&filesWaitingConditionLocks[sDirWatcher->myThread]);
	}
	vlog.debug("%s: Exiting thread that watched the queue of files to autotransfer.", __func__);
	THREAD_EXIT(THREAD_NULL);
}

void SAutotransferDirWatcher::addToProcessQueue(char* filename)
{
	bool addedToProcessingQueue;
	MUTEX_LOCK(&processingQueueLocks[myThread]);
	if (!processingQueue.empty() && !strcmp(processingQueue.back(), filename)) {
		// Do not add if it is the same as the last added file (because some inotify events happen one right after the other)
		addedToProcessingQueue = false;
	} else {
		processingQueue.push(filename);
		addedToProcessingQueue = true;
	}
	MUTEX_UNLOCK(&processingQueueLocks[myThread]);

	// We really don't need to lock the queue while we log
	if (addedToProcessingQueue) {
		TGVNC_CONDITION_SEND_SIG(&filesWaitingConditions[myThread]);
		vlog.verbose("%s: Added file '%s' to handling queue. The 'watch queue' thread will take care of it as soon as possible", __func__, filename);
	} else {
		vlog.verbose("%s: File '%s' NOT added to handling queue. because it was just added.", __func__, filename);
	}
}

#endif /* ifndef WIN32 */
