/* Copyright 2017-2021 m-privacy GmbH.
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 */

#ifndef __SAUTOTRANSFER_DIR_WATCHER_H__
#define __SAUTOTRANSFER_DIR_WATCHER_H__

#ifndef WIN32

#include <queue>

#include <rdr/mutex.h>
#include <rfb/Exception.h>

#define MAX_THREADS 100

/**
 * "abstract" class that watches the given directory and (sequentially) runs
 * processFile on every file that appears in it. It doesn't process directories!
 *
 * inotify watches the getDirToWatch() dir and stores files that appear in a queue
 * using a mutex to make sure the threads work nicely with eatch other. The queue
 * is constantly being watched by another thread and if something appears in it,
 * it calls processFile() on each one of the files.
 */
class SAutotransferDirWatcher {
public:
	SAutotransferDirWatcher();
	~SAutotransferDirWatcher();

	void start();

	static bool clientSupportsAutotransfer;
	/**
	 * The child classes *must* override this. It represents
	 * the dir as a string where inotify should be run on.
	 */
	virtual const char* getDirToWatch() {
		throw rfb::NotImplementedException("getDirToWatch() not implemented in SAutotransferDirWatcher's child.");
		return NULL;
	};
	/**
	 * Action that is taken for every file that inotify
	 * detects in dirToWatch
	 */
	virtual void processFile(char* filename) {
		throw rfb::NotImplementedException("processFile(char* filename) not implemented in SAutotransferDirWatcher's child.");
	}; // If the file is allowed to be transfered (or has been somehow cleaned, process file needs to move the file to /home/user/.autotransfer-now/$USER, which will take care of copying it o the client)

protected:
	void addToProcessQueue(char* filename); // In some cases, the children might want to add something to the queue (e.g. if a MIME check fails)
	static unsigned int threadCount;
	unsigned int myThread;

	static MUTEX_TYPE threadCountLock;

private:
	bool started = false;
	const char* dirToWatch;

	static MUTEX_TYPE processingQueueLocks[MAX_THREADS];
	std::queue<char*> processingQueue;

	static TGVNC_CONDITION_TYPE filesWaitingConditions[MAX_THREADS];
	static MUTEX_TYPE filesWaitingConditionLocks[MAX_THREADS];

	static THREAD_FUNC watchQueueThread(void* autotransfer);

	static THREAD_FUNC inotifyWatchDirThread(void* autotransferAbstract);
};

#endif /* ifndef WIN32 */
#endif /* __SAUTOTRANSFER_DIR_WATCHER_H__ */
