/* Copyright (C) 2002-2005 RealVNC Ltd.  All Rights Reserved.
 * Copyright 2015-2021 m-privacy GmbH, Berlin
 * 
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 */

#include <rdr/HexOutStream.h>
#include <rdr/Exception.h>

#define HEXKEY 5

using namespace rdr;

static rfb::LogWriter vlog("HexOutStream");

static inline size_t min(size_t a, size_t b) {return a<b ? a : b;}

HexOutStream::HexOutStream(OutStream& os, size_t buflen)
: out_stream(os) {
  classLog = &vlog;
}

HexOutStream::~HexOutStream() {
  resetClassLog();
}

char HexOutStream::intToHex(int i) {
  if ((i>=0) && (i<=9))
    return '0'+i;
  else if ((i>=10) && (i<=15))
    return 'a'+(i-10);
  else
    throw rdr::Exception("intToHex failed");
}

char* HexOutStream::binToHexStr(const char* data, size_t length) {
  char* buffer = new char[length*2+1];
  for (size_t i=0; i<length; i++) {
    buffer[i*2] = intToHex((data[i] >> 4) & 15);
    buffer[i*2+1] = intToHex((data[i] & 15));
    if (!buffer[i*2] || !buffer[i*2+1]) {
      delete [] buffer;
      return 0;
    }
  }
  buffer[length*2] = 0;
  return buffer;
}


void
HexOutStream::writeBuffer(struct queueBuffer * inBuffer, QPrio prio) {
  struct queueBuffer * outBuffer;
  unsigned sent = 0;
  unsigned requested;
  /* need multiple of 2 */
  unsigned maxBuf = (out_stream.getMaxBufferSize() / 2) * 2;

  while (sent < inBuffer->used) {
    requested = min( (inBuffer->used - sent) * 2, maxBuf);
    outBuffer = out_stream.getQueueBuffer(requested, prio);
    if (!outBuffer) {
      throw rdr::Exception("HexOutStream::writeBuffer() failed to get buffer");
    }

    for (unsigned i=0; i < requested / 2; i++) {
      outBuffer->data[(sent+i)*2] = intToHex((inBuffer->data[i] >> 4) & 0xf);
      outBuffer->data[(sent+i)*2+1] = intToHex(inBuffer->data[i] & 0xf);
    }
    sent += requested / 2;
    outBuffer->used = requested;
    out_stream.submitQueueBuffer(outBuffer, outBuffer, HEXKEY, prio);
  }
  returnQueueBuffer(inBuffer, prio);
}

void
HexOutStream::flush(int key, QPrio prio, bool wait) {
  struct queueBuffer * buffer;

//  if(!check_key(key, __func__))
//    return;
  if(!check_prio(prio, __func__))
    return;

  for (int q = QPRIOMAX; q >= prio; q--) {
    while (( buffer = popBuffer((QPrio) q) )) {
      writeBuffer(buffer, (QPrio) q);
    }
  }
  out_stream.flush(HEXKEY, prio);
}
