/* Copyright (C) 2002-2005 RealVNC Ltd.  All Rights Reserved.
 * Copyright (C) 2011 D. R. Commander.  All Rights Reserved.
 * Copyright (C) 2015-2021 m-privacy GmbH
 * 
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 */

#include <stdio.h>

#include <rdr/ZlibOutStream.h>
#include <rdr/Exception.h>
#include <rfb/LogWriter.h>

#include <zlib.h>

#undef ZLIBOUT_DEBUG

static rfb::LogWriter vlog("ZlibOutStream");

#define ZLIBKEY 4

using namespace rdr;

enum { DEFAULT_BUF_SIZE = 16384, MAX_BUF_SIZE = (1024 * 16384) };

ZlibOutStream::ZlibOutStream(OutStream* os, size_t bufSize_, int compressLevel)
  : underlying(os), compressionLevel(compressLevel), newLevel(compressLevel),
    bufSize(bufSize_ ? bufSize_ : DEFAULT_BUF_SIZE), offset(0)
{
  zs = new z_stream;
  zs->zalloc    = Z_NULL;
  zs->zfree     = Z_NULL;
  zs->opaque    = Z_NULL;
  zs->next_in   = Z_NULL;
  zs->avail_in  = 0;
  if (deflateInit(zs, compressLevel) != Z_OK) {
    delete zs;
    throw Exception("ZlibOutStream: deflateInit failed");
  }
  classLog = &vlog;
  classMemLog = &vlog;
  vlog.debug("init with bufSize %zu", bufSize);
  /* sanity check */
  if (bufSize > MAX_BUF_SIZE) {
    vlog.error("reducing insane bufSize %zu to %u", bufSize, DEFAULT_BUF_SIZE);
    if(vlog.getLevel() >= vlog.LEVEL_DEBUG)
      savebacktrace(__func__);
    bufSize = DEFAULT_BUF_SIZE;
  }
  if (bufSize > memBuffer.size) {
    U8 * newdata = (U8 *) realloc(memBuffer.data, bufSize);
    if (newdata) {
      memBuffer.size = bufSize;
      memBuffer.data = newdata;
    } else {
      vlog.error("realloc failed, reducing bufSize from %zu to %u", bufSize, memBuffer.size);
      bufSize = memBuffer.size;
    }
  }
}

ZlibOutStream::~ZlibOutStream()
{
  try {
    flush();
  } catch (Exception&) {
  }
  deflateEnd(zs);
  delete zs;
  resetClassLog();
  resetClassMemLog();
}

void ZlibOutStream::setUnderlying(OutStream* os)
{
  underlying = os;
}

void ZlibOutStream::setCompressionLevel(int level)
{
  if (level < -1 || level > 9)
    level = -1;                 // Z_DEFAULT_COMPRESSION

  newLevel = level;
}

size_t ZlibOutStream::length(int key, QPrio prio)
{
  return offset + memBuffer.used;
}

int ZlibOutStream::bufferUsage(int key, QPrio prio) {
  return underlying->bufferUsage(ZLIBKEY);
}

void ZlibOutStream::flush(int key, QPrio prio, bool wait)
{
//  if(!check_key(key, __func__))
//    return;
  if(!check_prio(prio, __func__))
    return;

  if (!memBuffer.data) {
    vlog.error("flush() called with NULL memBuffer.data");
  }

  checkCompressionLevel(prio);

  zs->next_in = memBuffer.data;
  zs->avail_in = memBuffer.used;

#ifdef ZLIBOUT_DEBUG
  vlog.debug("flush: avail_in %d",zs->avail_in);
#endif

  // Force out everything from the zlib encoder
  deflate(Z_SYNC_FLUSH, prio);

  offset += memBuffer.used;
  memBuffer.used = 0;
}

void ZlibOutStream::overrun(size_t needed, int key, QPrio prio)
{
#ifdef ZLIBOUT_DEBUG
  vlog.debug("overrun");
#endif

  if ((U32) needed > memBuffer.size)
    MemOutStream::overrun(needed - memBuffer.used);

  checkCompressionLevel();

  while (memBuffer.size - memBuffer.used < (U32) needed) {
    zs->next_in = memBuffer.data;
    zs->avail_in = memBuffer.used;

    deflate(Z_NO_FLUSH);

    // output buffer not full

    if (zs->avail_in == 0) {
      offset += memBuffer.used;
      memBuffer.used = 0;
    } else {
      // but didn't consume all the data?  try shifting what's left to the
      // start of the buffer.
      vlog.info("z out buf not full, but in data not consumed");
      memmove(memBuffer.data, zs->next_in, memBuffer.data + memBuffer.used - zs->next_in);
      offset += zs->next_in - memBuffer.data;
      memBuffer.used -= zs->next_in - memBuffer.data;
    }
  }
}

void ZlibOutStream::deflate(int flush, QPrio prio)
{
  int rc;
  struct queueBuffer * outBuffer;
  size_t chunk;

  if (!underlying)
    throw Exception("ZlibOutStream: underlying OutStream has not been set");

  if ((flush == Z_NO_FLUSH) && (zs->avail_in == 0))
    return;

  do {
    outBuffer = underlying->getQueueBuffer(zs->avail_in, prio);
    if (outBuffer) {
//    vlog.debug("deflate(): got outBuffer %u with size %u, prio %u", outBuffer->number, outBuffer->size, prio);
      zs->next_out = outBuffer->data;
      zs->avail_out = outBuffer->size;
    } else {
      zs->next_out = underlying->getptr(1);
      zs->avail_out = chunk = underlying->avail();
    }

#ifdef ZLIBOUT_DEBUG
    vlog.debug("calling deflate, avail_in %d, avail_out %d",
               zs->avail_in,zs->avail_out);
#endif

    rc = ::deflate(zs, flush);
    if (rc < 0) {
      // Silly zlib returns an error if you try to flush something twice
      if ((rc == Z_BUF_ERROR) && (flush != Z_NO_FLUSH))
      {
        if (outBuffer)
          underlying->returnQueueBuffer(outBuffer, prio);
        break;
      }

      throw Exception("ZlibOutStream: deflate failed");
    }

#ifdef ZLIBOUT_DEBUG
    vlog.debug("zos: after deflate: %d bytes\n",
            outBuffer ? zs->next_out - (outBuffer->data + outBuffer->used) : zs->next_out - underlying->getptr(1));
#endif

    if (outBuffer) {
      outBuffer->used = zs->next_out - outBuffer->data;
      while (!underlying->submitQueueBuffer(outBuffer, outBuffer, ZLIBKEY, prio)) {
        vlog.debug("deflate(): submitQueueBuffer() of buffer %u with used %u, prio %u failed - sleep and retry!", outBuffer->number, outBuffer->used, prio);
        Sleep(100);
      }
    } else {
      underlying->setptr(chunk - zs->avail_out);
    }
  } while (zs->avail_out == 0);
}

void ZlibOutStream::checkCompressionLevel(QPrio prio)
{
  int rc;

  if (newLevel != compressionLevel) {
#ifdef ZLIBOUT_DEBUG
    vlog.debug("change: avail_in %d",zs->avail_in);
#endif

    // zlib is just horribly stupid. It does an implicit flush on
    // parameter changes, but the flush it does is not one that forces
    // out all the data. And since you cannot flush things again, we
    // cannot force out our data after the parameter change. Hence we
    // need to do a more proper flush here first.
    deflate(Z_SYNC_FLUSH, prio);

    rc = deflateParams (zs, newLevel, Z_DEFAULT_STRATEGY);
    if (rc < 0) {
      // The implicit flush can result in this error, caused by the
      // explicit flush we did above. It should be safe to ignore though
      // as the first flush should have left things in a stable state...
      if (rc != Z_BUF_ERROR)
        throw Exception("ZlibOutStream: deflateParams failed");
    }

    compressionLevel = newLevel;
  }
}
