/* Copyright (C) 2002-2005 RealVNC Ltd.  All Rights Reserved.
 * Copyright (C) 2011 D. R. Commander.  All Rights Reserved.
 * Copyright 2014-2019 Pierre Ossman for Cendio AB
 * Copyright (C) 2014-2023 m-privacy GmbH.  All Rights Reserved.
 * 
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 */
#include <rfb/Exception.h>
#include <rfb/encodings.h>
#include <rfb/ledStates.h>
#include <rfb/clipboardTypes.h>
#include <rfb/ClientParams.h>

#include <rfb/LogWriter.h>

using namespace rfb;

static LogWriter vlog("ClientParams");

ClientParams::ClientParams()
  : majorVersion(0), minorVersion(0),
    compressLevel(2), qualityLevel(-1), fineQualityLevel(-1),
    subsampling(subsampleUndefined),
    supportsTightMP(false), supportsTightMPSolid(false), supportsTightMPJpegXL(false),
    mpLevel(-1), mpCompression(MP_COMPRESSION_DEFAULT),
    paPort(4713), supportsUTF8(false), supportsSound(true), supportsPrint(true),
    width_(0), height_(0), name_(0),
    cursorPos_(0, 0), ledState_(ledUnknown)
{
  vlog.debug("Creating one instance of ConnParams, supportsUTF8 is %i", supportsUTF8);
  setName("");

  cursor_ = new Cursor(0, 0, Point(), NULL);
  encodings_.clear();
  clipFlags = clipboardUTF8 | clipboardRTF | clipboardHTML |
              clipboardRequest | clipboardNotify | clipboardProvide;
  memset(clipSizes, 0, sizeof(clipSizes));
  clipSizes[0] = 20 * 1024 * 1024;
}

ClientParams::~ClientParams()
{
  delete [] name_;
  delete cursor_;
}

void ClientParams::setDimensions(int width, int height)
{
  ScreenSet layout;
  layout.add_screen(rfb::Screen(0, 0, 0, width, height, 0));
  setDimensions(width, height, layout);
}

void ClientParams::setDimensions(int width, int height, const ScreenSet& layout)
{
  if (!layout.validate(width, height))
    throw Exception("Attempted to configure an invalid screen layout");

  width_ = width;
  height_ = height;
  screenLayout_ = layout;
}

void ClientParams::setPF(const PixelFormat& pf)
{
  pf_ = pf;

  if (pf.bpp != 8 && pf.bpp != 16 && pf.bpp != 32)
    throw Exception("setPF: not 8, 16 or 32 bpp?");
}

void ClientParams::setName(const char* name)
{
  delete [] name_;
  name_ = strDup(name);
}

void ClientParams::setCursor(const Cursor& other)
{
  delete cursor_;
  cursor_ = new Cursor(other);
}

void ClientParams::setCursorPos(const Point& pos)
{
  cursorPos_ = pos;
}

bool ClientParams::supportsEncoding(rdr::S32 encoding) const
{
  return encodings_.count(encoding) != 0;
}

void ClientParams::setEncodings(int nEncodings, const rdr::S32* encodings)
{
  compressLevel = -1;
  qualityLevel = -1;
  fineQualityLevel = -1;
  subsampling = subsampleUndefined;
  supportsTightMP = false;
  supportsTightMPSolid = false;
  supportsTightMPJpegXL = false;
  mpLevel = -1;
  mpCompression = MP_COMPRESSION_DEFAULT;

  bool oldSupportsUTF8 = supportsUTF8;
  supportsUTF8 = false;

  encodings_.clear();
  encodings_.insert(encodingRaw);

  for (int i = nEncodings-1; i >= 0; i--) {
    switch (encodings[i]) {
    case pseudoEncodingUTF8:
      if (oldSupportsUTF8 != true)
        vlog.debug("Enabling UTF-8 clipboard transfer");
      supportsUTF8 = true;
      break;
    case pseudoEncodingNoSound:
      if (supportsSound != false)
        vlog.debug("Disabling sound support");
      supportsSound = false;
#if !defined(WIN32) && !defined(WIN64)
      setenv("VNCNOSOUND", "", 1);
#endif
      break;
    case pseudoEncodingWantSound:
      if (supportsSound != true)
        vlog.debug("Client wants sound support");
      supportsSound = true;
#if !defined(WIN32) && !defined(WIN64)
      unsetenv("VNCNOSOUND");
#endif
      break;
    case pseudoEncodingNoPrint:
      if (supportsPrint != false)
        vlog.debug("Disabling print support");
      supportsPrint = false;
#if !defined(WIN32) && !defined(WIN64)
      setenv("VNCNOPRINT", "", 1);
#endif
      break;
    case pseudoEncodingWantPrint:
      if (supportsPrint != true)
        vlog.debug("Client wants print support");
      supportsPrint = true;
#if !defined(WIN32) && !defined(WIN64)
      unsetenv("VNCNOPRINT");
#endif
      break;
    case pseudoEncodingSubsamp1X:
      subsampling = subsampleNone;
      break;
    case pseudoEncodingSubsampGray:
      subsampling = subsampleGray;
      break;
    case pseudoEncodingSubsamp2X:
      subsampling = subsample2X;
      break;
    case pseudoEncodingSubsamp4X:
      subsampling = subsample4X;
      break;
    case pseudoEncodingSubsamp8X:
      subsampling = subsample8X;
      break;
    case pseudoEncodingSubsamp16X:
      subsampling = subsample16X;
      break;
    case pseudoEncodingMPSolid:
      supportsTightMPSolid = true;
      vlog.debug("TightMPSolid enabled");
      break;
    case pseudoEncodingMPJpegXL:
      supportsTightMPJpegXL = true;
      vlog.debug("TightMPJpegXL enabled");
      break;
    }

    if (encodings[i] >= pseudoEncodingCompressLevel0 &&
        encodings[i] <= pseudoEncodingCompressLevel9)
      compressLevel = encodings[i] - pseudoEncodingCompressLevel0;

    if (encodings[i] >= pseudoEncodingQualityLevel0 &&
        encodings[i] <= pseudoEncodingQualityLevel9)
      qualityLevel = encodings[i] - pseudoEncodingQualityLevel0;

    if (encodings[i] >= pseudoEncodingMPLevel0 &&
        encodings[i] <= pseudoEncodingMPLevel9) {
      mpLevel = encodings[i] - pseudoEncodingMPLevel0;
      vlog.debug("mpLevel set to %u", mpLevel);
      supportsTightMP = true;
    }
    if (mpCompressionValid(encodings[i] - pseudoEncodingMPCompressionMin)) {
      mpCompression = encodings[i] - pseudoEncodingMPCompressionMin;
      vlog.debug("mpCompression set to %u", mpCompression);
      supportsTightMP = true;
    }
    if (encodings[i] <= pseudoEncodingPAPortBase &&
        encodings[i] >= pseudoEncodingPAPortMax) {
      paPort = (- encodings[i]) + pseudoEncodingPAPortBase;
#if !defined(WIN32) && !defined(WIN64)
      char paPortStr[10];
      vlog.debug("Setting PAPORT env var to %u", paPort);
      snprintf(paPortStr, 9, "%u", paPort);
      paPortStr[9] = 0;
      setenv("PAPORT", paPortStr, 1);
      FILE *f1;
      f1 = fopen("/tmp/paport", "w");
      if (!f1)
        vlog.error("Could not save PAPORT");
      else {
        fprintf(f1, "%u\n", paPort);
        fclose(f1);
        vlog.debug("Saved PAPORT %u", paPort);
      }
#endif
    }

    if (encodings[i] >= pseudoEncodingFineQualityLevel0 &&
        encodings[i] <= pseudoEncodingFineQualityLevel100)
      fineQualityLevel = encodings[i] - pseudoEncodingFineQualityLevel0;

    encodings_.insert(encodings[i]);
  }

#if !defined(WIN32) && !defined(WIN64)
  if (supportsSound) {
    if(!access(NOAUDIO_FILE, F_OK))
      vlog.debug("Remove old marker %s", NOAUDIO_FILE);
    unlink(NOAUDIO_FILE);
  } else {
    if(access(NOAUDIO_FILE, F_OK) < 0) {
      vlog.debug("Set marker %s", NOAUDIO_FILE);
      FILE *f1;
      f1 = fopen(NOAUDIO_FILE, "w");
      if (!f1)
        vlog.error("Could not set marker %s", NOAUDIO_FILE);
      else {
        fprintf(f1, "\n");
        fclose(f1);
      }
    }
  }
  if (supportsPrint) {
    if(!access(NOPRINT_FILE, F_OK))
      vlog.debug("Remove old marker %s", NOPRINT_FILE);
    unlink(NOPRINT_FILE);
  } else {
    if(access(NOPRINT_FILE, F_OK) < 0) {
      vlog.debug("Set marker %s", NOPRINT_FILE);
      FILE *f1;
      f1 = fopen(NOPRINT_FILE, "w");
      if (!f1)
        vlog.error("Could not set marker %s", NOPRINT_FILE);
      else {
        fprintf(f1, "\n");
        fclose(f1);
      }
    }
  }
#endif

}

void ClientParams::setLEDState(unsigned int state)
{
  ledState_ = state;
}

rdr::U32 ClientParams::clipboardSize(unsigned int format) const
{
  int i;

  for (i = 0;i < 16;i++) {
    if (((unsigned)1 << i) == format)
      return clipSizes[i];
  }

  throw Exception("Invalid clipboard format 0x%x", format);
}

void ClientParams::setClipboardCaps(rdr::U32 flags, const rdr::U32* lengths)
{
  int i, num;

  clipFlags = flags;

  num = 0;
  for (i = 0;i < 16;i++) {
    if (!(flags & (1 << i)))
      continue;
    clipSizes[i] = lengths[num++];
  }
}

bool ClientParams::supportsLocalCursor() const
{
  if (supportsEncoding(pseudoEncodingCursorWithAlpha))
    return true;
  if (supportsEncoding(pseudoEncodingVMwareCursor))
    return true;
  if (supportsEncoding(pseudoEncodingCursor))
    return true;
  if (supportsEncoding(pseudoEncodingXCursor))
    return true;
  return false;
}

bool ClientParams::supportsCursorPosition() const
{
  if (supportsEncoding(pseudoEncodingVMwareCursorPosition))
    return true;
  return false;
}

bool ClientParams::supportsDesktopSize() const
{
  if (supportsEncoding(pseudoEncodingExtendedDesktopSize))
    return true;
  if (supportsEncoding(pseudoEncodingDesktopSize))
    return true;
  return false;
}

bool ClientParams::supportsLEDState() const
{
  if (supportsEncoding(pseudoEncodingLEDState))
    return true;
  if (supportsEncoding(pseudoEncodingVMwareLEDState))
    return true;
  return false;
}

bool ClientParams::supportsFence() const
{
  if (supportsEncoding(pseudoEncodingFence))
    return true;
  return false;
}

bool ClientParams::supportsContinuousUpdates() const
{
  if (supportsEncoding(pseudoEncodingContinuousUpdates))
    return true;
  return false;
}
