/* Copyright 2021 m-privacy GmbH.
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 */

#include <pwd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#if !defined(WIN32) && !defined(WIN64)
#include <syslog.h>
#endif

#include <rfb/LogWriter.h>
#include <rfb/ServerCore.h>
#include <rfb/SSecurityMulti.h>

#include "SAutotransferNow.h"

static rfb::LogWriter vlog("SAutotransferNow");

SAutotransferNow::SAutotransferNow(rfb::SSecurityMulti* multi): SAutotransferDirWatcher()
{
	this->multi = multi;

	// Send autotransferClientDir set with the config menu
	const char* autotransferClientDir = rfb::Server::autotransferClientDir.getData();
	if (autotransferClientDir && *autotransferClientDir) {
		vlog.debug("%s: Sending signal with autotransferFolder to client: %s", __func__, autotransferClientDir);
		multi->sendSignal(AUTOTRANSFERPATH_SIGNAL_ID, autotransferClientDir, strlen(autotransferClientDir) + 1);
	} else {
		vlog.debug("%s: autotransferClientDir. The client will be able to decide where to auto-download stuff.", __func__);
	}
}

const char* SAutotransferNow::getDirToWatch()
{
	char* autotransferNowDir = (char*) malloc(4096);
	snprintf(autotransferNowDir, 4095, "/home/user/.autotransfer-now/%s", getpwuid(geteuid())->pw_name);
	autotransferNowDir[4095] = '\0';
	return (const char*) autotransferNowDir;
}

void SAutotransferNow::processFile(char* filename)
{
	FILE *handle;
	char buffer[60];

	vlog.verbose("%s: Checking file '%s' for error", __func__, filename);
	handle = fopen(filename, "r");
	if (!handle) {
		vlog.info("%s: failed to open file '%s', error: %s, trying to delete", __func__, filename, strerror(errno));
#if !defined(WIN32) && !defined(WIN64)
		openlog("Xtightgatevnc", LOG_PID, LOG_AUTH);
		syslog(LOG_DEBUG, "%s: failed to open file '%s', error: %s", __func__, filename, strerror(errno));
#endif
		if (unlink(filename) < 0) {
			vlog.info("%s: failed to delete file '%s', error: %s", __func__, filename, strerror(errno));
		}
		return;
	}
	memset(buffer, 0, 60);
	fread(buffer, 1, 59, handle);
	if (ferror(handle)) {
		fclose(handle);
		vlog.error("%s: failed to read file '%s'", __func__, filename);
#if !defined(WIN32) && !defined(WIN64)
		openlog("Xtightgatevnc", LOG_PID, LOG_AUTH);
		syslog(LOG_DEBUG, "%s: failed to read file '%s'", __func__, filename);
#endif
		return;
	}
	if (!strcmp(buffer, "{\"err\": \"File could not be found\"}")) {
		char logpath[128];

		vlog.info("%s: file '%s' got rejected, deleting error file", __func__, filename);
		unlink(filename);
		snprintf(logpath, 127, "/home/user/.opswatlog/%s", getenv("USER"));
		logpath[127] = 0;
		if (!access(logpath, F_OK)) {
			handle = fopen(logpath, "a");
			if (handle) {
				char * line = (char *) malloc(4096);

				if (line) {
					time_t nowsec = time(NULL);
					struct tm * now = localtime(&nowsec);

					snprintf(line, 4095, "%s %02u.%02u.%u %02u:%02u:%02u %s\n", getenv("HOSTNAME"), now->tm_mday, now->tm_mon+1, now->tm_year+1900, now->tm_hour, now->tm_min, now->tm_sec, filename);
					vlog.debug("%s: logging '%s' to %s", __func__, line, logpath);
					fwrite(line, 1, strlen(line), handle);
					if (ferror(handle)) {
						fclose(handle);
						vlog.error("%s: failed to write file '%s'", __func__, logpath);
#if !defined(WIN32) && !defined(WIN64)
						openlog("Xtightgatevnc", LOG_PID, LOG_AUTH);
						syslog(LOG_DEBUG, "%s: failed to write file '%s'", __func__, logpath);
#endif
					} else {
						fclose(handle);
						system("/usr/bin/pkill inotifywait");
					}
					free(line);
				}
			} else {
				vlog.info("%s: failed to open logfile %s, not logging, error: %s", __func__, logpath, strerror(errno));
			}
		} else {
			vlog.debug("%s: logfile %s not found, not logging", __func__, logpath);
		}
	} else {
		vlog.debug("%s: Sending file '%s' through multi", __func__, filename);
		multi->sendFile(filename, FILE_TRANSFER_STREAM_ID, AUTOTRANSFER_PROTOCOL_V1, true);
	}
}
