/* Copyright 2021 m-privacy GmbH.
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 */

#include <curl/curl.h>
#include <iostream>
#include <nlohmann/json.hpp>
#include <pwd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#if !defined(WIN32) && !defined(WIN64)
#include <syslog.h>
#endif

#include <rfb/LogWriter.h>

#include <tgpro_environment.h>
#include <mp_utils.h>

#include "SAutotransferOpswat.h"

using namespace std;
using json = nlohmann::json;

static rfb::LogWriter vlog("SAutotransferOpswat");

rfb::StringParameter opswatIp
("opswatIp",
 "IP address of the OPSWAT server to clean files and (auto)-download them to the client",
 "");
rfb::StringParameter opswatPort
("opswatPort",
 "Port of the OPSWAT API server to clean files and (auto)-download them to the client",
 "8008");
rfb::IntParameter opswatTimeout("OpswatTimeout", "OPSWAP timeout in seconds (min 60, max 3600)", 300, 60, 3600);

#define CA_CERTS_FILE "/home/user/.customopswatca/ca-certificates"

char* rule = NULL;
char* baseUrl = NULL;

const char* SAutotransferOpswat::getBaseUrl()
{
	if (baseUrl == NULL) {
		baseUrl = (char*) malloc(4096);
		if (!baseUrl) {
			vlog.error("%s: failed to allocate memory for a string. This is bad. Other stuff will probably break now", __func__);
			return NULL;
		}
		int useSsl = file_exists(CA_CERTS_FILE);
		snprintf(baseUrl, 4095, "%s://%s:%s", useSsl ? "https" : "http", opswatIp.getData(), opswatPort.getData());
		baseUrl[4095] = '\0';
		vlog.verbose("%s: base url for OPSWAT curl API requests: %s", __func__, baseUrl);
	}

	return baseUrl;
}

const char* SAutotransferOpswat::getRule()
{
	if (rule == NULL) {
		char* rulesFile = (char*) malloc(4096);
		if (!rulesFile) {
			vlog.error("%s: failed to allocate memory for a string. This is bad. Other stuff will probably break now", __func__);
			return "";
		}
		snprintf(rulesFile, 4095, "/home/user/.opswatenabled/%s", getpwuid(geteuid())->pw_name);
		rulesFile[4095] = '\0';

		char* line = (char*) malloc(4096);
		if (!line) {
			vlog.error("%s: failed to allocate memory for a string. This is bad. Other stuff will probably break now", __func__);
			return "";
		}

		FILE* f = fopen(rulesFile, "r");
		while (fgets(line, 4096, f) != NULL) {
			if(line[0] == '#' || line[0] == '\n') {
				vlog.verbose("%s: Skipping line (starts with # or is empty)", __func__);
				continue;
			}
		}
		fclose(f);

		char* newline = strchr(line, '\n');
		if (newline) {
			newline[0] = '\0';
		}
		vlog.verbose("%s: found OPSWAT rule in %s: %s", __func__, rulesFile, line);
		rule = line;
	}

	return rule;
}

const char* SAutotransferOpswat::getDirToWatch()
{
	char* autotransferDir = (char*) malloc(4096);
	snprintf(autotransferDir, 4095, "/home/user/.transfer/%s/opswat", getpwuid(geteuid())->pw_name);
	autotransferDir[4095] = '\0';
	return (const char*) autotransferDir;
}

void SAutotransferOpswat::processFile(char* filename)
{
	vlog.debug("%s: OPSWATting file '%s'", __func__, filename);

	string uploadResult = uploadFile(filename);
	if (!strstr(uploadResult.c_str(), "\"data_id\"")) {
		vlog.error("%s: failed to upload to the server. Here is the complete response: %s", __func__, uploadResult.c_str());
		vlog.error("%s: error uploading file to OPSWAT server. Couldn't read 'data_id' from answer. Can't process file '%s'", __func__, filename);
#if !defined(WIN32) && !defined(WIN64)
		openlog("Xtightgatevnc", LOG_PID, LOG_AUTH);
		syslog(LOG_DEBUG, "%s: failed to upload to the server. Here is the complete response: %s", __func__, uploadResult.c_str());
		syslog(LOG_DEBUG, "%s: error uploading file to OPSWAT server. Couldn't read 'data_id' from answer. Can't process file '%s'", __func__, filename);
#endif
		return;
	}

	json uploadJsonResult = json::parse(uploadResult);
	string dataId = uploadJsonResult["data_id"];

	vlog.debug("%s: waiting max. %us for OPSWAT result for '%s' ...", __func__, (int) opswatTimeout, filename);
	int i = 0;
	while (i < opswatTimeout) { // TODO: timeout? Aprox. 5 minutes if we sleep(1) in every loop (we don't really need high precision here)
		string fileAnalyzeResult = getFileAnalyzeResultJson(dataId);

		if (!strstr(fileAnalyzeResult.c_str(), "\"process_info\"")) {
			vlog.error("%s: failed to parse the answer from server. Here is the complete response: %s", __func__, fileAnalyzeResult.c_str());
			vlog.error("%s: couldn't read 'process_info' from json response. Can't process file '%s'", __func__, filename);
#if !defined(WIN32) && !defined(WIN64)
			openlog("Xtightgatevnc", LOG_PID, LOG_AUTH);
			syslog(LOG_DEBUG, "%s: failed to parse the answer from server. Here is the complete response: %s", __func__, fileAnalyzeResult.c_str());
			syslog(LOG_DEBUG, "%s: couldn't read 'process_info' from json response. Can't process file '%s'", __func__, filename);
#endif
			return;
		}
		if (!strstr(fileAnalyzeResult.c_str(), "\"progress_percentage\"")) {
			vlog.error("%s: failed to parse the answer from server. Here is the complete response: %s", __func__, fileAnalyzeResult.c_str());
			vlog.error("%s: couldn't read 'progress_percentage' from json response. Can't process file '%s'", __func__, filename);
#if !defined(WIN32) && !defined(WIN64)
			openlog("Xtightgatevnc", LOG_PID, LOG_AUTH);
			syslog(LOG_DEBUG, "%s: failed to parse the answer from server. Here is the complete response: %s", __func__, fileAnalyzeResult.c_str());
			syslog(LOG_DEBUG, "%s: couldn't read 'progress_percentage' from json response. Can't process file '%s'", __func__, filename);
#endif
			return;
		}

		json fileAnalyzeResultJson = json::parse(fileAnalyzeResult);
		vlog.debug("%s: parsed analyisis as JSON", __func__);
		if (!fileAnalyzeResultJson.contains("process_info") ||
			!fileAnalyzeResultJson["process_info"].contains("progress_percentage")) {
			vlog.error("%s: failed to get [\"process_info\"][\"progress_percentage\"] from json. Here is the complete response: %s", __func__, fileAnalyzeResult.c_str());
			vlog.error("%s: Can't process file '%s' (couldn't read json correctly)", __func__, filename);
			return;
		}

		unsigned int progressPercentage = fileAnalyzeResultJson["process_info"]["progress_percentage"];
		if (i > 0 && i % 60 == 0)
			vlog.debug("%s: '%s' progress percentage: %i", __func__, filename, (unsigned int) progressPercentage);
		else
			vlog.verbose("%s: '%s' progress percentage: %i", __func__, filename, (unsigned int) progressPercentage);

		if (progressPercentage == 100) {
			char* outputFile;
			const char * scanAllResult = NULL;
			char * convertedDestination = NULL;

			vlog.verbose("%s: '%s' fileAnalyzeResult: %s", __func__, filename, fileAnalyzeResult.c_str());

			if (fileAnalyzeResultJson.contains("process_info") &&
					fileAnalyzeResultJson["process_info"].contains("post_processing") &&
					fileAnalyzeResultJson["process_info"]["post_processing"].contains("converted_destination")) {

				const char* tmpConvertedDestination = fileAnalyzeResultJson["process_info"]["post_processing"]["converted_destination"].get<std::string>().c_str();
				if (tmpConvertedDestination && *tmpConvertedDestination) {
					if (starts_with(tmpConvertedDestination, "(user: ")) {
						const char* position = strstr(tmpConvertedDestination, ") "); // skip the (user: $USER) that we add ourselves
						if (position) {
							size_t lengthToKeep = strlen(position + 2);
							convertedDestination = (char*) malloc(lengthToKeep + 1);
							strncpy(convertedDestination, position + 2, lengthToKeep);
							convertedDestination[lengthToKeep] = '\0';
							vlog.debug("%s: found 'converted_destination' key in json analysis file: '%s'. Using it instead of the original filename", __func__, convertedDestination);
						}
					} else { // If for whatever reason user: $USER isn't to be found anymore, just use the string as is
						convertedDestination = strdup(tmpConvertedDestination);
					}
				}
			} else {
				vlog.debug("%s: json analysis result doesn't contain a key [\"process_info\"][\"post_processing\"][\"converted_destination\"]. Using the original filename: %s", __func__, filename);
			}

			// check scan result, if any
			if (fileAnalyzeResultJson.contains("scan_results") &&
				fileAnalyzeResultJson["scan_results"].contains("scan_all_result_a")) {
				scanAllResult = fileAnalyzeResultJson["scan_results"]["scan_all_result_a"].get<std::string>().c_str();
			} else {
				scanAllResult = "unknown reason (not specified in JSON)";
				vlog.error("%s: failed to read [\"scan_results\"][\"scan_all_result_a\"] (human-readable reason for allowing or blocking of file) from anaylsis json file", __func__);
			}

			if (!fileAnalyzeResultJson.contains("process_info") ||
				!fileAnalyzeResultJson["process_info"].contains("result")) {
				vlog.error("%s: failed to read [\"process_info\"][\"result\"]. Can't continue processing file! Aborting...", __func__);
				return;
			}

			if (strcmp(fileAnalyzeResultJson["process_info"]["result"].get<std::string>().c_str(), "Allowed")) {
				char logpath[128];

				delete_file(filename);
				vlog.info("%s: OPSWAT rejects %s, scan result: %s", __func__, filename, scanAllResult);
#if !defined(WIN32) && !defined(WIN64)
				openlog("Xtightgatevnc", LOG_PID, LOG_AUTH);
				syslog(LOG_DEBUG, "%s: OPSWAT rejects %s, scan result: %s", __func__, filename, scanAllResult);
#endif
				snprintf(logpath, 127, "/home/user/.opswatlog/%s", getenv("USER"));
				logpath[127] = 0;
				if (!access(logpath, F_OK)) {
					FILE *handle;

					handle = fopen(logpath, "a");
					if (handle) {
						char * line = (char *) malloc(4096);

						if (line) {
							time_t nowsec = time(NULL);
							struct tm * now = localtime(&nowsec);

							snprintf(line, 4095, "%s %02u.%02u.%u %02u:%02u:%02u %s\n", getenv("HOSTNAME"), now->tm_mday, now->tm_mon+1, now->tm_year+1900, now->tm_hour, now->tm_min, now->tm_sec, filename);
							vlog.debug("%s: logging '%s' to %s", __func__, line, logpath);
							fwrite(line, 1, strlen(line), handle);
							if (ferror(handle)) {
								fclose(handle);
								vlog.error("%s: failed to write file '%s'", __func__, logpath);
#if !defined(WIN32) && !defined(WIN64)
								openlog("Xtightgatevnc", LOG_PID, LOG_AUTH);
								syslog(LOG_DEBUG, "%s: failed to write file '%s'", __func__, logpath);
#endif
							} else {
								fclose(handle);
								system("/usr/bin/pkill inotifywait");
							}
							free(line);
						}
					} else {
						vlog.info("%s: failed to open logfile %s, not logging, error: %s", __func__, logpath, strerror(errno));
					}
				} else {
					vlog.debug("%s: logfile %s not found, not logging", __func__, logpath);
				}
				return;
			} else {
				vlog.debug("%s: OPSWAT accepts %s, scan result: %s", __func__, filename, scanAllResult);
			}

			outputFile = (char*) malloc(4096);
			if (!outputFile) {
				vlog.error("%s: failed to allocate memory for a string. This is bad. Other stuff will probably break now Not processing file '%s'", __func__, filename);
			}
			if (convertedDestination && *convertedDestination) { // If we found converted_destination in the analysis json file
				snprintf(outputFile, 4095, "/home/user/.autotransfer-now/%s/%s", getpwuid(geteuid())->pw_name, convertedDestination);
				free(convertedDestination);
			} else {
				snprintf(outputFile, 4095, "/home/user/.autotransfer-now/%s/%s", getpwuid(geteuid())->pw_name, basename(filename));
			}
			outputFile[4095] = '\0';

			// check: do we have a sanitized file?
			// sanitization_details gefunden?
			if (strstr(fileAnalyzeResult.c_str(), "\"sanitization_details\"")) {
				// successful? ("sanitization_details":{"description":"Sanitized successfully.")
				if (strstr(fileAnalyzeResult.c_str(), "\"Sanitized successfully")) {
					if (getSanitizedFile(dataId, outputFile) == CURLE_OK)
						vlog.debug("%s: '%s' sanitized. Downloaded to '%s'", __func__, filename, outputFile);
				} else {
					// This isn't true for eml files and archives so we just let it go through too
					mp_move(filename, outputFile, 1);
					vlog.debug("%s: '%s' not sanitized (possibly an eml file or an archive). Moved to '%s'", __func__, filename, outputFile);
				}
				delete_file(filename);
			} else {
				mp_move(filename, outputFile, 1);
				vlog.debug("%s: '%s' not sanitized, moved to '%s'", __func__, filename, outputFile);
			}
			free(outputFile);
			// do we have severity_index?
			if (strstr(fileAnalyzeResult.c_str(), "\"severity_index\""))
				vlog.debug("'%s' has severity_index", filename);
			break;
		}

		i += 2;
		sleep(2);
	}
	if (i >= opswatTimeout) {
		vlog.error("%s: OPSWAT timed out for '%s' after %us", __func__, filename, (int) opswatTimeout);
#if !defined(WIN32) && !defined(WIN64)
		openlog("Xtightgatevnc", LOG_PID, LOG_AUTH);
		syslog(LOG_DEBUG, "%s: OPSWAT timed out for '%s' after %us", __func__, filename, (int) opswatTimeout);
#endif
	}
}

/**
 * Get the size of a file.
 * @param filename The name of the file to check size for
 * @return The filesize, or 0 if the file does not exist.
 */
static size_t getFilesize(const char* filename) {
    struct stat st;
    if(stat(filename, &st) != 0) {
        return 0;
    }
    return st.st_size;
}

static size_t writeCallback(void *contents, size_t size, size_t nmemb, void *userp)
{
    ((string*) userp)->append((char*)contents, size * nmemb);
    return size * nmemb;
}

static size_t readCallback(char *ptr, size_t size, size_t nmemb, void *userdata)
{
	FILE *readhere = (FILE *)userdata;
	size_t retcode = fread(ptr, size, nmemb, readhere);
	return (curl_off_t)retcode;
}

string SAutotransferOpswat::getFileAnalyzeResultJson(const string dataId)
{
	char* url = (char*) malloc(4096);
	if (!url) {
		vlog.error("%s: failed to allocate memory for a string. This is bad. Other stuff will probably break now", __func__);
		return ""; // TODO something better than this. Maybe even create our own JSON?
	}
	snprintf(url, 4095, "%s/file/%s", getBaseUrl(), dataId.c_str());
	url[4095] = '\0';
	vlog.verbose("%s: getting file analysis result with '%s'", __func__, url);

	CURL* curl;
	CURLcode curlResult;
	string json;

	curl = curl_easy_init();
	if (curl) {
		if (file_exists(CA_CERTS_FILE)) {
			curl_easy_setopt(curl, CURLOPT_CAINFO, CA_CERTS_FILE);
		}
		curl_easy_setopt(curl, CURLOPT_URL, url);
		curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, writeCallback);
		curl_easy_setopt(curl, CURLOPT_WRITEDATA, &json);
		curlResult = curl_easy_perform(curl);
		curl_easy_cleanup(curl);

		free(url);

		if (curlResult == CURLE_OK) {
			return json;
		} else {
			string curlError = curl_easy_strerror(curlResult);
			vlog.error("%s: curl error: %s", __func__, curlError.c_str());
#if !defined(WIN32) && !defined(WIN64)
			openlog("Xtightgatevnc", LOG_PID, LOG_AUTH);
			syslog(LOG_DEBUG, "%s: curl error: %s", __func__, curlError.c_str());
#endif
			// TODO something better than not doing anything and returning a "" at the end. Maybe even create our own JSON?
		}
	}

	return ""; // TODO something better than this. Maybe even create our own JSON?
}

CURLcode SAutotransferOpswat::getSanitizedFile(const string dataId, const string outputFile)
{
	char* url = (char*) malloc(4096);
	if (!url) {
		vlog.error("%s: failed to allocate memory for a string. This is bad. Other stuff will probably break now", __func__);
		return CURLE_OUT_OF_MEMORY; // TODO something better than this? Maybe a CURL ERROR code?
	}
	snprintf(url, 4095, "%s/file/converted/%s", getBaseUrl(), dataId.c_str());
	url[4095] = '\0';
	vlog.debug("%s: trying to download sanitized file from: %s", __func__, url);

	CURL* curl;
	CURLcode curlResult = CURLE_OK;
	string json;
	FILE* fp;
	curl = curl_easy_init();
	if (curl) {
		if (file_exists(CA_CERTS_FILE)) {
			curl_easy_setopt(curl, CURLOPT_CAINFO, CA_CERTS_FILE);
		}
		fp = fopen(outputFile.c_str(), "wb");
		curl_easy_setopt(curl, CURLOPT_URL, url);
		curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, NULL); // We don't need to write a callback function because it's the "standard" way
		curl_easy_setopt(curl, CURLOPT_WRITEDATA, fp);
		curlResult = curl_easy_perform(curl);
		curl_easy_cleanup(curl);
		fclose(fp);
		if (curlResult != CURLE_OK) {
			string curlError = curl_easy_strerror(curlResult);
			vlog.error("%s: curl error: %s", __func__, curlError.c_str());
#if !defined(WIN32) && !defined(WIN64)
			openlog("Xtightgatevnc", LOG_PID, LOG_AUTH);
			syslog(LOG_DEBUG, "%s: curl error: %s", __func__, curlError.c_str());
#endif
			// TODO something better than not doing anything and returning a "" at the end. Maybe even create our own JSON?
		}

	}

	free(url);

	return curlResult; // TODO something better than this. Maybe even create our own JSON?
}

string SAutotransferOpswat::uploadFile(const char* filename)
{
	char* url = (char*) malloc(4096);
	if (!url) {
		vlog.error("%s: failed to allocate memory for a string. This is bad. Other stuff will probably break now", __func__);
		return ""; // TODO something better than this. Maybe even create our own JSON?
	}
	snprintf(url, 4095, "%s/file", getBaseUrl());
	url[4095] = '\0';

	struct curl_slist* headers = NULL;

	char* filenameHeader = (char*) malloc(4096);
	if (!filenameHeader) {
		vlog.error("%s: failed to allocate memory for a string. This is bad. Other stuff will probably break now", __func__);
		return ""; // TODO something better than this. Maybe even create our own JSON?
	}
	snprintf(filenameHeader, 4095, "filename: (user: %s) %s", getenv("USER"), basename(filename));
	filenameHeader[4095] = '\0';
	headers = curl_slist_append(headers, filenameHeader);
	vlog.verbose("%s: added HTTP header for curl POST: %s", __func__, filenameHeader);

	char* ruleHeader = (char*) malloc(4096);
	if (!ruleHeader) {
			vlog.error("%s: failed to allocate memory for a string. This is bad. Other stuff will probably break now", __func__);
			return ""; // TODO something better than this. Maybe even create our own JSON?
	}

	if (getRule() && *getRule()) {
		vlog.debug("%s: using OPSWAT rule '%s'", __func__, getRule());
		snprintf(ruleHeader, 4095, "rule: %s", getRule());
		ruleHeader[4095] = '\0';
		headers = curl_slist_append(headers, ruleHeader);
		vlog.verbose("%s: added HTTP header for curl POST: %s", __func__, ruleHeader);
	} else {
		vlog.debug("%s: no OPSWAT rule set for user. Calling OPSWAT without specifying a rule (this isn't necessarily bad)", __func__);
	}

	CURL* curl;
	CURLcode curlResult;
	string json;

	vlog.debug("%s: Uploading file to OPSWAT using this URL: %s", __func__, url);

	FILE* file = fopen(filename, "rb");
	if (!file) {
		vlog.error("%s: failed to open file '%s': %s", __func__, filename, strerror(errno));
#if !defined(WIN32) && !defined(WIN64)
		openlog("Xtightgatevnc", LOG_PID, LOG_AUTH);
		syslog(LOG_DEBUG, "%s: failed to open file '%s': %s", __func__, filename, strerror(errno));
#endif
		return "";
	}
	curl = curl_easy_init();
	if (curl) {
		if (file_exists(CA_CERTS_FILE)) {
			curl_easy_setopt(curl, CURLOPT_CAINFO, CA_CERTS_FILE);
		}
		curl_easy_setopt(curl, CURLOPT_URL, url);
		curl_easy_setopt(curl, CURLOPT_HTTPHEADER, headers);
		curl_easy_setopt(curl, CURLOPT_POST, 1L);
		curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE, getFilesize(filename));
		curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, writeCallback);
		curl_easy_setopt(curl, CURLOPT_WRITEDATA, &json);
		curl_easy_setopt(curl, CURLOPT_READFUNCTION, readCallback);
		curl_easy_setopt(curl, CURLOPT_READDATA, (void *)file);
		curlResult = curl_easy_perform(curl);
		curl_easy_cleanup(curl);

		free(url);
		free(filenameHeader);
		free(ruleHeader);

		if (curlResult == CURLE_OK) {
//			delete_file(filename);
			return json;
		} else {
			string curlError = curl_easy_strerror(curlResult);
			vlog.error("%s: curl error: %s", __func__, curlError.c_str());
#if !defined(WIN32) && !defined(WIN64)
			openlog("Xtightgatevnc", LOG_PID, LOG_AUTH);
			syslog(LOG_DEBUG, "%s: curl error: %s", __func__, curlError.c_str());
#endif
			// TODO something better than not doing anything and returning a "" at the end. Maybe even create our own JSON?
		}
	}

	return ""; // TODO something better than this. Maybe even create our own JSON?
}
