/* 
 * Copyright (C) 2010 TigerVNC Team
 * Copyright (C) 2014-2021 m-privacy GmbH, Berlin
 * 
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <rdr/Exception.h>
#include <rfb/Security.h>
#include <rfb/SSecurityNone.h>
#include <rfb/SSecurityStack.h>
#include <rfb/SSecurityPlain.h>
#include <rfb/SSecurityCert.h>
#include <rfb/SSecurityKrb.h>
#include <rfb/SSecurityVncAuth.h>
#include <rfb/SSecurityVeNCrypt.h>
#include <rfb/SSecurityMulti.h>
#include <rfb/LogWriter.h>
#ifdef HAVE_GNUTLS
#include <rfb/SSecurityTLS.h>
#endif

using namespace rdr;
using namespace rfb;

static LogWriter vlog("SecurityServer");

StringParameter SecurityServer::secTypes
("SecurityTypes",
 "Specify which security scheme to use (None, VncAuth, Plain, PlainMulti"
 ", TLSNone, TLSVnc, TLSPlain, TLSPlainMulti, X509None, X509Vnc, X509Plain, X509PlainMulti, X509Cert, X509CertMulti, X509Krb, X509KrbMulti"
 ")",
 "TLSPlainMulti,TLSPlain,X509CertMulti,X509Cert,X509KrbMulti,X509Krb,X509PlainMulti,X509Plain,X509Vnc,TLSVnc,X509None,TLSNone,VncAuth,None",
 ConfServer);

BoolParameter noMulti("NoMulti",
                      "Disable VNC multiplexing (which is used for printing, sound, etc.).",
                      false);

SecurityServer::SecurityServer(void) : Security(secTypes, !noMulti), mySec(NULL)
{}

SSecurity* SecurityServer::GetSSecurity(SConnection* sc, U32 secType)
{
  if (!IsSupported(secType))
    goto bail;

  switch (secType) {
  case secTypeNone: return mySec = new SSecurityNone(sc);
  case secTypeVncAuth: return mySec = new SSecurityVncAuth(sc);
  case secTypeVeNCrypt: return mySec = new SSecurityVeNCrypt(sc, this);
  case secTypePlain: return mySec = new SSecurityPlain(sc);
  case secTypeKrb: return mySec = new SSecurityKrb(sc);
  case secTypePlainMulti:
    return mySec = new SSecurityStack(sc, secTypePlainMulti, new SSecurityMulti(sc), new SSecurityPlain(sc));
#ifdef HAVE_GNUTLS
  case secTypeTLSNone:
    return mySec = new SSecurityStack(sc, secTypeTLSNone, new SSecurityTLS(sc, true, false));
  case secTypeTLSVnc:
    return mySec = new SSecurityStack(sc, secTypeTLSVnc, new SSecurityTLS(sc, true, false), new SSecurityVncAuth(sc));
  case secTypeTLSPlain:
    return mySec = new SSecurityStack(sc, secTypeTLSPlain, new SSecurityTLS(sc, true, false), new SSecurityPlain(sc));
  case secTypeTLSPlainMulti:
    return mySec = new SSecurityStack(sc, secTypeTLSPlainMulti, new SSecurityTLS(sc, true, false), new SSecurityMulti(sc), new SSecurityPlain(sc));
  case secTypeTLSKrb:
    return mySec = new SSecurityStack(sc, secTypeTLSKrb, new SSecurityTLS(sc, true, false), new SSecurityKrb(sc));
  case secTypeTLSKrbMulti:
    return mySec = new SSecurityStack(sc, secTypeTLSKrbMulti, new SSecurityTLS(sc, true, false), new SSecurityMulti(sc), new SSecurityKrb(sc));
  case secTypeX509None:
    return mySec = new SSecurityStack(sc, secTypeX509None, new SSecurityTLS(sc, false, false));
  case secTypeX509Vnc:
    return mySec = new SSecurityStack(sc, secTypeX509None, new SSecurityTLS(sc, false, false), new SSecurityVncAuth(sc));
  case secTypeX509Plain:
    return mySec = new SSecurityStack(sc, secTypeX509Plain, new SSecurityTLS(sc, false, false), new SSecurityPlain(sc));
  case secTypeX509PlainMulti:
    return mySec = new SSecurityStack(sc, secTypeX509PlainMulti, new SSecurityTLS(sc, false, false), new SSecurityMulti(sc), new SSecurityPlain(sc));
  case secTypeX509Cert:
    return mySec = new SSecurityStack(sc, secTypeX509Cert, new SSecurityTLS(sc, false, true), new SSecurityCert(sc));
  case secTypeX509CertMulti:
    return mySec = new SSecurityStack(sc, secTypeX509CertMulti, new SSecurityTLS(sc, false, true), new SSecurityMulti(sc), new SSecurityCert(sc));
  case secTypeX509Krb:
    return mySec = new SSecurityStack(sc, secTypeX509Krb, new SSecurityTLS(sc, false, false), new SSecurityKrb(sc));
  case secTypeX509KrbMulti:
    return mySec = new SSecurityStack(sc, secTypeX509KrbMulti, new SSecurityTLS(sc, false, false), new SSecurityMulti(sc), new SSecurityKrb(sc));
#endif
  }

bail:
  throw Exception("Security type not supported");
}

