/* Copyright (C) 2024 m-privacy GmbH
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 */

#ifndef NATIVE_MESSAGING_APP_H
#define NATIVE_MESSAGING_APP_H

namespace NativeMessagingApp {

	/**
	 * Determines if the application was started as a native messaging app.
	 *
	 * This function checks the command-line arguments to identify if the application
	 * was launched as a native messaging application, typically used for communication
	 * between the browser and the native app.
	 *
	 * @param argc Number of command-line arguments.
	 * @param argv Array of command-line argument strings.
	 * @return True if the app was started as a native messaging app, otherwise false.
	 */
	bool startedAsNativeMessagingApp(int argc, char* argv[]);

	/**
	 * Main read loop for handling communication with the browser extension.
	 *
	 * This function enters a loop to read messages sent to the native messaging
	 * app (typically in JSON format), process them, and then send responses back to the caller.
	 *
	 * @return An integer status code, 0 for success, or 1 for failure.
	 */
	int readLoop(void);

	/**
	 * Sends a response message back to the caller.
	 *
	 * This function formats and sends a response message (in JSON format)
	 * to the originating browser extension using native messaging.
	 *
	 * @param responseText The message or response to be sent, in JSON format.
	 */
	void sendResponse(const char* responseText);

	/**
	 * Sends a list of whitelisted URLs or actions.
	 *
	 * This function compiles and sends a JSON array or object containing whitelisted
	 * URLs or actions that are permitted for communication or processing.
	 */
	void sendWhitelists();

	/**
	 * Unescapes special characters in a JSON string.
	 *
	 * This function takes a JSON-encoded string with escape sequences (like \" or \\)
	 * and converts it back into a normal string by unescaping these characters.
	 *
	 * @param input The escaped JSON string input.
	 * @return A pointer to the unescaped string.
	 *         The returned `char*` is dynamically allocated, and must be freed by the caller.
	 */
	char* unescapeJsonString(const char* input);

	/**
	 * Escapes special characters in a JSON string.
	 *
	 * This function takes a normal string and escapes special characters (like quotes,
	 * backslashes, etc.) so it can be safely inserted into a JSON structure.
	 *
	 * @param input The raw string input to escape.
	 * @return A pointer to the escaped string.
	 *         The returned `char*` is dynamically allocated, and must be freed by the caller.
	 */
	char* escapeJsonString(const char* input);

	/**
	 * Parses a JSON string and extracts key information.
	 *
	 * This function parses a JSON string and extracts relevant values, such as an
	 * action and a URL, from the provided input. It modifies the actionP and urlP
	 * pointers to point to these extracted values.
	 *
	 * @param input The JSON-formatted input string.
	 * @param actionP Pointer to a string that will hold the extracted action.
	 * @param urlP Pointer to a string that will hold the extracted URL.
	 * @return An integer status code, usually 0 for success, or a non-zero error code.
	 */
	int parseJson(const char* input, char** actionP, char** urlP);

	/**
	 * Internal JSON parsing helper function.
	 *
	 * This function acts as a helper to the public `parseJson` function, performing
	 * the actual parsing logic. It extracts action and URL data from the input JSON
	 * and assigns them to the respective pointers.
	 *
	 * @param input The JSON-formatted input string.
	 * @param actionP Pointer to a string that will hold the extracted action.
	 * @param urlP Pointer to a string that will hold the extracted URL.
	 * @return An integer status code, usually 0 for success, or a non-zero error code.
	 */
	int doParseJson(const char* input, char** actionP, char** urlP);

	/**
	 * Registers the application as a native messaging host.
	 *
	 * This function performs the following steps:
	 * 1. Creates a chrome.json file in %appdata%/vnc (or in configdir).
	 * 2. Reads IDs from local unpacked extensions and adds them to the chrome.json file.
	 * 3. Updates the chrome.json with the correct path to the viewer.
	 *
	 * @param configDir The directory where the chrome.json file will be created.
	 * If NULL, the default directory (%appdata%/vnc) will be used.
	 */
	bool registerAsNativeMessagingHostApp(const char* configDir);
	bool registerAsNativeMessagingHostApp2(const char* configDir);
};

#endif  // NATIVE_MESSAGING_APP_H
