/* Copyright (C) 2024 m-privacy GmbH
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 */

#ifndef URL_WHITELIST_HANDLER_H
#define URL_WHITELIST_HANDLER_H

/**
 * Namespace to handle operations related to URL whitelists.
 */
namespace UrlWhitelistHandler {
	/**
	 * Reads all whitelist files from the provided viewer configuration directory and concatenates their contents into a single string.
	 *
	 * @param viewerConfigDir The directory where whitelist files are stored.
	 * @return A dynamically allocated char* containing the concatenated content of all whitelists. The caller is responsible for freeing this memory.
	 */
	char* readAllWhitelistsAsString(const char* viewerConfigDir);

	/**
	 * Concatenates the contents of multiple files specified by the given filenames array.
	 *
	 * @param filenames An array of file paths to concatenate.
	 * @param numFiles The number of files in the filenames array.
	 * @return A dynamically allocated char* containing the concatenated content of all files. The caller is responsible for freeing this memory.
	 */
	char* concatenateFiles(const char** filenames, const size_t numFiles);

	/**
	 * Reads the contents of a single whitelist file specified by the file path.
	 *
	 * @param filePath The path to the whitelist file.
	 * @return A dynamically allocated char* containing the content of the whitelist file. The caller is responsible for freeing this memory.
	 */
	char* readWhitelistAsString(const char* filePath);

	/**
	 * Retrieves a value from the BrowserChoice configuration file based on the provided key.
	 *
	 * @param key The key to look up in the configuration file.
	 * @return A dynamically allocated char* containing the value associated with the provided key. The caller is responsible for freeing this memory.
	 */
	char* getValueFromBrowserchoiceConfigFile(const char* key);

	/**
	 * Retrieves the path to the whitelist from the BrowserChoice configuration file.
	 *
	 * @return A dynamically allocated char* containing the whitelist path. The caller is responsible for freeing this memory.
	 */
	char* getWhitelistPathFromConfigFile();
};

#endif // URL_WHITELIST_HANDLER_H
