#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>
#include <stdint.h>

#include <fcntl.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#define PCRE2_STATIC

/* This macro must be defined before including pcre2.h. For a program that uses
only one code unit width, it makes it possible to use generic function names
such as pcre2_compile(). */
#define PCRE2_CODE_UNIT_WIDTH 8

#include <pcre2.h>

#if defined(WIN32) || defined(WIN64) /* Because pcre2 is not available in linux yet */
#include <windows.h>

#include <wchar.h>

#else

#define SENDFILELIMIT 0x7ffff000
#define COPYBUFFERSIZE (1024 * 1024)

#include <netdb.h>
#include <sys/socket.h>
#if !defined(__APPLE__)
#include <sys/sendfile.h>
#endif
#include <netinet/in.h>
#include <arpa/inet.h>

#endif /* else of if defined(WIN32) || defined(WIN64) */

#include "log.h"
#include "tgpro_environment.h"

/**
 * Returns true if string starts with beginning
 * Inspired by https://stackoverflow.com/questions/15515088/how-to-check-if-string-starts-with-certain-string-in-c
 */
int starts_with(const char* string, const char* beginning) {
	return strlen(string) >= strlen(beginning) &&
		!strncmp(string, beginning, strlen(beginning));
}

/**
 * Returns true if string starts ends with ending
 * Inspired by https://stackoverflow.com/questions/10347689/how-can-i-check-whether-a-string-ends-with-csv-in-c
 */
int ends_with(const char* string, const char* ending) {
	return strlen(string) >= strlen(ending) &&
		!strcmp(string + strlen(string) - strlen(ending), ending);
}

const char* strstr_insensitive(const char *haystack, const char *needle) {
	if ( !*needle ) {
		return haystack;
	}
	for ( ; *haystack; ++haystack ) {
		if (toupper(*haystack) == toupper(*needle)) { // Matched starting char -- loop through remaining chars.
			const char *h, *n;
			for (h = haystack, n = needle; *h && *n; ++h, ++n) {
				if (toupper(*h) != toupper(*n)) {
					break;
				}
			}
			if ( !*n ) { // matched all of 'needle' to null termination
				return haystack; // return the start of the match
			}
		}
	}
	return 0;
}

/**
 * Inspired by http://c-programming-blog.blogspot.de/2015/05/pcre2-simple-sample-program.html
 *
 * Returns only the first match and NULL if no match was found.
 * Remeber to free the returned match!
 */

char* match(const char* string, const char* regex, const int group) {
	char* match = NULL;

	PCRE2_SIZE error_offset;
	int error_code;
	PCRE2_SPTR pattern = (PCRE2_SPTR) regex;
	PCRE2_SPTR pcre2_string = (PCRE2_SPTR) string;
	pcre2_code* compiled_regex = pcre2_compile(pattern, PCRE2_ZERO_TERMINATED, 0, &error_code, &error_offset, NULL);
	if (!compiled_regex) {
		PCRE2_UCHAR8 buffer[120];
		pcre2_get_error_message(error_code, buffer, 120);
		log_error("PCRE2 regex compilation failed at offset %d: %s for pattern: %s", (int) error_offset, buffer, pattern);
		return NULL;
	}
	pcre2_match_data* match_data = pcre2_match_data_create(20, NULL);
	int rc = pcre2_match(compiled_regex, pcre2_string, -1, 0, 0, match_data, NULL);
	if (rc <= 0) {
		match = NULL;
	} else {
		PCRE2_SIZE* ovector = pcre2_get_ovector_pointer(match_data);
		PCRE2_SPTR match_start = pcre2_string + ovector[2*group];
		PCRE2_SIZE match_length = ovector[2*group + 1] - ovector[2*group];
		if (match_length) {
			match = malloc((int) match_length + 1);
			sprintf(match, "%.*s", (int) match_length, (char*) match_start);
		} else {
			match = NULL;
		}
	}

	pcre2_match_data_free(match_data);
	pcre2_code_free(compiled_regex);
	return match;
}

char* concat(char* s1, char* s2) {
    char* result = malloc(strlen(s1) + strlen(s2) + 1);
    if (result) {
	    strcpy(result, s1);
	    strcat(result, s2);
    }
    return result;
}

char* randomstring(size_t length) {

    static char charset[] = "abcdefghijklmnopqrstuvwxyz0123456789";
    char* randomString = NULL;

    if (length) {
        randomString = malloc(sizeof(char) * (length + 1));

        if (randomString) {
            for (int n = 0; n < length; n++) {
                int key = rand() % (int) (sizeof(charset) - 1);
                randomString[n] = charset[key];
                srand((intptr_t)&randomString[n]);
            }

            randomString[length] = '\0';
        }
    }

    return randomString;
}

/**
 * From:
 * https://stackoverflow.com/questions/779875/what-is-the-function-to-replace-string-in-c
 * Thank you!
 *
 * Don't forget to free the returned char*
 */
char* str_replace(char* orig, char* rep, char* with) {
	char* result; // the return string
	char* ins;    // the next insert point
	char* tmp;    // varies
	int len_rep;  // length of rep
	int len_with; // length of with
	int len_front; // distance between rep and end of last rep
	int count;    // number of replacements

	if (!orig)
		return NULL;
	if (!rep)
		rep = "";
	len_rep = strlen(rep);
	if (!with)
		with = "";
	len_with = strlen(with);

	ins = orig;
	for (count = 0; (tmp = strstr(ins, rep)); ++count) {
		ins = tmp + len_rep;
	}

	// first time through the loop, all the variable are set correctly
	// from here on,
	//    tmp points to the end of the result string
	//    ins points to the next occurrence of rep in orig
	//    orig points to the remainder of orig after "end of rep"
	tmp = result = (char*) malloc(strlen(orig) + (len_with - len_rep) * count + 1);

	if (!result)
		return NULL;

	while (count--) {
		ins = strstr(orig, rep);
		len_front = ins - orig;
		tmp = strncpy(tmp, orig, len_front) + len_front;
		tmp = strcpy(tmp, with) + len_with;
		orig += len_front + len_rep; // move to next "end of rep"
	}
	strcpy(tmp, orig);
	return result;
}

/**
 * Removes the surrounding double quotes for a string. If it starts
 * and ends with a double quote.
 *
 * It will change the input char* so please beware. If the input was
 * allocated, it should still be safe to free afterwards.
 *
 * Make sure input is a null-terminated char*.
 */
void remove_quotes(char* input)
{
	if (input == NULL || strlen(input) < 2) {
		return;
	}

	size_t len = strlen(input);
	if (input[0] == '"' && input[len - 1] == '"') {
		memmove(input, input + 1, len - 2);
		input[len - 2] = '\0';
	}
}

char* replace_invalid_windows_path_chars(const char* input) {
	if (!input) {
		return NULL;
	}

#ifdef WIN32
	size_t output_length = 0;
	size_t input_length = strlen(input);

	for (size_t i = 0; i < input_length; i++) {
		if (strchr("<>:\"/\\|?*", input[i])) { // invalid char for a Windows path
			output_length += 3;  // For the longest replacement ("%20")
		} else {
			output_length++;
		}
	}

	char* output = (char*) malloc(output_length + 1);
	if (!output) {
		log_error("%s: failed to allocate memory for string with replaced chars. Something is very wrong...", __func__);
		return NULL;
	}
	size_t output_index = 0;
	for (size_t i = 0; i < input_length; i++) {
		if (strchr("<>:\"/\\|?*", input[i])) { // invalid char for a Windows path
			sprintf(&output[output_index], "%%%02X", (unsigned char)input[i]);
			output_index += 3;
		} else {
			output[output_index] = input[i];
			output_index++;
		}
	}

	output[output_index] = '\0';
	return output;
#else
	return NULL;
#endif /* #ifdef WIN32 */
}
#if defined(WIN32) || defined(WIN64)
// TODO: Use iconv to transform to Windows' UTF16!
wchar_t* utf8_char_to_wchar_t(const char* string) {
	// Run once to see how much space we are going to need
	int neededSize = MultiByteToWideChar(CP_ACP, 0, string, strlen(string) + 1, NULL, 0);
	wchar_t* wstring = (wchar_t*) malloc((strlen(string) + 1) * 4);
	if (!wstring) {
		return NULL;
	}
	int result = MultiByteToWideChar(CP_ACP, 0, string, strlen(string) + 1, wstring, neededSize);
	if (!result) {
		return NULL;
	}
	return wstring;
}

wchar_t* convert_utf8_char_to_utf16_wchar_t(const char* string) {
	return utf8_char_to_wchar_t(string);
}

// TODO: Use iconv to transform from Windows' UTF16!
char* convert_utf16_wchar_to_utf8_char(const wchar_t* wstring) {
	int neededSize = WideCharToMultiByte(CP_UTF8, 0, wstring, wcslen(wstring) + 1, NULL, 0, NULL, NULL);
	char* string = (char*) malloc(neededSize + 1);
	int result = WideCharToMultiByte(CP_UTF8, 0, wstring, wcslen(wstring) + 1, string, neededSize, NULL, NULL);
	if (!result) {
		return NULL;
	}
	return string;
}
#endif /* if defined(WIN32) || defined(WIN64) */

void delete_file(char* file) {
	if (!file)
		return;
#if defined(WIN32) || defined(WIN64)
	wchar_t* wfile = utf8_char_to_wchar_t(file);
	DeleteFileW(wfile);
	free(wfile);
#else
	remove(file);
#endif /* defined(WIN32) || defined(WIN64) */
}

void mp_copy(const char* source, const char* destination, unsigned int overwrite) {
	if (file_exists(destination) && !overwrite) {
		log_warn("File '%s' exists and overwrite is set to false", destination);
		return;
	}

#if defined(WIN32) || defined(WIN64)
	wchar_t* wsource = utf8_char_to_wchar_t(source);
	wchar_t* wdestination = utf8_char_to_wchar_t(destination);
	CopyFileW(wsource, wdestination, 0);
	free(wsource);
	free(wdestination);
#else
	if (file_exists(destination)) {
		log_trace("Deleting '%s' before moving it because the user wants to overwrite it", destination);
		remove(destination);
	}

	int read_fd;
	int write_fd;
	struct stat stat_buf;
	off_t offset = 0;
	read_fd = open(source, O_RDONLY);
	if (fstat(read_fd, &stat_buf) < 0) {
		log_warn("fstat on file '%s' failed with error %s", source, strerror(errno));
		close(read_fd);
		return;
	}
	write_fd = open(destination, O_WRONLY | O_CREAT, stat_buf.st_mode);
/* no sendfile on APPLE */
#if !defined(__APPLE__)
	if (stat_buf.st_size > SENDFILELIMIT) {
#else
	{
#endif
		int read_err;
		int write_err;
		int written;
		char * buffer = (char *) malloc(COPYBUFFERSIZE);

#if !defined(__APPLE__)
		log_debug("file '%s' size %lu is bigger than sendfile() maximum size %u, copying by hand", source, stat_buf.st_size, SENDFILELIMIT);
#endif
		if (buffer) {
			while (offset < stat_buf.st_size) {
				read_err = read(read_fd, buffer, COPYBUFFERSIZE);
				if (read_err < 0) {
					log_warn("copying file '%s' to '%s' failed, read error: %s", source, destination, strerror(errno));
					unlink(destination);
					break;
				}
				offset += read_err;
				written = 0;
				while (written < read_err) {
					write_err = write(write_fd, buffer + written, read_err - written);
					if (write_err < 0) {
						log_warn("copying file '%s' to '%s' failed, write error: %s", source, destination, strerror(errno));
						unlink(destination);
						offset = stat_buf.st_size;
						break;
					}
					written += write_err;
				}
			}
			free(buffer);
		} else {
			log_warn("copying file '%s' failed, could not allocate memory", source);
		}
#if !defined(__APPLE__)
	} else {
		if (sendfile(write_fd, read_fd, &offset, stat_buf.st_size) < 0) {
			log_warn("copying file '%s' with sendfile() failed with error %s", source, strerror(errno));
			unlink(destination);
		}
#endif
	}
	close(read_fd);
	close(write_fd);
#endif /* defined(WIN32) || defined(WIN64) */
}

void mp_move(char* source, char* destination, unsigned int overwrite) {
	if (file_exists(destination) && !overwrite) {
		log_warn("File '%s' exists and overwrite is set to false", destination);
		return;
	}
#if defined(WIN32) || defined(WIN64)
	wchar_t* wsource = utf8_char_to_wchar_t(source);
	wchar_t* wdestination = utf8_char_to_wchar_t(destination);
	if (file_exists(destination)) {
		log_trace("Deleting '%s' before moving it because the user wants to overwrite it", destination);
		DeleteFileW(wdestination);
	}
	_wrename(wsource, wdestination);
	free(wsource);
	free(wdestination);
#else
	if (file_exists(destination)) {
		log_trace("Deleting '%s' before moving it because the user wants to overwrite it", destination);
		remove(destination);
	}
	int error = rename(source, destination);
	if (error && errno != EXDEV) {
		log_error("An error ocurred while moving file: %s", strerror(errno));
	} else if (errno == EXDEV) {
		log_trace("Warning. Failed to move file due to an EXDEV (Invalid cross-device link) error. Copying and deleting original file");
		mp_copy(source, destination, overwrite);
		remove(source);
	}
#endif /* defined(WIN32) || defined(WIN64) */
}

unsigned int get_host_by_name(const char* hostname, char (*ip_address_ptr)[16]) {
#if defined(WIN32) || defined(WIN64)
	WSADATA wsaData;
	int error = WSAStartup(MAKEWORD(2, 2), &wsaData);
	if (error) {
		log_error("WSAStartup() failed with error code %d", error);
		return 0;
	}
#endif /* defined(WIN32) || defined(WIN64) */

	struct sockaddr_in sock;
	memset(&sock, 0, sizeof(struct sockaddr_in));
	sock.sin_family = AF_INET;

	struct hostent* he = gethostbyname(hostname);
	if (!he) {
#if defined(WIN32) || defined(WIN64)
		error = WSAGetLastError();
		if (error != 0) {
			if (error == WSAHOST_NOT_FOUND) {
				log_error("gethostbyname failed. WSAHOST_NOT_FOUND: Host not found\n");
			} else if (error == WSANO_DATA) {
				log_error("gethostbyname failed. WSANO_DATA: No data record found\n");
			} else {
				log_error("gethostbyname failed with error: %ld\n", error);
			}
		}
#else
		if (h_errno == HOST_NOT_FOUND) {
			log_error("gethostbyname failed. HOST_NOT_FOUND: Host not found\n");
		} else if (h_errno == NO_DATA) {
			log_error("gethostbyname failed. NO_DATA: No data record found\n");
		} else {
			log_error("gethostbyname failed with error: %ld\n", h_errno);
		}
#endif /* defined(WIN32) || defined(WIN64) */
		return 0;
	}
	char* addr = inet_ntoa(*((struct in_addr *) he->h_addr_list[0]));

#if defined(WIN32) || defined(WIN64)
	WSACleanup();
#endif /* defined(WIN32) || defined(WIN64) */

	char* ip_address = *ip_address_ptr;
	memset(ip_address, 0, 16);
	strncpy(ip_address, addr, 15);
	return 1;
}

unsigned int is_cluster_node_available(const char* hostname) {
	if (!hostname) {
		log_error("Error! Hostname cannot be null.");
		error_check(EINVAL, L"Der Hostname des TightGate-Pro Servers ist leer. Bitte überprüfen Sie Ihre Konfiguration.");
	}

	char ip_address[16];
	ip_address[0] = 0;
	if (!get_host_by_name(hostname, &ip_address)) {
		log_error("Error! An error ocurred while trying to resolve hostname");
		error_check(errno, L"Aktuell ist sicheres Internet nicht verfügbar. Der DNS-Name des TightGate-Pro Servers ist nicht auflösbar.");
	}
	if(!strcmp(ip_address, "255.255.255.255")) { // The cluster load-balancing returns 255.255.255.255 if no nodes are available
		log_error("Error! Hostname '%s' resolved to '%s'. No cluster nodes available!", hostname, ip_address);
#ifdef WIN32
		error_check(ERROR_NETWORK_BUSY, L"Aktuell ist sicheres Internet nicht verfügbar. Bitte versuchen Sie es später (in ein paar Minuten) nochmal.");
#else
		error_check(EHOSTUNREACH, L"Aktuell ist sicheres Internet nicht verfügbar. Bitte versuchen Sie es später (in ein paar Minuten) nochmal.");
#endif
	} else {
		log_debug("Hostname '%s' resolved to '%s': Yay, there are nodes available (or it just isn't a cluster)!", hostname, ip_address);
		return 1;
	}

	return 0;
}

/* Read specific field from a CSV line */
char* get_field(const char* line, char delimiter, const unsigned int field) {
	if (!delimiter) {
		delimiter = '|';
	}

	char* line_dup = strdup(line);
	error_check(!line_dup, L"Die URL-Listen konnten nicht eingelesen werden. Der Arbeitsspeicher ist voll.");

	char* tmp = line_dup;

	char* value = NULL;
	for (unsigned int i=0; i<=field; i++) {
		if (i == field) {
			char* last_delimiter = strchr(tmp, delimiter);
			if (last_delimiter) {
				*last_delimiter = '\0';
			}
			log_trace("%s: read field %u: <%s>", __func__, field, tmp);
			value = strdup(tmp);

			error_check(!value, L"Die URL-Listen konnten nicht eingelesen werden. Der Arbeitsspeicher ist voll.");

			break;
		}
		// Get next field (if we haven't found ours yet)
		tmp = strchr(tmp, delimiter);
		if (!tmp) {
			log_warn("%s: couldn't reach field %u in %s (this isn't necessarily bad). Not enough fields separated by '%c'?", __func__, field, line, delimiter);
			break; // Failed to find
		}
		tmp++; // Skip the delimiter
	}

	free(line_dup);
	return value;
}

char * prepareuser_error(int errorcode)
{
	char * langstring = getenv("LANG");
	int langnum = 0;

	if (!strncmp(langstring, "de_", 3))
		langnum = 1;
	switch(errorcode) {
		case 1:
			if (langnum == 1)
				return "USER ist leer";
			else
				return "USER is empty";
		case 2:
			if (langnum == 1)
				return "USER mit Leerzeichen";
			else
				return "USER contains space character";
		case 3:
			if (langnum == 1)
				return "USER mit /";
			else
				return "USER contains /";
		case 4:
			if (langnum == 1)
				return "USER ist eine Zahl";
			else
				return "USER is numeric";
		case 5:
			if (langnum == 1)
				return "Benutzer-Verzeichnis-Basis fehlt";
			else
				return "User home base is missing";
		case 7:
			if (langnum == 1)
				return "Benutzer nicht in AD-Gruppe tgprouser";
			else
				return "User is not in AD group tgprouser";
		case 8:
			if (langnum == 1)
				return "Benutzer-Verzeichnis fehlt und darf nicht erzeugt werden";
			else
				return "User home is missing, auto creation disabled";
		case 9:
			if (langnum == 1)
				return "Keine Lizenz";
			else
				return "No license";
		case 10:
			if (langnum == 1)
				return "Alle Lizenzen werden verwendet";
			else
				return "Licensed accounts exceeded";
		case 11:
			if (langnum == 1)
				return "Benutzer hat falsche ID";
			else
				return "User has invalid ID";
		default:
			if (langnum == 1)
				return "Unbekannt";
			else
				return "Unknown";
	}
}
