#include <stdlib.h>
#include <string.h>

#if defined(WIN32) || defined(WIN64)
#include <windows.h>
#endif
#include <stdio.h>
#include "log.h"
#include "tgpro_environment.h"
#include "winscp_ini_handler.h"
#include "winscp_connector.h"


char* winscp_bin = NULL;
char* winscp_ini = NULL;
#if defined(WIN32) || defined(WIN64)
HANDLE winscp_child_o_read = NULL;
HANDLE winscp_child_o_write = NULL;
HANDLE winscp_child_i_read = NULL;
HANDLE winscp_child_i_write = NULL;
SECURITY_ATTRIBUTES winscp_sa;
PROCESS_INFORMATION winscp_pi;
STARTUPINFO winscp_si;
#define BUF_SIZE 4096
#endif
char* winscp_host_fingerprint = NULL;
int winscp_prompt_ready = 0;


/**
 * Sets winscp_connector's WinSCP binary path variable to a new value.
 */
void winscp_set_bin(const char* new_bin) {
	error_check(!new_bin, L"Jemand hat einen NULL-pointer an winscp_set_bin übergeben. Das ist nicht erlaubt.");
	winscp_bin = (char*) realloc(winscp_bin, strlen(new_bin) + 1);
	error_check(winscp_bin == NULL, L"winscp_set_bin: Ich konnte keinen Arbeitsspeicher anfordern. Das geht so nicht.");
	strcpy(winscp_bin, new_bin);
}


/**
 * Sets winscp_connector's WinSCP ini file path variable to a new value.
 */
void winscp_set_ini(const char* new_ini) {
	error_check(!new_ini, L"Jemand hat einen NULL-pointer an winscp_set_ini übergeben. Das ist nicht erlaubt.");
	winscp_ini = (char*) realloc(winscp_ini, strlen(new_ini) + 1);
	error_check(winscp_ini == NULL, L"winscp_set_ini: Ich konnte keinen Arbeitsspeicher anfordern. Das geht so nicht.");
	strcpy(winscp_ini, new_ini);
}


const char* winscp_get_ini() {
	if (winscp_ini)
		return winscp_ini;
	if (file_exists(SPOOL_INI))
		return (winscp_ini = strdup(SPOOL_INI));
	if (file_exists(SPOOL_INI_SUBDIR))
		return (winscp_ini = strdup(SPOOL_INI_SUBDIR));
	size_t ini_str_len = strlen(get_appdata_path()) + strlen(SPOOL_INI_SUBDIR) + 1;
	winscp_ini = malloc(ini_str_len);
	snprintf(winscp_ini, ini_str_len, "%s%s", get_appdata_path(), SPOOL_INI_SUBDIR);
	return winscp_ini;
}


/**
 * Grabs a host key fingerprint out of the given WinSCP message msg.
 * Allocates memory for the new fingerprint string a gives back a pointer to it.
 * Do not forget to free the memory. Returns NULL if there was no fingerprint
 * found in msg.
 */
char* winscp_get_fingerprint_from_msg(char* msg) {
	/*
	 Parsing these outputs:
	 - English: Host key fingerprint is ssh-ed25519 256 9ZmmyEv3LdVdwBDefJSjjpQgEUn/t7zcNdr7/cBVBzo=.
	 - German: Schlüssel-Fingerabdruck des Servers ist ssh-ed25519 256 9ZmmyEv3LdVdwBDefJSjjpQgEUn/t7zcNdr7/cBVBzo=.
	 */
	char* fingerprint_indicator = "Host key fingerprint is ";
	char* pre_fp_msg = strstr(msg, fingerprint_indicator);
	if (pre_fp_msg == NULL) {
		/* No english fingerprint message. Maybe a german one? */
		log_warn("*** No English fingerprint. A german one?\n");
		fingerprint_indicator = "Fingerabdruck des Hostschlüssels ist ";
		pre_fp_msg = strstr(msg, fingerprint_indicator);
	}
	if (pre_fp_msg == NULL) {
		log_warn("*** No German fingerprint. Maybe an old WinSCP version?\n");
		fingerprint_indicator = "Fingerabdruck des Servers ist ";
		pre_fp_msg = strstr(msg, fingerprint_indicator);
	}

	if (pre_fp_msg == NULL) {
		log_error("***NO GERMAN FINGERPRINT: <%s>\n", fingerprint_indicator);
		return NULL;
	}
	char* fingerprint_begin = pre_fp_msg + strlen(fingerprint_indicator);
	char* fingerprint_end = strstr(fingerprint_begin, ".");
	error_check(fingerprint_end == NULL, L"Nach dem Fingerprint in der WinSCP-Nachricht konnte ich kein Punkt finden.");
	const size_t fingerprint_len = fingerprint_end - fingerprint_begin;
	char* ret = malloc(fingerprint_len + 1);
	error_check(ret == NULL, L"Allozierung des Host-Fingerprint-Speichers in get_fingerprint ging schief.");
	strncpy(ret, fingerprint_begin, fingerprint_len);
	ret[fingerprint_len] = '\0';
	log_trace("got key fingerprint from winscp output: %s\n", ret);
	return ret;
}


/**
 * Opens an altert box window for the user. Informs the user about a changed host key
 * fingerprint and asks for a decision: take new key, retry or cancel.
 * Returns the Win32-API MessageBox return value.
 */
int winscp_unequal_fingerprints(char* new_fingerprint) {
#if defined(WIN32) || defined(WIN64)
	LPCWSTR box_title = L"TightGate-Pro Druckdienst";
	LPCWSTR alert_msg = L"Die Identität des TightGate-Pro-Servers scheint sich \
geändert zu haben. Dies kann auf einen Angriff auf ihr Netzwerk hindeuten. Es \
ist aber auch möglich, dass der TightGate-Pro-Server einfach neu installiert \
wurde. Bitte fragen sie ihre Systemadministratorin oder ihren Systemadministrator, \
wenn sie sich unsicher sind.\n\nWenn sie auf „Abbrechen“ klicken, dann wird der \
Druckdienst auf ihrem Computer beendet. Klicken sie auf „Weiter“, dann wird die \
neue Identität des TightGate-Pro-Servers als richtig angenommen und gespeichert.";
	return MessageBoxW(NULL, alert_msg, box_title, MB_CANCELTRYCONTINUE
			   | MB_ICONWARNING | MB_DEFBUTTON1 | MB_TASKMODAL);
#else
	return 0;
#endif
}


/**
 * Waits for the given message from a running WinSCP instance. Stops
 * waiting and returns after (a little more than) max_time_msec milliseconds
 * if the given msg did not appear.
 * The return value is one of the WINSCP_... values from winscp_connector.h:
 *
 * WINSCP_TIMEOUT: Waited, but nothing came.
 * WINSCP_GOT_PROMPT: Not msg came, but a prompt. Input is not possible.
 * WINSCP_GOT_TARGET: The msg came, but not a prompt.
 * WINSCP_GOT_TARGET_AND_PROMPT: The msg and a prompt were send.
 * WINSCP_GOT_CHANGED_KEY: A message about a key different than the stored one came.
 * WINSCP_GOT_NEW_KEY: A message about an unknown key came.
 */
int winscp_wait_for_msg(char* msg, int max_time_msec) {
#if defined(WIN32) || defined(WIN64)
	char buffer[BUF_SIZE];
	DWORD read_bytes = 0;
	DWORD still_avail;
	DWORD left;
	int time_waited = 0;

	while (time_waited < max_time_msec) {
		PeekNamedPipe(winscp_child_o_read, buffer, 1, &read_bytes, &still_avail, &left);

		if (read_bytes > 0) {
			ReadFile(winscp_child_o_read, buffer, BUF_SIZE-1, &read_bytes, NULL);
			buffer[read_bytes] = '\0';
			log_trace("* WinSCP Output: %s\n", buffer);

			if (strstr(buffer, "winscp>"))
				winscp_prompt_ready = 1;

			if (strstr(buffer, "host key does not match the one WinSCP has in cache")) {
				/* This must be an SSO (w/o AD) installation. A fresh installation uses WinSCP's
				   mechanism für host key storage. If that key changes, it gets here. Now
				   I use my own host key storage technique. */
				log_warn("* Key changed!\n");
				char* new_fingerprint = winscp_get_fingerprint_from_msg(buffer);
				log_debug("* Got new fingerprint\n");
				int user_choice = winscp_unequal_fingerprints(new_fingerprint);
				if (user_choice == IDCANCEL) {
					free(new_fingerprint);
					return WINSCP_CANCEL;
				}
				if (user_choice == IDTRYAGAIN) {
					free(new_fingerprint);
					return WINSCP_RETRY;
				}
				free(winscp_host_fingerprint);
				winscp_host_fingerprint = new_fingerprint;
				return WINSCP_GOT_CHANGED_KEY;
			}

			if (strstr(buffer, "host key was not found in the cache") ||
			    strstr(buffer, "ssel des Servers wurde nicht im Speicher gefunden.")) {
				if (winscp_host_fingerprint == NULL) {
					winscp_host_fingerprint = winscp_get_fingerprint_from_msg(buffer);
				} else {
					char* new_fingerprint = winscp_get_fingerprint_from_msg(buffer);
					if (strcmp(winscp_host_fingerprint, new_fingerprint) != 0) {
						int user_choice = winscp_unequal_fingerprints(new_fingerprint);
						if (user_choice == IDCANCEL) {
							free(new_fingerprint);
							return WINSCP_CANCEL;
						}
						if (user_choice == IDTRYAGAIN) {
							free(new_fingerprint);
							return WINSCP_RETRY;
						}
					}
					free(winscp_host_fingerprint);
					winscp_host_fingerprint = new_fingerprint;
				}
				return WINSCP_GOT_NEW_KEY;
			}

			if (strstr(buffer, "Password:") || strstr(buffer, "Passwort:")) {
				winscp_send_command("\r\r\r\r\r");
				return WINSCP_ERROR;
			}

			if (strstr(buffer, "Error"))
				return WINSCP_ERROR;

			if (strstr(buffer, "Access denied") || strstr(buffer, "Server refused our key"))
				return WINSCP_ERROR;

			if (strstr(buffer, msg) != NULL) {
				return (strstr(buffer, "winscp>") == NULL) ? WINSCP_GOT_TARGET : WINSCP_GOT_TARGET_AND_PROMPT;
			} else if (strstr(buffer, "winscp>") != NULL) {
				log_error("DAMN IT! There is a command prompt and not what I wanted.");
				return WINSCP_GOT_PROMPT;
			}

		} else {
			printf(" . ");
			Sleep(100);
			time_waited += 100;
		}
	}

	log_debug("WAIT FOR MSG TIMEOUT");
	return WINSCP_TIMEOUT;
#else
	return 0;
#endif
}


/**
 * Waits three seconds for a WinSCP command prompt.
 * Returns WINSCP_GOT_TARGET_AND_PROMPT if the prompt came. Returns
 * all other winscp_wait_for_msg() return values.
 */
int winscp_wait_for_prompt() {
	if (winscp_prompt_ready)
		return WINSCP_GOT_TARGET_AND_PROMPT;
	return winscp_wait_for_msg("winscp>", 1500);
}


/**
 * Sends a command to a running WinSCP instance.
 */
void winscp_send_command(char* winscp_command) {
#if defined(WIN32) || defined(WIN64)
	if (winscp_wait_for_prompt() != WINSCP_GOT_TARGET_AND_PROMPT) {
		log_error("ERROR: I want to send a winscp command, but there is no prompt.");
		log_debug("I try it anyway.");
	}
	winscp_prompt_ready = 0;
	DWORD wrote_bytes;
	WriteFile(winscp_child_i_write, winscp_command, strlen(winscp_command), &wrote_bytes, NULL);
	error_check(wrote_bytes != strlen(winscp_command), L"Die Verbindung mit WinSCP ist fehlgeschlagen. Ich kann das Programm nicht steuern.");
	WriteFile(winscp_child_i_write, "\n", strlen("\n"), &wrote_bytes, NULL);
	error_check(wrote_bytes != strlen("\n"), L"Die Verbindung mit WinSCP ist fehlgeschlagen. Ich konnte keinen RETURN-Code senden.");
	log_debug("I told WinSCP: %s", winscp_command);
#endif
}


/**
 * Sends a <set option> command to a running WinSCP instance.
 */
void winscp_set_option(char* winscp_option, char* winscp_value) {
	error_check(winscp_option == NULL || winscp_value == NULL, L"winscp_set_option bekam einen NULL-Pointer-Parameter.");
	const char* option_cmd = "option ";
	char buffer[strlen(option_cmd) + strlen(winscp_option) + strlen(winscp_value) + 2];
	strcpy(buffer, option_cmd);
	strcat(buffer, winscp_option);
	strcat(buffer, " ");
	strcat(buffer, winscp_value);
	winscp_send_command(buffer);
}

/**
 * It just checks if there is any active session at all.
 * See #define WINSCP_ in winscp_connector or even better the
 * winscp_open_connection function for an explanation of the
 * possible return values.
 */

int get_winscp_prompt_status() {
    winscp_send_command("session");
    return winscp_wait_for_msg("Active session:", 15000);
}

/**
 * Tells a running instance of WinSCP to open a connection to session.
 * session must be specified (with this name) in the spool.ini file.
 */
int winscp_open_connection(char* session) {
	error_check(session == NULL, L"winscp_open_connection bekam einen NULL-Pointer-Parameter.");

        /* These options have no effect with curren WinSCP version: */
	/* winscp_set_option("echo", "off");  */
	/* winscp_set_option("batch", "off"); */
	winscp_set_option("confirm", "off");

	char* hostkey_parameter;
	if (winscp_get_host_fingerprint(session) != NULL) {
#define HOSTKEY_STR " -hostkey=\""
		hostkey_parameter = malloc(strlen(HOSTKEY_STR) + strlen(winscp_host_fingerprint) + 2);
		strcpy(hostkey_parameter, HOSTKEY_STR);
		strcat(hostkey_parameter, winscp_host_fingerprint);
		strcat(hostkey_parameter, "\"");
	} else {
		log_debug("***** NO HOST FINGERPRINT IN STORE");
		hostkey_parameter = malloc(1);
		*hostkey_parameter = '\0';
	}

#define OPEN_CMD "open "
	char open_cmd_line[strlen(OPEN_CMD) + strlen(session) + strlen(hostkey_parameter) + 1];
	strcpy(open_cmd_line, OPEN_CMD);
	strcat(open_cmd_line, session);
	strcat(open_cmd_line, hostkey_parameter);
	free(hostkey_parameter);
	winscp_send_command(open_cmd_line);

	int wait_res = winscp_wait_for_msg("Active session:", 15000);
	if (wait_res == WINSCP_ERROR) {
		log_error("Error: Winscp returned an error! Stopping.");
	} else if (wait_res == WINSCP_GOT_TARGET) {
		log_debug("Found active session.");
	} else if (wait_res == WINSCP_GOT_TARGET_AND_PROMPT) {
		log_debug("All right, fine, I got my active session string. We can go on.");
	} else if (wait_res == WINSCP_GOT_PROMPT) {
		log_warn("Huh, there is a prompt, but no active session? NO CONNECTION THEN, I assume. I'll try it in one minute again.");
	} else if (wait_res == WINSCP_TIMEOUT) {
		log_error("CONNECTION WENT WRONG! TIMEOUT! What to do now? And there is no prompt.");
	} else if (wait_res == WINSCP_GOT_NEW_KEY || wait_res == WINSCP_GOT_CHANGED_KEY) {
		log_warn("got a NEW or CHANGED key!");
	} else if (wait_res == WINSCP_CANCEL) {
		log_warn("got a CANCEL message!");
	} else if (wait_res == WINSCP_RETRY) {
		log_warn("Wrong KEY -- retry.");
		return winscp_open_connection(session);
	}

	return wait_res;
}



void winscp_cd_spool() {
#if defined(WIN32) || defined(WIN64)
	DWORD wrote_bytes;
	WriteFile(winscp_child_i_write, "pwd\n", 4, &wrote_bytes, NULL);
	error_check(wrote_bytes != 4, L"Die Verbindung mit WinSCP ist fehlgeschlagen. Ich kann das Programm nicht steuern. Ich konnte kein pwd senden.");

	char transfer_dir_buf[BUF_SIZE];
	transfer_dir_buf[0] = '\0';
	char* transfer_dir = "";
	DWORD read_bytes = 0;
	DWORD still_avail;
	DWORD left;
	int time_waited = 0;

	while (!read_bytes && time_waited < 2000) {
		PeekNamedPipe(winscp_child_o_read, transfer_dir_buf, 1, &read_bytes, &still_avail, &left);

		if (read_bytes > 0) {
			ReadFile(winscp_child_o_read, transfer_dir_buf, BUF_SIZE - 1, &read_bytes, NULL);
			transfer_dir_buf[read_bytes] = '\0';
			char* search_cr = strchr(transfer_dir_buf, '\n');
			if (search_cr)
				*search_cr = '\0';
			transfer_dir = search_cr ? search_cr + 1 : transfer_dir_buf;
			if ((search_cr = strchr(transfer_dir, '\n')))
				*search_cr = '\0';
			if ((search_cr = strchr(transfer_dir, '\r')))
				*search_cr = '\0';
		} else {
			printf(" . ");
			Sleep(100);
			time_waited += 100;
		}
	}

	char* transfer_subdir = strstr(transfer_dir, ".transfer");
	if (!transfer_subdir) {
		log_error("I can not cd to spool dir because I am not in transfer dir! I am in <%s>.\n", transfer_dir);
		return;
	}
	const size_t spool_cmd_len = strlen(transfer_dir) + 2;
	char spool_cmd[spool_cmd_len];
	*transfer_subdir = '\0';
	snprintf(spool_cmd, spool_cmd_len, "cd %s.spool%s\n", transfer_dir, transfer_subdir + 9);

	log_debug("I will cd to spool dir: <%s>", spool_cmd);
	WriteFile(winscp_child_i_write, spool_cmd, spool_cmd_len, &wrote_bytes, NULL);
#endif
}


/**
 * Tells a running WinSCP instance to copy some files of file_type to
 * local destination_dir without deleting the original files.
 */
void winscp_copy(const char* file_type, const char* destination_dir) {
#define GET_CMD "get -transfer=binary -preservetime "
	winscp_wait_for_prompt();
	winscp_send_command("\r\r\r\r\r");
	char get_cmd[strlen(GET_CMD) +
		     strlen(file_type) +
		     strlen(destination_dir) + 2];
	strcpy(get_cmd, GET_CMD);
	strcat(get_cmd, file_type);
	strcat(get_cmd, " ");
	strcat(get_cmd, destination_dir);
	winscp_send_command(get_cmd);
#undef GET_CMD
}


/**
 * Tells a running WinSCP instance to copy some files of file_type to
 * local destination_dir and to delete the original files (to move them).
 */
void winscp_move(const char* file_type, const char* destination_dir) {
#define GET_CMD "get -delete -transfer=binary -preservetime "
	winscp_wait_for_prompt();
	winscp_send_command("\r\r\r\r\r");
	char get_cmd[strlen(GET_CMD) +
		     strlen(file_type) +
		     strlen(destination_dir) + 2];
	strcpy(get_cmd, GET_CMD);
	strcat(get_cmd, file_type);
	strcat(get_cmd, " ");
	strcat(get_cmd, destination_dir);
	winscp_send_command(get_cmd);
#undef GET_CMD
}


/**
 * Returns the host fingerprint.
 * This function get the fingerprint either from memory or tries to get it out
 * of the ini file.
 * Returns NULL if there is no fingerprint stored.
 */
char* winscp_get_host_fingerprint(const char* session) {
	error_check(winscp_ini == NULL, L"Die Funktion winscp_get_host_fingerprint wurde gestartet, ohne dass die ini-Datei festgelegt wurde.");
	if (winscp_host_fingerprint == NULL) {
#define SESSIONS_STR "[Sessions\\"
		char section[strlen(SESSIONS_STR) + strlen(session) + 2];
		strcpy(section, SESSIONS_STR);
		strcat(section, session);
		strcat(section, "]");
		winscp_host_fingerprint = get_winscp_ini_value(winscp_ini, section, "HostFingerprint");
	}
	return winscp_host_fingerprint;
}


void winscp_start_interactive() {
#if defined(WIN32) || defined(WIN64)
	error_check(winscp_bin == NULL || winscp_ini == NULL, L"Die Funktion winscp_start_interactive wurde gestartet, ohne das die bin-Datei oder die ini-Datei festgelegt wurden.");

	size_t call_len = strlen(winscp_bin) + strlen(winscp_ini) + 50;
	char winscp_call[call_len];
	snprintf(winscp_call, call_len, "\"%s\" /console /ini=\"%s\"", winscp_bin, winscp_ini);

	log_debug("Interactive WinSCP call: %s", winscp_call);

	/* Clean up the mem for process' and pipes' init structures. */
	ZeroMemory(&winscp_sa, sizeof(winscp_sa));
	ZeroMemory(&winscp_pi, sizeof(winscp_pi));
	ZeroMemory(&winscp_si, sizeof(winscp_si));

	/* Init SECURITY_ATTRIBUTES winscp_sa, needed for the anonymous pipes. */
	winscp_sa.nLength = sizeof(SECURITY_ATTRIBUTES);
	winscp_sa.bInheritHandle = TRUE;
	winscp_sa.lpSecurityDescriptor = NULL;

	/* Set up the process' input and output pipes (stdin & stdout redirections). */
	error_check(!CreatePipe(&winscp_child_o_read, &winscp_child_o_write, &winscp_sa, 0),
		    L"Could not create output pipe.");
	error_check(!SetHandleInformation(winscp_child_o_read, HANDLE_FLAG_INHERIT, 0),
		    L"Could not set handle information for winscp_child_o_read.");
	error_check(!CreatePipe(&winscp_child_i_read, &winscp_child_i_write, &winscp_sa, 0),
		    L"Could not create input pipe.");
	error_check(!SetHandleInformation(winscp_child_i_write, HANDLE_FLAG_INHERIT, 0),
		    L"Could not set handle information for winscp_child_i_write.");

	/* Init STARTUPINFO winscp_si. */
	winscp_si.cb = sizeof(winscp_si);
	winscp_si.hStdInput = winscp_child_i_read;
	winscp_si.hStdOutput = winscp_child_o_write;
	winscp_si.hStdError = winscp_child_o_write;
	winscp_si.dwFlags |= STARTF_USESTDHANDLES;
	winscp_si.wShowWindow = SW_HIDE;

	/* Launch the process. */
	error_check(!CreateProcess(NULL,             // No module name
				   winscp_call,
				   NULL,             // Process handle not inheritable
				   NULL,             // Thread handle not inheritable
				   TRUE,             // Set handle inheritance
				   CREATE_NO_WINDOW, // creation flags
				   NULL,             // Use parent's environment block
				   NULL,             // Use parent's starting directory
				   &winscp_si,              // Pointer to STARTUPINFO structure
				   &winscp_pi),             // Pointer to PROCESS_INFORMATION structure
		    L"Das Erstellen des WinSCP-Prozesses schlug fehl.");
#endif
}


/**
 * Runs a new instance of WinSCP as a child process.
 */
void winscp_start(char* session, const int open_immediately) {
#if defined(WIN32) || defined(WIN64)
	error_check(winscp_bin == NULL || winscp_ini == NULL, L"Die Funktion winscp_start wurde gestartet, ohne das die bin-Datei oder die ini-Datei festgelegt wurden.");

	char* con_str;
#define HOSTKEY_ACCEPT_NEW_PARAM  "/hostkey=\"acceptnew\""
#define NEWINSTANCE_PARAM  "/newinstance"

	if (open_immediately) {
		size_t con_str_size = strlen(session) + strlen(HOSTKEY_ACCEPT_NEW_PARAM) + strlen(NEWINSTANCE_PARAM) + 2 + 1; // +2 for whitespaces, +1 for '\0'
		con_str = malloc(con_str_size);
		snprintf(con_str, con_str_size, "%s %s %s", session, HOSTKEY_ACCEPT_NEW_PARAM, NEWINSTANCE_PARAM);
		con_str[con_str_size - 1] = '\0';
	} else {
		con_str = malloc(1);
		*con_str = '\0';
	}

	char winscp_call[strlen(winscp_bin) +
			 strlen(" /ini=\"") +
			 strlen(winscp_ini) +
			 strlen(con_str) + 8];
	strcpy(winscp_call, "\"");
	strcat(winscp_call, winscp_bin);
	strcat(winscp_call, "\" /ini=\"");
	strcat(winscp_call, winscp_ini);
	strcat(winscp_call, "\" ");
	strcat(winscp_call, con_str);

	free(con_str);

	log_debug("WinSCP call: %s", winscp_call);

	/* Clean up the mem for process' and pipes' init structures. */
	ZeroMemory(&winscp_sa, sizeof(winscp_sa));
	ZeroMemory(&winscp_pi, sizeof(winscp_pi));
	ZeroMemory(&winscp_si, sizeof(winscp_si));

	/* Init SECURITY_ATTRIBUTES winscp_sa, needed for the anonymous pipes. */
	winscp_sa.nLength = sizeof(SECURITY_ATTRIBUTES);
	winscp_sa.bInheritHandle = TRUE;
	winscp_sa.lpSecurityDescriptor = NULL;

	/* Set up the process' input and output pipes (stdin & stdout redirections). */
	error_check(!CreatePipe(&winscp_child_o_read, &winscp_child_o_write, &winscp_sa, 0),
		    L"Could not create output pipe.");
	error_check(!SetHandleInformation(winscp_child_o_read, HANDLE_FLAG_INHERIT, 0),
		    L"Could not set handle information for winscp_child_o_read.");
	error_check(!CreatePipe(&winscp_child_i_read, &winscp_child_i_write, &winscp_sa, 0),
		    L"Could not create input pipe.");
	error_check(!SetHandleInformation(winscp_child_i_write, HANDLE_FLAG_INHERIT, 0),
		    L"Could not set handle information for winscp_child_i_write.");

	/* Init STARTUPINFO winscp_si. */
	winscp_si.cb = sizeof(winscp_si);
	winscp_si.hStdInput = winscp_child_i_read;
	winscp_si.hStdOutput = winscp_child_o_write;
	winscp_si.hStdError = winscp_child_o_write;
	winscp_si.dwFlags |= STARTF_USESTDHANDLES;
	winscp_si.wShowWindow = SW_HIDE;

	/* Launch the process. */
	error_check(!CreateProcess(NULL,             // No module name
				   winscp_call,
				   NULL,             // Process handle not inheritable
				   NULL,             // Thread handle not inheritable
				   TRUE,             // Set handle inheritance
				   CREATE_NO_WINDOW, // creation flags
				   NULL,             // Use parent's environment block
				   NULL,             // Use parent's starting directory
				   &winscp_si,              // Pointer to STARTUPINFO structure
				   &winscp_pi),             // Pointer to PROCESS_INFORMATION structure
		    L"Das Erstellen des WinSCP-Prozesses schlug fehl.");
#endif
}


/**
 * Stops a running WinSCP child process, removes all used handlers and frees the mem.
 */
void winscp_cleanup() {
#if defined(WIN32) || defined(WIN64)
	winscp_wait_for_prompt();
	winscp_send_command("\r\r\r\r\r");
	Sleep(300);
	winscp_send_command("exit");
	if (WaitForSingleObject(winscp_pi.hProcess, 10000) == WAIT_TIMEOUT)
		winscp_send_command("exit");

	if (WaitForSingleObject(winscp_pi.hProcess, 10000) == WAIT_TIMEOUT) {
		log_error("ERROR: WinSCP does not want to stop. I kill it.");
		TerminateProcess(winscp_pi.hProcess, -1);
		WaitForSingleObject(winscp_pi.hProcess, 5000);
	}

	CloseHandle(winscp_child_i_read);
	CloseHandle(winscp_child_i_write);
	CloseHandle(winscp_child_o_read);
	CloseHandle(winscp_child_o_write);
	CloseHandle(winscp_pi.hProcess);
	CloseHandle(winscp_pi.hThread);

	if (winscp_host_fingerprint != NULL) {
		error_check(winscp_ini == NULL, L"Host fingerprint set but no ini in winscp_cleanup.");
		set_winscp_ini_value(winscp_ini, "[Sessions\\tgpro]", "HostFingerprint", winscp_host_fingerprint);
	}

	winscp_prompt_ready = 0;

	/* I do not free the mem of winscp_bin and winscp_ini because I could use them in the future. */

	log_trace("Cleanup finished");
#endif
}
