#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#if defined(WIN32) || defined(WIN64)
#include <windows.h>
#include <direct.h>
#include <iconv.h>
#include <lmcons.h>
#include <tchar.h>
#else
#include <sys/stat.h>
#include <errno.h>
#endif
#include <ctype.h>
#include "log.h"
#include "tgpro_environment.h"
#include "winscp_connector.h"
#include "winscp_ini_handler.h"

#ifdef WIN32
#define mkdir(path, mode) _mkdir(path)
#endif

char* tgpro_user_name = NULL;


/**
 * Returns a pointer to a string containing the TG-Pro user name.
 * This could be the same like the windows user name, but it need
 * not.
 * If the given ini_file_path is NULL it returns the OS's user name.
 * Do not try to free the memory of the string you got. If it is from
 * the OS, it could point to a system internal data structure.
 * NOTE: user_name from winscp.ini is never freed, too.
 */
const char* get_tgpro_user_name(const char* ini_file_path, const char* section) {
	if (tgpro_user_name)
		return tgpro_user_name;

	if (ini_file_path) {
		tgpro_user_name = get_winscp_ini_value(ini_file_path, section, "UserName");
#if 0
		/* UserName may be UTF-8 -> convert to URL format, if not yet done */
		if (tgpro_user_name) {
			char * result = (char *) malloc(strlen(tgpro_user_name) * 3 + 1);
			if (result) {
				char * in_pos = tgpro_user_name;
				char * out_pos = result;
				int percentfound = 0;

				while (*in_pos) {
					if (*in_pos == '%') {
						percentfound = 1;
						break;
					} else if (isalnum(*in_pos) || *in_pos == '_' || *in_pos == '-' || *in_pos == '.' || *in_pos == '~' || *in_pos == '%') {
						*out_pos = *in_pos;
						out_pos++;
					} else {
						out_pos += sprintf(out_pos, "%%%02X", tolower(*in_pos));
					}
					in_pos++;
				}
				if (percentfound) {
					free(result);
				} else {
					*out_pos = 0;
					free(tgpro_user_name);
					tgpro_user_name = result;
				}
			}
		}
#endif
	}
#if 0
#if defined(WIN32) || defined(WIN64)
	if (!tgpro_user_name) {
		/* Get the Windows user name and convert to UTF-8 and lowercase. */
		char * lower_caser;
		iconv_t iconv_desc;
		char * iconv_inbuf;
		char * iconv_outbuf;
		char * iconv_outbuf_start = iconv_outbuf;
		size_t iconv_inleft;
		size_t iconv_inlen;
		size_t iconv_outleft;
		size_t iconv_result;
		DWORD max_uname_len = UNLEN + 1;
		LPWSTR win_user_name = calloc(max_uname_len, sizeof(TCHAR));
		error_check(!win_user_name, L"Ich konnte keinen Speicher belegen.");
		error_check(!GetUserNameW(win_user_name, &max_uname_len), L"Ich konnte den Betriebssystem-Usernamen nicht feststellen.");

		/* Convert from UTF-16LE to UTF-8 */
		iconv_desc = iconv_open ("UTF-8//TRANSLIT", "UTF-16LE");
		error_check((iconv_desc == (iconv_t) -1), L"Ich konnte die Konvertierung nicht initialisieren.");
		iconv_inlen = max_uname_len;
		iconv_inleft = iconv_inlen;
		iconv_inbuf = (char *) win_user_name;
		iconv_outleft = iconv_inleft * 2;
		iconv_outbuf = (char *) malloc(iconv_outleft + 1);
		error_check(!iconv_outbuf, L"Ich konnte keinen Speicher belegen.");
		iconv_result = iconv(iconv_desc, &iconv_inbuf, &iconv_inleft, &iconv_outbuf, &iconv_outleft);
		iconv_close(iconv_desc);
		free(win_user_name);
		error_check((iconv_result == (size_t) -1), L"Ich konnte den Windows-Benutzernamen nicht konvertieren.");
		*iconv_outbuf = 0;
		tgpro_user_name = iconv_outbuf_start;
		for (lower_caser = tgpro_user_name; *lower_caser != '\0'; lower_caser++)
			*lower_caser = tolower(*lower_caser);
	}
#endif /* WIN32 and WIN64 */
#endif /* 0 */

#if defined(WIN32) || defined(WIN64)
	if (!tgpro_user_name) {
		tgpro_user_name = strdup(get_user_name());
		char * lower_caser;
		for (lower_caser = tgpro_user_name; *lower_caser != '\0'; lower_caser++) {
			*lower_caser = tolower(*lower_caser);
		}
	}
#endif /* WIN32 and WIN64 */
	error_check(!tgpro_user_name, L"Der TG-Pro-Username konnte nicht ermittelt werden.");

	log_debug("The new user name is %s", tgpro_user_name);

	return tgpro_user_name;
}


/**
 * Checks if this is a valid ini-file key-value definition ("foo=bar"). Comapares
 * key to the left-side value and returns 1 if there are equal. Returns 0 if there
 * are not equal.
 */
int cmp_key(const char* text_line, const char* key) {
	if (!text_line || !key)
		return 0;
	const size_t text_line_len = strlen(text_line);
	const size_t key_len = strlen(key);
	if (key_len > text_line_len ||
	    text_line_len > 4096)
		return 0;

	char key_copy[key_len + 1];
	memcpy(key_copy, key, key_len);
	key_copy[key_len] = '\0';
	char* trimmed_key = str_trim(key_copy);

	const char* equals_sign = strchr(text_line, '=');
	if (!equals_sign)
		return 0;
	const size_t line_key_len = equals_sign - text_line;
	char line_key[line_key_len + 1];
	strncpy(line_key, text_line, line_key_len);
	line_key[line_key_len] = '\0';
	char* trimmed_line_key = str_trim(line_key);

	return !strcasecmp(trimmed_key, trimmed_line_key);
}


/**
 * Grabs the strings on the left side (key) and the right side (value) of
 * the equals sign in text_line. Changes the pointers key and value to
 * their positions inside of text_line.
 * Danger: This does also manipulate text_line. The equals sign is replaced
 * by a \0.
 */
void get_key_value(char* text_line, char** key, char** value) {
	error_check(strlen(text_line) < 3, L"Zu kurze Eingabe in get_key_value.");
	*key = text_line;
	*value = strchr(text_line, '=');
	if (*value == NULL) {
		log_trace("get_key_value text_line: %s", text_line);
	}
	error_check(*value == NULL, L"Kein Gleichheitszeichen in get_key_value.");
	**value = '\0';
	++(*value);
	str_trim(*value);
}


/**
 * Returns TRUE if text_line represents the beginning of a section in
 * a WinSCP configuration file (winscp.ini).
 */
int is_section(char* text_line) {
	return *text_line == '[';
}

int is_numeric(char* str) {
	while (*str >= '0' && *str <= '9')
		str++;
	return *str == '\0';
}

char* get_winscp_remote_directory(const char* ini_file, const int ini_type, char* section) {
	char* rd_prefix = ini_type == WINSCP_INI_TYPE_SPOOL ? "/home/user/.spool/" : "/home/user/.transfer/";
	char* tgpro_user_name = strdup(get_winscp_ini_value(ini_file, section, "UserName"));
	if (!tgpro_user_name) {
		tgpro_user_name = strdup(get_user_name());
		if (tgpro_user_name) {
			/* Lower-case the Windows user name. TG-Pro uses internally always lower-case user names. */
			char* lower_caser;

			for (lower_caser = tgpro_user_name; *lower_caser != '\0'; lower_caser++)
				*lower_caser = tolower(*lower_caser);
		}
	}
	// TODO: This assumes tgpro_user_name isn't NULL. This is most probably but not necessarily the case!
	char* rd_value = malloc(strlen(rd_prefix) + strlen(tgpro_user_name) + 2);
	strcpy(rd_value, rd_prefix);
	char* pivot = strchr(tgpro_user_name, '+');
	if (pivot)
		strcat(rd_value, pivot + 1);
	else {
		strcat(rd_value, tgpro_user_name);
		if (is_numeric(tgpro_user_name))
			strcat(rd_value, "#");
	}
	free(tgpro_user_name);
	return rd_value;
}

/**
 * Sets the value for RemoteDirectory in the WinSCP ini file.
 */
void set_winscp_remote_directory(const char* ini_file, const int ini_type, char* section) {
	char* remdir = get_winscp_remote_directory(ini_file, ini_type, section);
	log_debug("I set remote dir in ini file to %s", remdir);
	set_winscp_ini_value(ini_file, section, "RemoteDirectory", remdir);
	free(remdir);
}


void set_winscp_ini_user_name(const char* ini_file) {
	set_winscp_ini_value(ini_file, "[Sessions\\tgpro]", "UserName", get_tgpro_user_name(ini_file, "[Sessions\\tgpro]"));
}


/**
 * Creates a WinSCP ini file. Writes one section: Sessions\tgpro.
 * All important data for TG-Pro communication are written.
 * An old ini_file is not deleted! Only the given key / value pairs
 * are overwritten.
 */
void generate_winscp_ini(const char* ini_file, const int ini_type) {
	log_debug("I generate a new winscp ini file.");
	char* section = "[Sessions\\tgpro]";

	set_winscp_remote_directory(ini_file, ini_type, section);
	set_winscp_ini_user_name(ini_file);
	set_winscp_ini_value(ini_file, section, "HostName", get_host_name());
	set_winscp_ini_value(ini_file, section, "LogicalHostName", get_krb_host_name());
	set_winscp_ini_value(ini_file, section, "AuthKI", "0");
	set_winscp_ini_value(ini_file, section, "AuthKIPassword", "0");
	set_winscp_ini_value(ini_file, section, "AuthGSSAPI", "1");
	set_winscp_ini_value(ini_file, section, "GSSAPIFwdTGT", "1");
}


void generate_winscp_ini_defaultpath(const int ini_type) {
	generate_winscp_ini(winscp_get_ini(), ini_type);
}


/**
 * Looks for a value in the winscp.ini file.
 * Searchs only in the given section for the given key. Returns
 * the key's value in string. If this key does not exist it returns
 * a NULL pointer.
 * Danger: this function allocates memory for the return string. Please
 * free() that memory after use.
 */
char* get_winscp_ini_value(const char* ini_file_path, const char* section, const char* key) {
	FILE* winscp_ini_file = fopen(ini_file_path, "r");
	if (!winscp_ini_file) {
		log_error("%s: Failed to open %s to read value (%s) from it!", __func__, ini_file_path, key);
		return NULL;
	}

	char current_section[4096] = "";
	char text_line_buffer[4096];
	while (fgets(text_line_buffer, 4095, winscp_ini_file)) {
		char* text_line = str_trim(text_line_buffer);
		if (is_section(text_line))
			strcpy(current_section, text_line);
		else
			if (!strcasecmp(current_section, section) &&
			    cmp_key(text_line, key)) {
				fclose(winscp_ini_file);
				char* value = strchr(text_line, '=');
				error_check(!value, L"get_winscp_ini_value() konnte den Key-Wert nicht finden.");
				return strdup(str_trim(++value));
			}
	}

	fclose(winscp_ini_file);
	return NULL;
}


/**
 * Deletes a line from the ini file.
 */
void delete_winscp_ini_key(const char* ini_file_path, const char* delete_key) {
	log_trace("%s", __func__);
	FILE* winscp_ini_file = fopen(ini_file_path, "rb");
	if (!winscp_ini_file) {
		log_error("%s: Failed to open %s to read value (%s) from it!", __func__, ini_file_path, delete_key);
		return;
	}

	/* Read the whole winscp.ini file into memory. */
	fseek(winscp_ini_file, 0, SEEK_END);
	long fsize = ftell(winscp_ini_file);
	fseek(winscp_ini_file, 0, SEEK_SET);
	char* winscp_content = malloc(fsize + 1);
	error_check(!winscp_content, L"Ich konnte keinen Speicher zum Auslesen der ini-Datei anfordern.");
	size_t read_bytes = fread(winscp_content, fsize, 1, winscp_ini_file);
	error_check(read_bytes < 0, L"Etwas ging schief beim Aulesen der ini-Konfigurationsdatei.");
	fclose(winscp_ini_file);
	winscp_content[fsize] = '\0';

	/* Write whole data to a new winscp.ini without the key to delete. */
	winscp_ini_file = fopen(ini_file_path, "w");
	if (!winscp_ini_file) {
		log_error("%s: Failed to write to %s! Probably missing permissions", __func__, ini_file_path);
		errno = EACCES;
		error_check(1, L"Fehlende Schreibrechte! Die Schleuse benötigt Schreibzugriff auf die Konfigurationsdatei (transfer.ini).\n\nBitte Berechtigungen des Konfiguratiosverzeichnisses prüfen (lassen).");
	}

	char* text_line = strtok(winscp_content, "\n");
	while (text_line) {
		if (!cmp_key(text_line, delete_key)) {
			for (char* tl_pt = text_line; *tl_pt; ++tl_pt)
				if (*tl_pt != '\r' && *tl_pt != '\n')
					fputc(*tl_pt, winscp_ini_file);
			fputc('\r', winscp_ini_file);
			fputc('\n', winscp_ini_file);
		}
		text_line = strtok(NULL, "\n");
	}

	fclose(winscp_ini_file);
	free(winscp_content);
}

void create_parent_dir_if_necessary(const char* ini_file_path) {
	char* dir = strdup(ini_file_path);
	if (!dir) {
		log_error("Error! Failed to create parent folder for %s (couldn't strdup string)", ini_file_path);
		return;
	}

	char* transfer_ini = strstr(dir, "transfer.ini");
	if (!transfer_ini) {
		log_error("Error! This is not a path to a 'transfer.ini' file and can't be handled: %s.", ini_file_path);
		return;
		free(dir);
	}

	transfer_ini[0] = 0;

	if (dir_exists(dir)) {
		free(dir);
		return;
	}

	char* mkdir_cmd = (char*) malloc(4096);
	if (!mkdir_cmd) {
		log_error("Error! Failed to create parent folder for %s (couldn't allocate enough memory)", ini_file_path);
		return;
	}

	int create_dir_succeeded = (mkdir(dir, 700) == 0 || errno == EEXIST) ? 1 : 0;
	log_debug("Creating directory (%s) for transfer.ini %s", dir, create_dir_succeeded ? "succeeded" : "failed");

	free(mkdir_cmd);
	free(dir);
}

/**
 * Changes the value of the key-value pair in the winscp.ini in the
 * given section.
 * If the give key does not exist in winscp.ini it is created.
 * If the section does not exist it is created at the end of the file.
 */
void set_winscp_ini_value(const char* ini_file_path, const char* new_section, const char* new_key, const char* new_value) {
	create_parent_dir_if_necessary(ini_file_path);

	char* old_value = get_winscp_ini_value(ini_file_path, new_section, new_key);
	if (old_value && strcmp(old_value, new_value) == 0) {
		free(old_value);
		return;
	}
	free(old_value);

	/* Read the whole winscp.ini file into memory. */
	FILE* winscp_ini_file = fopen(ini_file_path, "rb");
	char* winscp_content;
	if (winscp_ini_file) {
		fseek(winscp_ini_file, 0, SEEK_END);
		long fsize = ftell(winscp_ini_file);
		fseek(winscp_ini_file, 0, SEEK_SET);
		winscp_content = malloc(fsize + 1);
		error_check(!winscp_content, L"Ich konnte keinen Speicher zum Auslesen der ini-Datei anfordern.");
		size_t read_bytes = fread(winscp_content, fsize, 1, winscp_ini_file);
		error_check(read_bytes < 0, L"Etwas ging schief beim Aulesen der ini-Konfigurationsdatei.");
		fclose(winscp_ini_file);
		winscp_content[fsize] = '\0';
	} else {
		size_t new_section_len = strlen(new_section);
		winscp_content = malloc(new_section_len + 3);
		error_check(!winscp_content, L"Ich konnte keinen Speicher zum Anlegen der ini-Datei anfordern.");
		strcpy(winscp_content, new_section);
		winscp_content[new_section_len + 0] = '\r';
		winscp_content[new_section_len + 1] = '\n';
		winscp_content[new_section_len + 2] = '\0';
	}

	/* Write whole data to a new winscp.ini and insert the new data. */
	winscp_ini_file = fopen(ini_file_path, "w");
	if (!winscp_ini_file) {
		log_error("%s: Failed to write to %s! Probably missing permissions", __func__, ini_file_path);
		errno = EACCES;
		error_check(1, L"Fehlende Schreibrechte! Die Schleuse benötigt Schreibzugriff auf die Konfigurationsdatei (transfer.ini).\n\nBitte Berechtigungen des Konfiguratiosverzeichnisses prüfen (lassen).");
	}
	char* current_section = "nirvana";
	char* text_line  = strtok(winscp_content, "\r\n");
	char* key = NULL;
	char* value = NULL;
	int pair_written = 0;  /* Did I write the new key-value-pair? */
	while (text_line) {
		text_line = str_trim(text_line);
		if (is_section(text_line)) {
			if (strcasecmp(current_section, new_section) == 0) {
				if (!pair_written) {
					fputs(new_key, winscp_ini_file);
					fputc('=', winscp_ini_file);
					fputs(new_value, winscp_ini_file);
					fputc('\r', winscp_ini_file);
					fputc('\n', winscp_ini_file);
					pair_written = 1;
				}
			}
			if (strcmp(current_section, "nirvana") != 0) {
				fputc('\r', winscp_ini_file);
				fputc('\n', winscp_ini_file);
			}
			current_section = text_line;
			fputs(text_line, winscp_ini_file);
			fputc('\r', winscp_ini_file);
			fputc('\n', winscp_ini_file);
		} else {
			if (strlen(text_line) > 0) {
				get_key_value(text_line, &key, &value);
				fputs(key, winscp_ini_file);
				if (strcasecmp(key, new_key) == 0 &&
				    strcasecmp(current_section, new_section) == 0) {
					if (new_value) {
						fputc('=', winscp_ini_file);
						fputs(new_value, winscp_ini_file);
					}
					pair_written = 1;
				} else if (value) {
					fputc('=', winscp_ini_file);
					fputs(value, winscp_ini_file);
				}
				fputc('\r', winscp_ini_file);
				fputc('\n', winscp_ini_file);
			}
		}

		text_line = strtok(NULL, "\r\n");
		/* We are at the end of the original data. If I did not write the new data I
		   must do it now. */
		if ((!text_line) && (!pair_written)) {
			if (strncasecmp(current_section, new_section, strlen(current_section)) != 0) {
				fputc('\r', winscp_ini_file);
				fputc('\n', winscp_ini_file);
				fputs(new_section, winscp_ini_file);
				fputc('\r', winscp_ini_file);
				fputc('\n', winscp_ini_file);
			}
			fputs(new_key, winscp_ini_file);
			fputc('=', winscp_ini_file);
			fputs(new_value, winscp_ini_file);
			fputc('\r', winscp_ini_file);
			fputc('\n', winscp_ini_file);
			pair_written = 1;
		}
	}

	fclose(winscp_ini_file);
	free(winscp_content);
}
