/*************************************************** */
/* Rule Set Based Access Control                     */
/*                                                   */
/* Author and (c) 1999-2021: Amon Ott <ao@rsbac.org> */
/*                                                   */
/* Last modified: 16/Jun/2021                        */
/*************************************************** */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <pwd.h>
#include <grp.h>
#include <iconv.h>

#include <rsbac/types.h>
#include <rsbac/aci_data_structures.h>
#include <rsbac/getname.h>
#include <rsbac/res_getname.h>
#include <rsbac/syscalls.h>
#include <rsbac/error.h>
#include <rsbac/helpers.h>
#include "nls.h"
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

char * progname;

void use(void)
  {
    printf(gettext("%s (RSBAC %s)\n***\n"), progname, VERSION);
    printf(gettext("Use: %s [switches] module user attribute [position|request-name]\n\n"), progname);  
    printf(gettext(" -h = this help, -- = no more flags,\n"));
    printf(gettext(" -e = show effective (maybe inherited) value, not real\n"));
    printf(gettext(" -n = numeric value, -b = both names and numbers,\n"));
    printf(gettext(" -z = do not also try numeric username as uid\n"));
    printf(gettext(" -l list all users, -L list all Linux groups\n"));
    printf(gettext(" -p = print request names instead of values\n"));
    printf(gettext(" -Z = truncate names at first invalid UTF-8 character,\n"));
    printf(gettext(" -c list all Linux capabilities, -R = list all RES resource names\n"));
    printf(gettext(" -a = list attributes and values\n"));
    printf(gettext(" -N ta = transaction number (default = value of RSBAC_TA, if set, or 0)\n"));
    printf(gettext(" module = CAP, GEN, MAC, FC, SIM, UDF, DAZ, FF, RC or AUTH\n"));
    printf(gettext(" mac_[min_]categories\t\t(with additional parameter position)\n\t\t\t0=no, 1=yes\n"));
    printf(gettext(" log_user_based\t(with additional parameter request-name)\n\t\t\t0=no, 1=yes\n"));
  }

/* returns 1, if string has been changed, and 0, if not */
int do_sanitize(char * string, unsigned int buflen)
{
  iconv_t iconv_string_utf8_desc;
  int len = strlen(string);
  char * iconv_inbuf;
  char * iconv_outbuf;
  size_t iconv_inleft;
  size_t iconv_inlen;
  size_t iconv_outleft;
  size_t iconv_result = (size_t) -1;
  char * iconv_outbuf_start;
  unsigned int newlen;

  if (!string || !buflen)
    return 0;

  iconv_string_utf8_desc = iconv_open ("UTF-8//TRANSLIT", "UTF-8");
  if (iconv_string_utf8_desc == (iconv_t) -1) {
    fprintf(stderr, gettext("%s: do_sanitize(): iconv_open() failed with error: %s\n"), progname, strerror(errno));
    exit(1);
  }

  iconv_inlen = len;
  iconv_inbuf = string;
  iconv_outleft = iconv_inlen * 2;
  iconv_outbuf = (char *) malloc(iconv_outleft + 1);
  if (!iconv_outbuf) {
    fprintf(stderr, gettext("%s: do_sanitize(): failed to malloc\n"), progname);
    exit(1);
  }
  iconv_outbuf_start = iconv_outbuf;

  while(iconv_result == (size_t) -1 && iconv_inlen > 0) {
    iconv_inleft = iconv_inlen;
    iconv_result = iconv(iconv_string_utf8_desc, &iconv_inbuf, &iconv_inleft, &iconv_outbuf, &iconv_outleft);
    if (iconv_result == (size_t) -1) {
      iconv_inlen--;
      string[iconv_inlen] = 0;
    }
  }
  if (iconv_inlen > 0) {
    *iconv_outbuf = 0;
    newlen = iconv_outbuf - iconv_outbuf_start;
    if (newlen >= buflen)
      newlen = buflen - 1;
    strncpy(string, iconv_outbuf_start, newlen);
    string[newlen] = 0;
  } else {
    string[0] = 0;
    newlen = 0;
  }

  free(iconv_outbuf_start);
  iconv_close(iconv_string_utf8_desc);
  if (newlen == len)
    return 0;
  else
    return 1;
}

int main(int argc, char ** argv)
{
  int attr_list[RSBAC_USER_NR_ATTRIBUTES] = RSBAC_USER_ATTR_LIST;
  int res = 0;
  u_int position;
  char tmp1[RSBAC_MAXNAMELEN],tmp2[RSBAC_MAXNAMELEN],tmp3[RSBAC_MAXNAMELEN];
  int j;
  enum rsbac_switch_target_t module = SW_NONE;
  union rsbac_attribute_value_t value;
  union rsbac_target_id_t tid;
  enum rsbac_attribute_t attr;
  rsbac_res_limit_t rlimit_value;
  rsbac_time_t ttl;
  int inherit = 0;
  int numeric = 0;
  int both = 0;
  int bothr = 0;
  int nonumeric = 0;
  int printall = 0;
  int scripting = 0;
  int sanitize = 0;
  rsbac_list_ta_number_t ta_number = 0;
  u_int stopflags = FALSE;

  progname = argv[0];
  locale_init();
  {
    char * env = getenv("RSBAC_TA");

    if(env)
      ta_number = strtoul(env,0,0);
  }

  while((argc > 1) && (argv[1][0] == '-') && !stopflags)
    {
      char * pos = argv[1];
      pos++;
      while(*pos)
        {
          switch(*pos)
            {
              case '-':
                stopflags = TRUE;
                break;
              case 'h':
                use();
                return 0;
              case 'e':
                inherit=1;
                break;
              case 'p':
                printall=1;
                break;
              case 'n':
                numeric=1;
                break;
              case 'b':
                both=1;
                break;
              case 'B':
                bothr=1;
                break;
              case 'z':
                nonumeric=1;
                break;
              case 's':
                scripting=1;
                break;
              case 'Z':
                sanitize = 1;
                break;
              case 'l':
                {
                  struct passwd * user_info_p; 

                  setpwent();
                  while((user_info_p = getpwent()))
                    {
                      if(numeric)
                        printf("%u\n", user_info_p->pw_uid);
                      else
                      if(both)
                        printf("%s %u\n", user_info_p->pw_name, user_info_p->pw_uid);
                      else
                      if(bothr)
                        printf("%u %s\n", user_info_p->pw_uid, user_info_p->pw_name);
                      else
                        printf("%s\n", user_info_p->pw_name);
                    }
                  exit(0);
                }
              case 'L':
                {
                  struct group * group_info_p; 

                  setgrent();
                  while((group_info_p = getgrent()))
                    {
                      if(numeric)
                        printf("%u\n", group_info_p->gr_gid);
                      else
                      if(both)
                        printf("%s %u\n", group_info_p->gr_name, group_info_p->gr_gid);
                      else
                      if(bothr)
                        printf("%u %s\n", group_info_p->gr_gid, group_info_p->gr_name);
                      else
                        printf("%s\n", group_info_p->gr_name);
                    }
                  exit(0);
                }
              case 'u':
                if(argc > 2)
                  {
                    rsbac_uid_t uid;

                    if(rsbac_get_uid(ta_number, &uid, argv[2], nonumeric))
                      {
                        fprintf(stderr, gettext("%s: Invalid User %s!\n"),
                                progname, argv[2]);
                        exit(1);
                      }
                    if (RSBAC_UID_SET(uid))
                      printf("%u/%u\n", RSBAC_UID_SET(uid), RSBAC_UID_NUM(uid));
                    else
                      printf("%u\n", RSBAC_UID_NUM(uid));
                    exit(0);
                  }
                else
                  {
                    fprintf(stderr, "Missing argument to parameter u!\n");
                    exit(1);
                  }
              case 'c':
                {
                  char tmp[RSBAC_MAXNAMELEN];
                  int i;

                  for(i=0; i<CAP_NONE; i++)
                    printf("%s\n", get_cap_name(tmp, i));
                  exit(0);
                }
              case 'R':
                {
                  char tmp[RSBAC_MAXNAMELEN];
                  int i;

                  for(i=0; i<=RSBAC_RES_MAX; i++)
                    printf("%s\n", get_res_name(tmp, i));
                  exit(0);
                }
              case 'a':
              case 'A':
                if(   (argc > 2)
                   && ((attr = get_attribute_nr(argv[2])) != A_none)
                  )
                  {
                    get_switch_target_name(tmp1, get_attr_module(attr));
                    get_attribute_name(tmp2, attr);
                    get_attribute_param(tmp3, attr);
                    printf("[%-4s] %s\n\t%s\n",tmp1,tmp2,tmp3);
                    exit(0);
                  }
                printf(gettext("- attribute (string) and returned value = see following list:\n"));
                for (j=0;j<RSBAC_USER_NR_ATTRIBUTES;j++)
                  {
                    get_switch_target_name(tmp1, get_attr_module(attr_list[j]));
                    get_attribute_name(tmp2,attr_list[j]);
                    get_attribute_param(tmp3,attr_list[j]);
                    printf("[%-4s] %s\n\t%s\n",tmp1,tmp2,tmp3);
                  }
                printf(gettext("[RES ] res_min|res_max (with additional parameter position)\n\tnon-negative integer (0 for unset)\n"));
                exit(0);
              case 'N':
                if(argc > 2)
                  {
                    ta_number = strtoul(argv[2], 0, 10);
                    argc--;
                    argv++;
                  }
                else
                  {
                    fprintf(stderr, gettext("%s: missing transaction number value for parameter %c\n"), progname, *pos);
                    exit(1);
                  }
                break;

              default:
                fprintf(stderr, gettext("%s: unknown parameter %c\n"), progname, *pos);
                exit(1);
            }
          pos++;
        }
      argv++;
      argc--;
    }
  if(argc > 1)
    {
      module = get_switch_target_nr(argv[1]);
      if(module != SW_NONE)
        {
          argv++;
          argc--;
        }
    }
  switch(argc)
    {
      case 3:
        if(   !strcmp("group_nr",argv[2])
           || !strcmp("group_name",argv[2])
          )
          {
            if(rsbac_get_gid_name(ta_number, &tid.group, tmp1, argv[1]))
              {
                fprintf(stderr, gettext("%s: Invalid Group %s!\n"),
                        progname, argv[1]);
                exit(1);
              }
            if(!strcmp("group_nr",argv[2]))
              {
                if (RSBAC_GID_SET(tid.group))
                  printf("%u/%u\n", RSBAC_GID_SET(tid.group), RSBAC_GID_NUM(tid.group));
                else
                  printf("%u\n", RSBAC_GID_NUM(tid.group));
                exit(0);
              }
            else
              {
                if (sanitize)
                  do_sanitize(tmp1, RSBAC_MAXNAMELEN);
                printf("%s\n", tmp1);
                exit(0);
              }
          }
        value.dummy = -1;
        if(rsbac_get_uid_name(ta_number, &tid.user, tmp1, argv[1], nonumeric))
          {
            fprintf(stderr, gettext("%s: Invalid User %s!\n"),
                    progname, argv[1]);
            exit(1);
          }
        if(!strcmp("user_nr",argv[2]))
          {
            if (RSBAC_UID_SET(tid.user))
              printf("%u/%u\n", RSBAC_UID_SET(tid.user), RSBAC_UID_NUM(tid.user));
            else
              printf("%u\n", RSBAC_UID_NUM(tid.user));
            exit(0);
          }
        if(!strcmp("user_name",argv[2]))
          {
            if (sanitize)
              do_sanitize(tmp1, RSBAC_MAXNAMELEN);
            printf("%s\n", tmp1);
            exit(0);
          }
        if(!strcmp("full_name",argv[2]))
          {
            if(!rsbac_get_fullname(ta_number, tmp2, tid.user)) {
              if (sanitize)
                do_sanitize(tmp2, RSBAC_MAXNAMELEN);
              printf("%s\n", tmp2);
            }
            exit(0);
          }

        attr = get_attribute_nr(argv[2]);
        if(attr == A_none)
          {
            fprintf(stderr, gettext("%s: Invalid Attribute %s!\n"), progname, argv[2]);
            exit(1);
          }

        if (attr == A_res_min || attr == A_res_max)
          {
            int i;
            int tmperr = 0;

            for (i=0; i<RLIM_NLIMITS; i++)
              {
                res = rsbac_res_get_user_limit(ta_number, tid.user, attr, i, &rlimit_value, &ttl, inherit);
                if (res < 0)
                  {
                    if(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                       || (res == -RSBAC_ENOTFOUND)
                      )
                      {
                        if(scripting)
                          {
                            printf("u ");
                          }
                        else
                          {
                            get_res_name(tmp1,i);
                            printf("%s\tunset\n", tmp1);
                          }
                      }
                    else
                      {
//                        fprintf(stderr, gettext("%s: rsbac_res_get_user_limit() return error: "), progname);
//                        show_error(res);
                        tmperr = res;
                        break;
                      }
                  }
                else
                  {
                    if(scripting)
                      {
                        if (ttl)
                          printf("%llu[%us] ", rlimit_value, ttl);
                        else
                          printf("%llu ", rlimit_value);
                      }
                    else
                      {
                        get_res_name(tmp1,i);
                        if (ttl)
                          printf("%s\t%llu[%us]\n", tmp1, rlimit_value, ttl);
                        else
                          printf("%s\t%llu\n", tmp1, rlimit_value);
                      }
                  }
              }
            if(scripting)
              {
                printf("\n");
              }
            if (!tmperr)
              exit(0);
          }

        res = rsbac_get_attr(ta_number, module, T_USER, &tid, attr, &value, inherit);
        error_exit(res);
        switch(attr)
          {
            case A_mac_role:
            case A_mac_user_flags:
            case A_daz_role:
            case A_udf_role:
            case A_ff_role:
            case A_auth_role:
            case A_cap_role:
            case A_jail_role:
            case A_res_role:
            case A_pax_role:
            case A_system_role:
            case A_security_level:
            case A_initial_security_level:
            case A_min_security_level:
	    case A_cap_ld_env:
              printf("%u\n",value.u_char_dummy);
              break;
            case A_rc_type:
            case A_rc_type_fd:
            case A_rc_force_role:
            case A_rc_role:
            case A_rc_def_role:
              printf("%u\n",value.rc_role);
              break;
            case A_pseudo:
              printf("%u\n",value.pseudo);
              break;
            case A_mac_categories:
            case A_mac_initial_categories:
            case A_mac_min_categories:
              printf("%s\n",u64tostrmac(tmp1,value.mac_categories));
              break;
            case A_log_user_based:
              if(printall)
                {
                  int i;

                  for (i=0; i<R_NONE; i++)
                    if(value.log_user_based & ((rsbac_request_vector_t) 1 << i))
                      printf(" %s\n", get_request_name(tmp1,i));
                }
              else
                printf("%s\n",u64tostrlog(tmp1,value.log_user_based));
              break;
            case A_min_caps:
            case A_max_caps:
              if(printall)
                {
                  int i;

                  for (i=0; i<32; i++)
                    if(value.min_caps.cap[0] & ((__u32) 1 << i))
                      printf(" %s\n", get_cap_name(tmp1,i));
                  for (i=32; i<CAP_NONE; i++)
                    if(value.min_caps.cap[1] & ((__u32) 1 << (i - 32)))
                      printf(" %s\n", get_cap_name(tmp1,i));
                }
              else
                {
                  kcaptostrcap(tmp1,value.min_caps);
                  printf("%s\n", tmp1);
                }
              break;
            case A_res_min:
            case A_res_max:
                {
                  int i;

                  if(scripting)
                    {
                      for (i=0; i<=RSBAC_RES_MAX; i++)
                        printf("%u ", value.res_array[i]);
                      printf("\n");
                    }
                  else
                    for (i=0; i<=RSBAC_RES_MAX; i++)
                      {
                        get_res_name(tmp1,i);
                        printf("%s\t%u\n", tmp1, value.res_array[i]);
                      }
                }
              break;
            default:
              printf("%i\n",value.dummy);
          }
        exit(0);

      case 4:
        attr = get_attribute_nr(argv[2]);
        if(rsbac_get_uid(ta_number, &tid.user, argv[1], nonumeric))
          {
            fprintf(stderr, gettext("%s: Invalid User %s!\n"),
                    progname, argv[1]);
            exit(1);
          }
        switch(attr)
          {
            case A_mac_categories:
            case A_mac_initial_categories:
            case A_mac_min_categories:
              position = strtoul(argv[3],0,10);
              if(position > RSBAC_MAC_MAX_CAT)
                {
                  fprintf(stderr, gettext("Invalid position counter %s\n"), argv[3]);
                  exit(1);
                }
              res = rsbac_get_attr(ta_number, module, T_USER, &tid, attr, &value, inherit);
              error_exit(res);
              printf("%u\n",
                     (u_int) (value.mac_categories >> position) & 1);
              exit(0);

            case A_log_user_based:
              position = get_request_nr(argv[3]);
              if(position >= R_NONE)
                {
                  fprintf(stderr, gettext("Invalid request %s\n"), argv[3]);
                  exit(1);
                  }
                res = rsbac_get_attr(ta_number, module, T_USER, &tid, A_log_user_based, &value, inherit);
                error_exit(res);
                printf("%u\n",
                     (u_int) (value.log_program_based >> position) & 1);
                exit(0);

            case A_res_min:
            case A_res_max:
              position = get_res_nr(argv[3]);
              if(position == RSBAC_RES_NONE)
                {
                  position = strtoul(argv[3],0,10);
                  if(   (!position && strcmp(argv[3], "0"))
                     || (position >= RLIM_NLIMITS)
                    )
                    {
                      fprintf(stderr, gettext("Invalid position counter %s\n"), argv[3]);
                      exit(1);
                    }
                }
              res = rsbac_res_get_user_limit(ta_number, tid.user, attr, position, &rlimit_value, &ttl, inherit);
              if(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                 || (res == -RSBAC_ENOTFOUND)
                )
                {
                  printf("unset\n");
                  res = 0;
                }
              else if (!res)
                {
                  if (ttl)
                    printf("%llu[%us]\n", rlimit_value, ttl);
                  else
                    printf("%llu\n", rlimit_value);
                }
              else if(   ((res == -1) && (errno == RSBAC_EINVALIDREQUEST))
                      || (res == -RSBAC_EINVALIDREQUEST)
                )
                {
                  res = rsbac_get_attr(ta_number, module, T_USER, &tid, attr, &value, inherit);
                  error_exit(res);
                  printf("%u\n",
                     value.res_array[position]);
                }
              error_exit(res);
              exit(0);

            default:
              break;
          }

      default:
        use();
        return 1;
    }
  exit(1);
}
