/*************************************************** */
/* Rule Set Based Access Control                     */
/*                                                   */
/* Author and (c) 1999-2014: Amon Ott <ao@rsbac.org> */
/*                                                   */
/* Last modified: 16/Oct/2014                        */
/*************************************************** */

#include <stdio.h>
#include <stdlib.h>
#include <pwd.h>
#include <sys/types.h>
#include <rsbac/types.h>
#include <rsbac/getname.h>
#include <rsbac/syscalls.h>
#include <rsbac/error.h>
#include "nls.h"
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

char * progname;

void use(void)
    {
      printf(gettext("%s (RSBAC %s)\n***\n"), progname, VERSION);
      printf(gettext("Use: %s [flags] user(s)\n\n"), progname);  
      printf(gettext(" -h = this help, -- = no more flags,\n"));
      printf(gettext(" -z = do not also try numeric username as uid\n"));
      printf(gettext(" -N ta = transaction number (default = value of RSBAC_TA, if set, or 0)\n"));
    }

int main(int argc, char ** argv)
{
  int res = 0;
  int i;
  union rsbac_target_id_t tid;
  rsbac_list_ta_number_t ta_number = 0;
  u_int stopflags = FALSE;
  int nonumeric = 0;

  progname = argv[0];
  locale_init();
  
  {
    char * env = getenv("RSBAC_TA");

    if(env)
      ta_number = strtoul(env,0,0);
  }
  while((argc > 1) && (argv[1][0] == '-') && !stopflags)
    {
      char * pos = argv[1];
      pos++;
      while(*pos)
        {
          switch(*pos)
            {
              case '-':
                stopflags = TRUE;
                break;
              case 'h':
                use();
                return 0;
              case 'z':
                nonumeric = TRUE;
                break;
              case 'N':
                if(argc > 2)
                  {
                    ta_number = strtoul(argv[2], 0, 10);
                    argc--;
                    argv++;
                  }
                else
                  {
                    fprintf(stderr, gettext("%s: missing transaction number value for parameter %c\n"), progname, *pos);
                    exit(1);
                  }
                break;
              default:
                fprintf(stderr, gettext("%s: unknown parameter %c\n"), progname, *pos);
                exit(1);
            }
          pos++;
        }
      argv++;
      argc--;
    }

  if (argc > 1)
    {
      printf(gettext("%s: %i users\n\n"), progname, argc - 1);
      for (i=1;i < (argc);i++)
        {
          if(rsbac_get_uid(ta_number, &tid.user, argv[i], nonumeric))
            {
              fprintf(stderr, gettext("Invalid User %s!\n\n"), argv[i]);
              continue;
            }
          if (RSBAC_UID_SET(tid.user))
            printf(gettext("Processing user %s (uid %u/%u)\n"),
                   argv[i], RSBAC_UID_SET(tid.user), RSBAC_UID_NUM(tid.user));
          else
            printf(gettext("Processing user %s (uid %u)\n"),
                   argv[i], RSBAC_UID_NUM(tid.user));

          res = rsbac_remove_target(ta_number, T_USER, &tid);
          show_error(res);
        }
    }
  else
    {
      use();
      return 1;
    }
    
  return (res);
}
