/* Copyright 2017-2021 m-privacy GmbH
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 */

#include <winsock2.h>
#include <windows.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <poppler.h>
#include <cairo.h>
#include <cairo-win32.h>
#include <stdlib.h>
#include <stdio.h>

#include <assert.h>

#include <rfb/LogWriter.h>
#include <FL/fl_utf8.h>
#include "pdfprinter.h"

static rfb::LogWriter vlog("pdfprinter");

PRINTDLG* get_print_dlg(int num_pages) {
	PRINTDLG* print_dlg = (PRINTDLG*) malloc(sizeof(PRINTDLG));
	ZeroMemory(print_dlg, sizeof(PRINTDLG));

	print_dlg->lStructSize = sizeof(PRINTDLG);
	print_dlg->Flags = PD_RETURNDC
		| PD_NOSELECTION
		| PD_COLLATE
		| PD_USEDEVMODECOPIESANDCOLLATE
		| PD_DISABLEPRINTTOFILE
		| PD_HIDEPRINTTOFILE;
	print_dlg->nCopies = 1;
	print_dlg->nFromPage = 1;
	print_dlg->nToPage = num_pages;
	print_dlg->nMinPage = 1;
	print_dlg->nMaxPage = num_pages;
	print_dlg->hDevNames = NULL;
	print_dlg->hDevMode = NULL;

	int print_dlg_result = PrintDlg(print_dlg);
	if (!print_dlg_result) {
		if ((int) CommDlgExtendedError() == 0) {
			vlog.info("User canceled print dialog\n");
		} else {
			vlog.error("Print dialog ended with unkown error: CommDlgExtendedError() %d\n" , (int) CommDlgExtendedError());
		}
		free(print_dlg);
		return NULL;
	}
	if (!print_dlg->hDC) {
		vlog.error("I couldn't retrieve printer device context (hDC) from the print dialog\n");
		free(print_dlg);
		return NULL;
	}
	return print_dlg;
}

char* get_uri_from_path(const char* _path, GError* error) {
	vlog.info("get_uri_from_path()\n");

	size_t src_len = strlen(_path);
	char path[src_len * 2];
	fl_utf8froma(path, src_len * 2, _path, src_len);

	char* absolute;
	if (g_path_is_absolute(path)) {
		absolute = g_strdup (path);
	} else {
		gchar* dir = g_get_current_dir ();
		absolute = g_build_path (dir, path, (gchar *) 0);
	}
	char* uri;
	uri = g_filename_to_uri (absolute, NULL, &error);
	return uri;
}

void scale_page(PopplerPage* page, HDC hdc) {
	/* Inspired from poppler/utils/pdftocairo-win32.cc - if there
	   are scale problems, take a look at the original code */
	double source_w, source_h;
	poppler_page_get_size (page, &source_w, &source_h);

	// Transform the hdc scale to points to be consistent with other cairo backends
	int x_dpi = GetDeviceCaps (hdc, LOGPIXELSX);
	int y_dpi = GetDeviceCaps (hdc, LOGPIXELSY);
	int x_off = GetDeviceCaps (hdc, PHYSICALOFFSETX);
	int y_off = GetDeviceCaps (hdc, PHYSICALOFFSETY);

	double printable_w = GetDeviceCaps (hdc, PHYSICALWIDTH)*72.0/x_dpi;
	double printable_h = GetDeviceCaps (hdc, PHYSICALHEIGHT)*72.0/y_dpi;

	const double target_scale = min((printable_w / source_w), (printable_h/source_h));
	XFORM xform;
	xform.eM11 = x_dpi/72.0 * target_scale;
	xform.eM12 = 0;
	xform.eM21 = 0;
	xform.eM22 = y_dpi/72.0 * target_scale;
	xform.eDx = -x_off;
	xform.eDy = -y_off;

	if(!SetGraphicsMode (hdc, GM_ADVANCED)) {
		vlog.error("scale_page(): I could not SetGraphicsMode()\n");
	}
	if (!SetWorldTransform (hdc, &xform)) {
		vlog.error("scale_page(): I could not SetWorldTransform()\n");
	}
}

int do_print(PRINTDLG* print_dlg, PopplerDocument* document) {
	vlog.info("do_print()\n");
	cairo_surface_t* surface = cairo_win32_printing_surface_create (print_dlg->hDC);
	cairo_status_t surface_status = cairo_surface_status(surface);
	if (surface_status) {
		vlog.error("cairo_win32_printing_surface_create(print_dlg->hDC) failed: %s\n.Aborting printing...", cairo_status_to_string(surface_status));
		return -1;
	} else {
		vlog.debug("cairo_win32_printing_surface_create(print_dlg->hDC) succeeded.");
	}
	cairo_t* cr = cairo_create (surface);
	cairo_status_t cr_status = cairo_status(cr);
	if (cr_status) { // CAIRO_STATUS_SUCCESS = 0
		vlog.error("cairo_create(surface) failed: %s\n.Aborting printing...", cairo_status_to_string(cr_status));
		return -1;
	} else {
		vlog.debug("cairo_create(surface) succeeded.");
	}

	vlog.verbose("Printing from page %d to page %d", print_dlg->nFromPage, print_dlg->nToPage);
	for (int i = print_dlg->nFromPage-1; i < print_dlg->nToPage; i++) {
		vlog.verbose("Printing page %d", i);
		PopplerPage* page = poppler_document_get_page (document, i);
		if (page == NULL) {
			vlog.info("Poppler: Could not open page %d (page not found)\n", i);
			return 1;
		}
		scale_page(page, print_dlg->hDC);

		int start_page_result = StartPage(print_dlg->hDC);
		if (start_page_result <=0 ) {
			vlog.error("I could not StartPage with HDC: '%d'\n", start_page_result);
			return -1;
		}

		cairo_save (cr);
		poppler_page_render_for_printing (page, cr);
		cairo_restore (cr);
		cairo_surface_show_page (surface);

		g_object_unref (page);
		int end_page_result = EndPage(print_dlg->hDC);
		if (end_page_result <=0) {
			vlog.error("Failed to end page: %d\n", end_page_result);
			return -1;
		}
	}
	cairo_status_t status = cairo_status(cr);
	if (status)
		vlog.info("%s\n", cairo_status_to_string (status));
	cairo_destroy (cr);
	cairo_surface_finish (surface);
	status = cairo_surface_status(surface);
	if (status)
		vlog.info("%s\n", cairo_status_to_string (status));
	cairo_surface_destroy (surface);
	return 0;
}

int windows_print_pdf(char* path) {
	gobject_init (); // I could not register objects with an unitialized gobject if not (see glib patch)
	GError* error = NULL;

	char* uri = get_uri_from_path(path, error);
	if (!uri) {
		vlog.error("I couldn't convert path to uri: %s\n", error->message);
		return -1;
	}

	PopplerDocument* document = poppler_document_new_from_file (uri, NULL, &error);
	if (!document) {
		vlog.error("Poppler couldn't open document '%s': %s\n", uri, error->message);
		return -1;
	}
	int num_pages = poppler_document_get_n_pages (document);

	PRINTDLG* print_dlg = get_print_dlg(num_pages);
	if (!print_dlg) {
		g_object_unref (document);
		return -1;
	}

	DOCINFO doc_info = {0};
	doc_info.cbSize = sizeof(DOCINFO);
	const char* final_slash = strrchr(uri, '\\');
	doc_info.lpszDocName = final_slash ? final_slash + 1 : uri;

	if (StartDoc(print_dlg->hDC, &doc_info) <= 0) {
		vlog.error("I couldn't StartDoc with printer hDC\n");
		g_object_unref (document);
		free(print_dlg);
		return -1;
	}

	do_print(print_dlg, document);

	if (EndDoc(print_dlg->hDC)<=0) {
		vlog.error("I coudln't EndDocument with printer hDC\n");
		g_object_unref (document);
		free(print_dlg);
		return -1;
	}

	g_object_unref (document);
	free(print_dlg);

	return 0;
}
