/* Copyright (C) 2002-2005 RealVNC Ltd.  All Rights Reserved.
 * Copyright 2020 Pierre Ossman for Cendio AB
 * Copyright (C) 2014-2024 m-privacy GmbH
 * 
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <rdr/BufferedInStream.h>
#include <rdr/Exception.h>
#include <rdr/mutex.h>
#include <rfb/LogWriter.h>

using namespace rdr;

static rfb::LogWriter vlog("BufferedInStream");

static const size_t DEFAULT_BUF_SIZE = 128 * 1024;
static const size_t MAX_BUF_SIZE = 4 * 1024 * 1024;

BufferedInStream::BufferedInStream()
  : bufSize(DEFAULT_BUF_SIZE), offset(0), nextSerial(0)
{
  ptr = end = start = new U8[bufSize];
  gettimeofday(&lastSizeCheck, NULL);
  peakUsage = 0;
  classLog = &vlog;
  MUTEX_INIT(&serialMutex);
}

BufferedInStream::~BufferedInStream()
{
  delete [] start;
  MUTEX_DESTROY(&serialMutex);
  classLog->debug("BufferedInStream::peakUsage was %zu", peakUsage);
  resetClassLog();
}

size_t BufferedInStream::pos()
{
  return offset + ptr - start;
}

bool BufferedInStream::overrun(size_t needed)
{
  struct timeval now;

  if (needed > bufSize) {
    size_t newSize;
    U8* newBuffer;

    if (needed > MAX_BUF_SIZE)
      throw Exception("BufferedInStream overrun: requested size of "
                      "%lu bytes exceeds maximum of %lu bytes",
                      (long unsigned)needed, (long unsigned)MAX_BUF_SIZE);

    newSize = DEFAULT_BUF_SIZE;
    while (newSize < needed)
      newSize *= 2;

    classLog->debug("BufferedInStream::overrun(): needed %zu, increase buffer size from %zu to %zu", needed, bufSize, newSize);

    newBuffer = new U8[newSize];
    memcpy(newBuffer, ptr, end - ptr);
    delete [] start;
    bufSize = newSize;

    offset += ptr - start;
    end = newBuffer + (end - ptr);
    if (restorePoint)
      restorePoint = (const U8*) newBuffer + (restorePoint - start);
    ptr = start = newBuffer;

    gettimeofday(&lastSizeCheck, NULL);
    peakUsage = needed;
  }

  if (needed > peakUsage)
    peakUsage = needed;

  // Time to shrink an excessive buffer?
  gettimeofday(&now, NULL);
  if ((avail() == 0) && (bufSize > DEFAULT_BUF_SIZE) &&
      ((now.tv_sec < lastSizeCheck.tv_sec) ||
       (now.tv_sec > (lastSizeCheck.tv_sec + 5)))) {
    if (peakUsage < (bufSize / 2)) {
      size_t newSize;

      newSize = DEFAULT_BUF_SIZE;
      while (newSize < peakUsage)
        newSize *= 2;

      classLog->debug("BufferedInStream::overrun(): shrink buffer size from %zu to %zu", bufSize, newSize);

      // We know the buffer is empty, so just reset everything
      delete [] start;
      ptr = end = start = new U8[newSize];
      bufSize = newSize;
      if (restorePoint)
        restorePoint = start;
    }

    gettimeofday(&lastSizeCheck, NULL);
    peakUsage = needed;
  }

  // Do we need to shuffle things around?
  if ((bufSize - (ptr - start)) < needed) {
    memmove(start, ptr, end - ptr);

    offset += ptr - start;
    end -= ptr - start;
    if (restorePoint)
      restorePoint -= ptr - start;
    ptr = start;
  }

  while (avail() < needed) {
    if (!fillBuffer(start + bufSize - end))
      return false;
  }

  return true;
}

bool BufferedInStream::checkCheckHeader(U8 header[CHECKHEADERSIZE], U32 * serialNumber, U32 * crc, U16 * size) {
  bool result;

  memcpy(serialNumber, header, sizeof(U32));
  memcpy(crc, header + sizeof(U32), sizeof(U32));
  memcpy(size, header + sizeof(U32) + sizeof(U32), sizeof(U16));
//  classLog->debug("got serial number: %u", *serialNumber);
  MUTEX_LOCK(&serialMutex);
  if (*serialNumber == nextSerial) {
    result = true;
    nextSerial++;
  } else {
    classLog->error("serial number mismatch: expected %u, got %u with size %u", nextSerial, *serialNumber, *size);
    result = false;
  }
  MUTEX_UNLOCK(&serialMutex);
  return result;
}
