/*
 * Copyright (C) 2004 Red Hat Inc.
 * Copyright (C) 2005 Martin Koegler
 * Copyright (C) 2010 TigerVNC Team
 * Copyright (C) 2010-2021 m-privacy GmbH
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 */

#ifndef __C_SECURITY_MULTI_H__
#define __C_SECURITY_MULTI_H__

#ifdef WIN32
#include <winsock2.h>
#else
#include <sys/socket.h>
#include <sys/un.h>
#include <arpa/inet.h>
#endif

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#if defined(__APPLE__)
#include </usr/SDK/MacOSX10.15.sdk/usr/include/c++/v1/math.h>
#include </usr/SDK/MacOSX10.15.sdk/usr/include/c++/v1/cmath>
#include </usr/SDK/MacOSX10.15.sdk/usr/include/c++/v1/queue>
#include </usr/SDK/MacOSX10.15.sdk/usr/include/c++/v1/unordered_map>
#else
#include <math.h>
#include <cmath>
#include <queue>
#include <unordered_map>
#endif

#include <rfb/CSecurity.h>
#include <rfb/SSecurityVeNCrypt.h>
#include <rfb/Security.h>
#include <rfb/Timer.h>
#include <rdr/InStream.h>
#include <rdr/OutStream.h>
#include <rdr/MultiInStream.h>
#include <rdr/MultiOutStream.h>
#include <rdr/types.h>
#include <rdr/mutex.h>

#define MAXPABUFFERSIZE (10 * 1024 * 1024)

namespace rfb {
	typedef struct {
		rdr::U8* data;
		size_t len;
		char* filename;
		rdr::U32 packetNumber;
		rdr::U8 finalPacket;
		rdr::U8 fileTransferProtocol;
	} fileQueueData;

	class CSecurityMulti;

	typedef std::unordered_map<rdr::U32, int> handleToSocketMap_t;
	typedef std::unordered_map<rdr::U32, time_t> handleTimeMap_t;
	typedef std::unordered_map<rdr::U32, rdr::U8 *> bufferMap_t;
	typedef std::unordered_map<rdr::U32, int> bufferLenMap_t;

	class CSecurityMulti : public CSecurity, public Timer::Callback {
	public:
		CSecurityMulti(CConnection* cc);
		virtual ~CSecurityMulti();
		virtual bool processMsg();
		virtual unsigned int getType() const {
			return secTypeMulti;
		}
		virtual const char* description() const {
			return "Multiplexed vnc data";
		}
		void setAutotransferSupport(bool value);
		static THREAD_FUNC fileWriterThread(void* param);
		static THREAD_FUNC paReadThread(void* param);
		static THREAD_FUNC udpSocketThread(void* param);
		static THREAD_FUNC ffmpegThread(void* param);

#if !defined(WIN32) && !defined(WIN64) && !defined(__APPLE__)
		static THREAD_FUNC magicurlListenerThread(void* param);
		static THREAD_FUNC magicurlReadThread(void* param);
		static THREAD_ID magicurlListenerThreadId;
#endif

		static void stopSound();
		static bool runThreads;
		static THREAD_ID fileWriterThreadId;
		static THREAD_ID udpSocketThreadId;
		static THREAD_ID ffmpegThreadId;
		static MUTEX_TYPE fileQueueMutex;
		static std::queue<fileQueueData> fileQueue;
		static handleToSocketMap_t handleToSocketMap;
		static handleTimeMap_t handleTimeMap;
		static bufferMap_t paBufferMap;
		static bufferLenMap_t paBufferLenMap;
		static rdr::U8 congestionLevel;

	protected:
		bool handleTimeout(Timer * t);
		Timer keepAliveTimer;

	private:
		static bool incomingPaCallback(const rdr::U8* buf, int bufLen);
		static bool incomingSignalCallback(const rdr::U8* buf, int bufLen);
		static bool incomingFileTransferPacketCallback(const rdr::U8* buf, int bufLen);
		static int connectPa(rdr::U32 handle);
		static const char* getTgproTmpPath();
		static void addFilePacketToDataQueue(rdr::U8* newData, size_t len, char* fileName, rdr::U32 packetNumber, rdr::U8 finalPacket, const rdr::U8 fileTransferProtocol);

		static TGVNC_CONDITION_TYPE packetsWaitingCondition;
		static MUTEX_TYPE packetsWaitingConditionLock;

		static rdr::MultiInStream* multiIS;
		static rdr::MultiOutStream* multiOS;
		rdr::U8 runtimeAutotransferSupport;
#if !defined(WIN32) && !defined(WIN64) && !defined(__APPLE__)
		static int magicurlInSocket;
		struct sockaddr_un magicurlInSockaddr;
#endif
		Timer checkIdleTimer;
	};
}

#endif
