/* Copyright 2023 m-privacy GmbH.
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 */

#ifndef WIN32

#include <iostream>
#include <nlohmann/json.hpp>
#include <pwd.h>
#include <stdlib.h>
#include <stdio.h>
#include <iostream>
#include <fstream>
#include <curl/curl.h>
#include <libgen.h>
#include <string>
#include <string.h>
#include <sys/stat.h>
#include <archive.h>
#include <archive_entry.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>

#if !defined(WIN32) && !defined(WIN64)
#include <syslog.h>
#endif

#include <rfb/LogWriter.h>

#include <tgpro_environment.h>
#include <mp_utils.h>

#include "SAutotransferVaithex.h"

using namespace std;
using json = nlohmann::json;

static rfb::LogWriter vlog("SAutotransferVaithex");

rfb::StringParameter vaithexIp("vaithexIp","IP address of the VAITHEX server to clean files and (auto)-download them to the client","");

rfb::StringParameter vaithexPort("vaithexPort","Port of the VAITHEX server to clean files and (auto)-download them to the client","443");

// const char *JOBS_POST_BODY = "{\"normalize\": true,\"scanners_enabled\": [\"avira\",\"ikarus_scanserver\",\"sophos_savdi\"],\"timeout_sec\":0}";

rfb::IntParameter vaithexTimeout("vaithexTimeout", "VAITHEX timeout in seconds (min 60, max 3600)", 300, 60, 3600);

rfb::StringParameter vaithexScanners
("vaithexScanners",
 "List of scanners the VAITHEX server should use to process files",
 "avira, ikarus_scanserver,sophos_savdi,symantec");

char* vaithexScannersAsJsonStr = NULL;

char* listFirstFilePathInDir(const char* dirName) {
	DIR* dir;
	struct dirent* entry;

	// Open the directory
	dir = opendir(dirName);

	if (dir == NULL) {
		vlog.error("%s: failed to open %s", __func__, dirName);
		return NULL;
	}

	// Initialize a variable to store the first filename
	char* firstFilePath = NULL;
	entry = readdir(dir);

	if (entry != NULL) {
		if (entry->d_type == DT_REG) { // regular file?
			firstFilePath = (char*)malloc(strlen(dirName) + 1 + strlen(entry->d_name) + 1);
			if (firstFilePath != NULL) {
				sprintf(firstFilePath, "%s/%s", dirName, entry->d_name);
			}
		}
	}

	closedir(dir);

	return firstFilePath;
}

char* getVaithexScannersAsJsonStr() {
	if (!vaithexScannersAsJsonStr) {
		const char* scanners = vaithexScanners.getData();
		vlog.debug("Value of scanners(vaithexScanners.getData()): %s", scanners);
		if (!strcmp(vaithexScanners.getData(), "avira, ikarus_scanserver,sophos_savdi,symantec")) {
			// No scanners selected, return an empty JSON array
			vaithexScannersAsJsonStr = strdup("[]");
		} else {
			char* input = strdup(scanners);
			vlog.debug("Value of input after strdup(vaithexScanners.getData()): %s", input);

			unsigned int numElements = 1;
			const char* p = input;
			while (*p) {
				if (*p == ',') {
					numElements++;
				}
				p++;
			}
			size_t totalSize = strlen(input) + numElements * 2 + 2 + 1; // 2 for quotes per element, 2 for [] and 1 for null-termination (the commas are already accounted for in input)

			char* jsonStr = (char*) malloc(totalSize);
			if (jsonStr != NULL) {
				strcpy(jsonStr, "[");
				char* token = strtok((char*) input, ",");
				while (token != NULL) {
					strcat(jsonStr, "\"");
					strcat(jsonStr, token);
					strcat(jsonStr, "\",");
					token = strtok(NULL, ",");
				}
				// Remove the trailing comma, if it exists
				size_t len = strlen(jsonStr);
				if (len > 1 && jsonStr[len - 1] == ',') {
					jsonStr[len - 1] = ']';
				} else {
					jsonStr[len] = ']';
					jsonStr[len + 1] = '\0';
				}
			}
			free(input);

			if (jsonStr != NULL) {
				jsonStr[totalSize - 1] = '\0'; // Really make sure it's null-terminated
				vaithexScannersAsJsonStr = jsonStr;
			}
		}
	}
	return vaithexScannersAsJsonStr;
}

const char *SAutotransferVaithex::getDirToWatch()
{
	char *autotransferDir = (char *)malloc(4096);
	snprintf(autotransferDir, 4095, "/home/user/.transfer/%s/vaithex", getpwuid(geteuid())->pw_name);
	autotransferDir[4095] = '\0';
	return (const char *)autotransferDir;
}

// Declare job_id at a higher scope, accessible to both createJob and uploadPayload functions
long job_id = 0;
std::string token;
char uploadUrl[600];
char jobsPostBody[1024];
char metadata_json[4096];

// Function declaration for removeNonPrintableCharacters
void removeNonPrintableCharacters(char* str, size_t size);

void checkCACertificate(const char* caCertPath) {
	if (access(caCertPath, F_OK) != -1) {
		// File exists
		vlog.debug("%s: CA certificate file found at %s", __func__, caCertPath);
	} else {
		// File does not exist, log error and exit
		vlog.error("%s: CA certificate file not found at %s", __func__, caCertPath);
		exit(EXIT_FAILURE); // Exit with an error code
	}
}

bool createJob(char* jobIdOutput, char* tokenOutput, const char* caCertPath) {
	// Define the job creation JSON
	const char* scanners = getVaithexScannersAsJsonStr();
	if (strlen(scanners) == 0) {
		// No scanners selected, use empty array for scanners_enabled
		snprintf(jobsPostBody, sizeof(jobsPostBody), "{\"normalize\": true,\"scanners_enabled\": [],\"server_timeout_sec\":0}");
	} else {
		// Scanners selected, include scanners_enabled in the JSON
		snprintf(jobsPostBody, sizeof(jobsPostBody), "{\"normalize\": true,\"scanners_enabled\": %s,\"server_timeout_sec\":0}", scanners);
	}
	// Check if the CA certificate file exists
	checkCACertificate(caCertPath);

	// Build the wget command with the CA certificate
	char getJobCmd[1500];
	snprintf(getJobCmd, sizeof(getJobCmd),"wget --ca-certificate=%s --quiet --progress=dot --method=POST --output-document=- --body-data='%s' 'https://%s:%s/v1/jobs'", caCertPath, jobsPostBody, vaithexIp.getData(), vaithexPort.getData());

	vlog.debug("%s: running command: %s", __func__, getJobCmd);
	FILE* responseStream = popen(getJobCmd, "r");
	if (!responseStream) {
		vlog.error("%s: Failed to execute command", __func__);
		return false;
	}
	// Read the response into the buffer
	char response_buffer[4096]; // Adjust the buffer size as needed
	size_t responseSize = fread(response_buffer, 1, sizeof(response_buffer) - 1, responseStream);
	if (responseSize == (size_t)-1) {
		vlog.error("%s: Failed to read response from the stream", __func__);
		pclose(responseStream);
		return false;
	}
	pclose(responseStream);
	if (responseSize == 0) {
		vlog.error("%s: Empty response from the server", __func__);
		return false;
	}
	// Null-terminate the response buffer
	response_buffer[responseSize] = '\0';

	// After reading the response into response_buffer
	vlog.debug("Response from server: %s", response_buffer);

	// Parse the JSON response using nlohmann::json
	try {
		json response_json = json::parse(response_buffer);

		// Check if the JSON contains "id" and "token" fields
		if (response_json.contains("id") && response_json.contains("token")) {
			job_id = response_json["id"];
			token = response_json["token"];
			// Store the job ID and token (in output parameters)
			snprintf(jobIdOutput, 4096, "%ld", job_id);
			snprintf(tokenOutput, 4096, "%s", token.c_str());
			// Log job ID and token
			vlog.debug("%s: Job ID: %ld, Token: %s", __func__, job_id, token.c_str());
		} else {
			vlog.error("%s: JSON response is missing 'id' or 'token' fields", __func__);
			return false;
		}
	} catch (const json::parse_error& e) {
		vlog.error("%s: JSON parsing error: %s", __func__, e.what());
		return false;
	}

	return true;
}

void removeNonPrintableCharacters(char* str, size_t size) {
	int len = strlen(str);
	int i, j = 0;
	for (i = 0; i < len; i++) {
		if (isprint(str[i])) {
			str[j++] = str[i];
		}
	}
	// Null-terminates the modified string
	str[j] = '\0';
}

void uploadPayload(char* jobIdOutput, char* tokenOutput, const char* filename, const char* dir, const char* caCertPath) {
	// Print request for the upload payload
	vlog.debug("%s: Uploading payload for filename: %s", __func__, filename);

	// Upload URL with the job ID
	char uploadUrl[600];
	snprintf(uploadUrl, sizeof(uploadUrl), "https://%s:%s/v1/jobs/%ld/payload", vaithexIp.getData(), vaithexPort.getData(), job_id);

	// Create the tar file with the upload file
	char tar_cmd[700];
	snprintf(tar_cmd, sizeof(tar_cmd), "tar -cf '/tmp/%s.tar' -C '%s' '%s'", filename, dir, filename);
	system(tar_cmd);
	vlog.info("%s: Executing tar cmd: %s", __func__, tar_cmd);

	// Construct the upload command with the CA certificate
	char uploadCmd[1500];
	snprintf(uploadCmd, sizeof(uploadCmd), "wget --ca-certificate=%s --quiet --progress=dot --method=POST --output-document=- --body-file='/tmp/%s.tar' --header='X-Waas-Token: %s' '%s'", caCertPath, filename, tokenOutput, uploadUrl);

	// Print and execute the wget upload command
	vlog.debug("%s: Executing wget cmd for uploadPayload: %s", __func__, uploadCmd);
	int uploadResult = system(uploadCmd);
	if (uploadResult == 0) {
		vlog.debug("%s: Payload uploaded for filename: %s", __func__, filename);
	} else {
		vlog.debug("%s: Payload upload failed for filename: %s. Error code: %d", __func__, filename, uploadResult);
		return;
	}
	// Check the name of the uploaded tar file
	vlog.debug("Uploaded tar file name: %s.tar", filename);
	// Clean up the temporary tarball file
	char cleanupCmd[1000];
	snprintf(cleanupCmd, sizeof(cleanupCmd), "rm -f '/tmp/%s.tar'", filename);
	system(cleanupCmd);

	// Clean up allocated memory
	free(jobIdOutput);
	free(tokenOutput);
}

void fetchMetadata(const char* token, long job_id, char* metadata_json_buffer, const char* caCertPath) {
	// Build the URL to fetch metadata
	char metadata_url[600];
	snprintf(metadata_url, sizeof(metadata_url), "https://%s:%s/v1/jobs/%ld", vaithexIp.getData(), vaithexPort.getData(), job_id);
	// Build the command to fetch metadata using wget with the CA certificate
	char metadata_cmd[1000];
	snprintf(metadata_cmd, sizeof(metadata_cmd), "wget --ca-certificate=%s --quiet --method=GET --output-document=- --header='X-Waas-Token: %s' '%s'", caCertPath, token, metadata_url);
	// Log the metadata_cmd
	vlog.debug("%s: Executing the wget metadata cmd: %s", __func__, metadata_cmd);

	// Execute the command to fetch the metadata
	FILE* metadata_output = popen(metadata_cmd, "r");
	if (metadata_output == NULL) {
		// Log the error
		vlog.error("Failed to execute the metadata cmd: %s", strerror(errno));
		return;
	}
	// Create a buffer to store the metadata response
	char metadata_response[4096];
	if (fgets(metadata_response, sizeof(metadata_response), metadata_output) == NULL) {
		// Log the error
		vlog.error("Failed to read metadata response: %s", strerror(errno));
		pclose(metadata_output);
		return;
	}
	pclose(metadata_output);
	// Log the complete response
	vlog.debug("%s: Complete metadata response: %s", __func__, metadata_response);
	// Remove non-printable characters from the metadata_response
	removeNonPrintableCharacters(metadata_response, sizeof(metadata_response));
	// Parse the metadata response using nlohmann/json
	json parsed_metadata_json;
	try {
		parsed_metadata_json = json::parse(metadata_response);
	} catch (const json::parse_error &e) {
		vlog.error("JSON parsing error: %s", e.what());
		return;
	}
	// Assuming metadata_json_buffer is a character buffer passed to the function and you want to store the parsed JSON in that buffer
	snprintf(metadata_json_buffer, 4096, "%s", parsed_metadata_json.dump().c_str());
}

//Global var
const char* outputDir = "/tmp/vaithex-tmp";

// Function prototype for extractAll
static int extractAll(const char* filename, const char* outputDir, char* lastExtractedFile);

void processCleanOrNormalized(const char* filePath, const char* filename, const char* metadata_response, const char* caCertPath) {
	json metadata_json;
	try {
		metadata_json = json::parse(metadata_response);
	} catch (const json::parse_error &e) {
		vlog.error("JSON parsing error: %s", e.what());
		vlog.error("Metadata response that caused the error: %s", metadata_response);
		return;
	}

	if (metadata_json.contains("result_report") && metadata_json["result_report"].contains("result_worst")) {
		std::string result_worst = metadata_json["result_report"]["result_worst"];

		if (result_worst == "clean") {
			vlog.debug("Result report: clean (could also be normalized)");

			if (strstr(metadata_response, "\"normalized_state\":\"successful\"") == nullptr) {
				//Construct the destination path
				std::string autotransferNowFile = "/home/user/.autotransfer-now/" + std::string(getpwuid(geteuid())->pw_name) + "/" + filename;
				vlog.debug("%s: trying to mp_move from '%s' to '%s'", __func__, filePath, autotransferNowFile.c_str());
				//Move the file to the autotransferNowFile destination
				char* nonConstFilePath = strdup(filePath);
				mp_move(nonConstFilePath, const_cast<char*>(autotransferNowFile.c_str()), 1);
				free(nonConstFilePath);
				//Log the completion of the move operation
				vlog.debug("%s: moved '%s' to '%s'", __func__, filePath, autotransferNowFile.c_str());

			} else {
				// If the file is normalized
				vlog.debug("File was normalized (actually only check if \"normalized_state\":\"successful\" is in the response at all)");
				vlog.debug("%s: removing %s because it was normalized. Will auto-transfer the normalized file to the client.", __func__, filePath);
				// Make a non-const copy of filePath
				char* nonConstFilePath = strdup(filePath);
				// delete file in 'vaithex' as we will be downloading a normalized one and auto-transferring that one
				delete_file(nonConstFilePath);
				free(nonConstFilePath);

				// Archive the uploaded file directly into the result.tar
				char result_payload_url[600];
				snprintf(result_payload_url, sizeof(result_payload_url), "https://%s:%s/v1/jobs/%ld/payload", vaithexIp.getData(), vaithexPort.getData(), job_id);
				// Log the result_payload_url
				vlog.error("Result Payload URL: %s", result_payload_url);

				char tarredResult[1024];
				snprintf(tarredResult, sizeof(tarredResult), "/tmp/%s-result.tar", filename);
				// Log the value of tarredResult
				vlog.debug("Value of tarredResult: %s", tarredResult);

				char result[4096];
				snprintf(result, sizeof(result), "/tmp/%s", filename);

				// Execute the wget command to fetch the result payload
				char result_payload_cmd[4096];
				snprintf(result_payload_cmd, sizeof(result_payload_cmd), "wget --ca-certificate=%s --quiet --progress=dot --method=GET --output-document='%s' --header='X-Waas-Token: %s' '%s'",caCertPath, tarredResult, token.c_str(), result_payload_url);
				vlog.debug("Running: %s", result_payload_cmd);

				if (system(result_payload_cmd) == 0) {
					vlog.debug("Result (was cleaned if necessary) archived in %s", tarredResult);
					// Log the value of tarredResult before calling extractAll
					vlog.debug("Tarred Result Path: %s", tarredResult);

					// Untar the downloaded payload
					if (mkdir("/tmp/vaithex-tmp", 0700) && errno != EEXIST) {
						vlog.error("Failed to create a tgpro temporary directory for autotransfer.");
						return;
					}

					char* extractedFile = (char*) malloc(4096);
					int untarResult = extractAll(tarredResult, "/tmp/vaithex-tmp", extractedFile);
					if (untarResult != 0) {
						vlog.error("Failed to untar the downloaded payload.");
						free(extractedFile);
						return;
					}

					if (!extractedFile || !*extractedFile) {
						vlog.error("%s: failed to get name of normalized file in /tmp/vaithex", __func__);
						free(extractedFile);
						return;
					}
					vlog.debug("%s: extracted normalized file to %s", __func__, extractedFile);

					char* normalizedFilename = basename(strdup(extractedFile));
					vlog.debug("%s: normalized file's filename: %s", __func__, normalizedFilename);
					if (!normalizedFilename) {
						vlog.error("%s: failed to get filename for file '%s'. Aborting...", __func__, extractedFile);
						free(extractedFile);
						return;
					}

					// Append "___vaithexed" to the normalized filename
					// Copy normalized filename
					std::string modifiedFilename = std::string(normalizedFilename);
					// Find the position of the last occurrence of '.'
					size_t dotPos = modifiedFilename.find_last_of('.');
					if (dotPos != std::string::npos) {
						// Insert "_vaithexed" just before the dot position
						modifiedFilename.insert(dotPos, "___vaithexed");
					} else {
						// If no dot is found (no extension), simply append "_vaithexed" to the filename
						modifiedFilename += "___vaithexed";
					}
					vlog.debug("Filename %s has been modified to: %s", normalizedFilename, modifiedFilename.c_str());

 					//// Append "_vaithexed" to the normalized filename
 					//std::string modifiedFilename = std::string(normalizedFilename) + "___vaithexed";
					////Log the file name and the modified file name
					//vlog.debug("Filename: %s has been modified to: %s", normalizedFilename, modifiedFilename.c_str());

					std::string autotransferNowFile = "/home/user/.autotransfer-now/" + std::string(getpwuid(geteuid())->pw_name) + "/" + modifiedFilename;
					vlog.debug("%s: trying to mp_move '%s' to '%s'", __func__, extractedFile, autotransferNowFile.c_str());
					// Add logging to indicate that autotransfer is about to occur
					vlog.debug("Autotransfer will move file: %s to path: %s", extractedFile, autotransferNowFile.c_str());
					char* nonConstExtractedFile = strdup(extractedFile);
					mp_move(nonConstExtractedFile, const_cast<char*>(autotransferNowFile.c_str()), 1);
					free(nonConstExtractedFile);
					// Log autotransfer completion with filename and path
					vlog.debug("Autotransfer completed successfully for file: %s at path: %s", modifiedFilename.c_str(), autotransferNowFile.c_str());
					// Free memory
					free(extractedFile);
				} else {
					vlog.error("Failed to untar downloaded result!");
					return;
				}
			}
		}
	}
}

void processInfected(const char* filePath, const char* filename, const char* metadata_response) {
	json metadata_json;
	try {
		metadata_json = json::parse(metadata_response);
	} catch (const json::parse_error &e) {
		vlog.error("JSON parsing error: %s", e.what());
		vlog.error("Metadata response that caused the error: %s", metadata_response);
		return;
	}

	if (metadata_json.contains("result_report") && metadata_json["result_report"].contains("result_worst")) {
		std::string result_worst = metadata_json["result_report"]["result_worst"];

		if (result_worst == "infected") {
			vlog.debug("Result report: Blocked!");

			// Retrieve the hostname
			char hostname[HOST_NAME_MAX];
			gethostname(hostname, HOST_NAME_MAX);

			// Similar as the logic in the OPSWAT Code
			char logpath[128];

			delete_file((char*) filePath);
			vlog.info("%s: VAITHEX rejects %s, scan result: %s", __func__, filename, metadata_response ? metadata_response : "Unknown");

			#if !defined(WIN32) && !defined(WIN64)
			openlog("Xtightgatevnc", LOG_PID, LOG_AUTH);
			syslog(LOG_DEBUG, "%s: VAITHEX rejects %s, scan result: %s", __func__, filename, metadata_response);
			#endif

			snprintf(logpath, 127, "/home/user/.vaithexlog/%s", getenv("USER"));
			logpath[127] = 0;
			if (!access(logpath, F_OK)) {
				FILE *handle;

				handle = fopen(logpath, "a");
				if (handle) {
					char * line = (char *) malloc(4096);

					if (line) {
						time_t nowsec = time(NULL);
						struct tm * now = localtime(&nowsec);
						snprintf(line, 4095, "%s %02u.%02u.%u %02u:%02u:%02u %s %s\n", hostname, now->tm_mday, now->tm_mon+1, now->tm_year+1900, now->tm_hour, now->tm_min, now->tm_sec, filename, filePath);

						vlog.debug("%s: logging '%s' to %s", __func__, line, logpath);
						fwrite(line, 1, strlen(line), handle);
						if (ferror(handle)) {
							fclose(handle);
							vlog.error("%s: failed to write file '%s'", __func__, logpath);
							#if !defined(WIN32) && !defined(WIN64)
							openlog("Xtightgatevnc", LOG_PID, LOG_AUTH);
							syslog(LOG_DEBUG, "%s: failed to write file '%s'", __func__, logpath);
							#endif
						} else {
							fclose(handle);
							system("/usr/bin/pkill inotifywait");
						}
						free(line);
					}
				} else {
					vlog.error("%s: failed to open logfile %s, not logging, error: %s (notifications with mp-watch won't work!)", __func__, logpath, strerror(errno));
				}
			} else {
				vlog.error("%s: logfile %s not found, not logging (notifications with mp-watch won't work!)", __func__, logpath);
			}

			return;
		}
	} else {
		vlog.error("%s: Error! wget's response size is 0 or smaller", __func__);
		return;
	}
}

static int extractAll(const char* filename, const char* outputDir, char* lastExtractedFileOutput) {
	vlog.debug("%s: extracting all files from '%s' to '%s'", __func__, filename, outputDir);

	// Cmd to extract the tarred file
	char untarCmd[4096];
	snprintf(untarCmd, sizeof(untarCmd), "tar -xf %s -C %s", filename, outputDir);

	// Execute the untar cmd
	int untarResult = system(untarCmd);

	if (untarResult != 0) {
		vlog.error("%s: Failed to untar the downloaded payload.", __func__);
		return 1;
	}

	// Find the name of the last extracted file
	DIR *dir;
	struct dirent *ent;
	char *lastFile = NULL;
	if ((dir = opendir(outputDir)) != NULL) {
		while ((ent = readdir(dir)) != NULL) {
			if (strcmp(ent->d_name, ".") != 0 && strcmp(ent->d_name, "..") != 0) {
				lastFile = strdup(ent->d_name);
			}
		}
		closedir(dir);
	} else {
		vlog.error("%s: Unable to open directory '%s'.", __func__, outputDir);
		return 1;
	}

	if (lastFile != NULL) {
		snprintf(lastExtractedFileOutput, PATH_MAX, "%s/%s", outputDir, lastFile);
		free(lastFile);
	} else {
		vlog.error("%s: No files extracted from the tarball.", __func__);
		return 1;
	}

	return 0;
}

void SAutotransferVaithex::processFile(char *filePath)
{
	char* filename = basename(strdup(filePath)); // Because basename(filePath) is changing filePath
	char* dir = dirname(strdup(filePath)); // Because basename(filePath) is changing filePath
	vlog.debug("%s: VAITHEXing file '%s' (filename '%s')", __func__, filePath, filename);

	// Allocate memory for jobIdOutput and tokenOutput
	char* jobIdOutput = (char*) malloc(4096);
	char* tokenOutput = (char*) malloc(4096);

	// Define the CA certificate path
	const char* caCertPath = "/home/user/.customvaithexca/ca-certificates";

	// Pass the CA certificate path to createJob
	bool jobCreationSucceeded = createJob(jobIdOutput, tokenOutput, caCertPath);
	if (!jobCreationSucceeded) {
		vlog.error("%s: failed to create a job. Aborting VAITHEXing of file '%s'", __func__, filename);
		free(jobIdOutput); // Free memory to prevent leaks
		free(tokenOutput); // Free memory to prevent leaks
		return;
	}
	vlog.debug("Yay! Got me some id (%s) and some token (%s)", jobIdOutput, tokenOutput);

	// Call uploadPayload function with the CA certificate path
	uploadPayload(jobIdOutput, tokenOutput, filename, dir, caCertPath);

	// Create a buffer to store the response
	char response_buffer[4096]; // Adjust the buffer size as needed;

	// Calculate the size of the response string
	size_t size = strlen(response_buffer);

	// Call removeNonPrintableCharacters with response_buffer as the parameter and its size
	removeNonPrintableCharacters(response_buffer, size);

	// Call the fetchMetadata function
	char metadataOutput[4096];
	fetchMetadata(tokenOutput, job_id, metadataOutput, caCertPath);

	// Call the function to process clean or normalized result
	processCleanOrNormalized(filePath, filename, metadataOutput, caCertPath);

	char*  autotransferNowFile = strdup(filePath);
	autotransferNowFile[4095] = '\0';

	// Call the function to process infected result
	processInfected(filePath, filename, metadataOutput);

	char tarredResult[1024];
	snprintf(tarredResult, sizeof(tarredResult), "/tmp/%s-result.tar", filename);

	// Free allocated memory
	free(jobIdOutput);
	free(tokenOutput);

	return;
}

#endif /* #ifndef WIN32 */
