/**
 * Before including this file you have to define the mplog preprocessor macro or it
 * will just log using printf's.
 *
 * I know, I really shouldn't define stuff in a header file. It's just this is the best idea
 * we had to use functions in the VNC and SSH servers *and* have the great advantage of
 * logging stuff.
 *
 */

#if !defined(WIN32) && !defined(__APPLE__)

#include "tgpro_environment.h"
#include "mp_utils.h"

#include <math.h>
#define _GNU_SOURCE
#include <string.h>
#include <b64/cdecode.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <syslog.h>
#include <errno.h>
#include <ctype.h>

#ifndef mplog_info
#define mplog_info(...) printf(__VA_ARGS__); printf("\n");
#endif

#ifndef mplog_debug
#define mplog_debug(...) printf(__VA_ARGS__); printf("\n");
#endif

#ifndef mplog_verbose
#define mplog_verbose(...) mplog_debug(__VA_ARGS__);
#endif

#ifndef mplog_error
#define mplog_error(...) printf(__VA_ARGS__); printf("\n");
#endif

#ifndef mplog_syslog_error
#define mplog_syslog_error(...) printf(__VA_ARGS__); printf("\n");
#endif

#ifndef mplog_syslog_info
#define mplog_syslog_info(...) printf(__VA_ARGS__); printf("\n");
#endif

#ifndef mplog_syslog_debug
#define mplog_syslog_debug(...) printf(__VA_ARGS__); printf("\n");
#endif

#define LDAP_SERVER_BIND_STRING_MAX_LEN 77 /* 8 + 64 + 5 */ /* Something like ldaps://win2012.m-privacy.ad:3269 ; 8 -> ldaps:// ; 64 -> activedirectoryfqdn ; 5 because of :port */
#define LDAP_SERVER_MAX_LIST_SIZE 160 // In theory, we should have 20*4KDC*2protocols(ldap and ldaps)=160 max (TODO Check this!!).

#define BUFFER_SIZE 4096
#define TGGROUPS_BUF_SIZE (16 * 1024)
#define MAX_ALLOCATABLE_SIZE (1024 * 1024) // 1MB should be enough for responses and group lists. More than enough, if not, we'll have to check other stuff

/* Our MP errors. To use both with VNC and SSH */
#define MP_LDAP_OK 0
#define MP_LDAP_ERROR 500
#define MP_LDAP_MALLOC_ERROR 600
#define MP_LDAP_BASE_AUTO_ERROR 700
#define MP_LDAP_SERVER_AUTO_ERROR 800
#define MP_LDAP_QUERY_EXEC_ERROR 900
#define MP_LDAP_EMPTY_RESPONSE_ERROR 950
#define MP_LDAP_NO_GROUPS_ERROR 960
#define MP_LDAP_DN_QUERY_EXEC_ERROR 1000
#define MP_LDAP_DN_EMPTY_ERROR 1100

#define MP_LDAP_DN_ERROR 1200
#define MP_LDAP_MEMBEROF_QUERY_EXEC_ERROR 1300
#define MP_LDAP_MEMBEROF_EMPTY_ERROR 1400
#define MP_LDAP_MEMBEROF_ERROR 1500
#define MP_LDAP_INVALID_ARG 1600
/* End of our MP errors */

#define GCS_AND_DCS "/etc/cu/discovered-active-directory-gcs-and-dcs" /* Global Catalogs and Domain Controllers file periodically populated with /usr/local/bin/discover-active-directory-gcs-and-dcs */

// This is just a horrible work-around for a problem we are having with GCC 10 (as of 4.6.2021).
// strcasestr might now work normaly and could be worth a try. Removing the following #define should
// do it.
#define strcasestr strcasestrugly
static char* strcasestrugly(const char* haystack, const char* needle) {
	if (!haystack || !needle) {
		return NULL;
	}

	char* haystack_lowercase = strdup(haystack);

	char* p = haystack_lowercase;
	while(*p) {
		*p = tolower(*p);
		p++;
	}

	char* needle_lowercase = strdup(needle);
	p = needle_lowercase;
	while(*p) {
		*p = tolower(*p);
		p++;
	}

	char* uglystrcasestr = NULL;
	char* strstr_lowercase = strstr(haystack_lowercase, needle_lowercase);
	if (strstr_lowercase) {
		uglystrcasestr = (char*) haystack + (strstr_lowercase - haystack_lowercase);
	}

	free(haystack_lowercase);
	free(needle_lowercase);

	return uglystrcasestr;
}

static char** error_list = NULL;

static void add_error_to_list(const char* error_message, ...)
{
	if (!error_list) {
		mplog_error("%s: error_list is NULL but shouldn't be. Failed to add error message to it.", __func__);
		return;
	}

	int first_empty_index = -1;

	for (unsigned int i=0; i < LDAP_SERVER_MAX_LIST_SIZE; i++) {
		if (!error_list[i] || !*error_list[i]) {
			first_empty_index = i;
			break;
		}
	}

	if (first_empty_index == -1) {
		mplog_error("%s: error_list is full. Failed to add error message to it.", __func__);
		return;
	}

	error_list[first_empty_index] = (char*) malloc(BUFFER_SIZE);

	va_list args;
	va_start(args, error_message);
	vsnprintf(error_list[first_empty_index], BUFFER_SIZE - 1, error_message, args);
	va_end(args);

	error_list[first_empty_index][BUFFER_SIZE - 1] = 0;
}

static void free_error_list() {
	if (!error_list) {
		return;
	}

	for (unsigned int i=0; i<LDAP_SERVER_MAX_LIST_SIZE; i++) {
		if (error_list[i]) {
			free(error_list[i]);
		}
	}

	free(error_list);
	error_list = NULL;
}

static void init_error_list() {
	if (error_list) {
		free_error_list();
	}

	error_list = (char** ) malloc(LDAP_SERVER_MAX_LIST_SIZE * sizeof(char *));
	if (!error_list) {
		mplog_error("init_error_list: Could not allocate memory for the error list");
		syslog(LOG_ERR, "init_error_list: Could not allocate memory for the error list");
		return;
	}

	for (unsigned int i=0; i<LDAP_SERVER_MAX_LIST_SIZE; i++) {
		error_list[i] = NULL;
	}
}

char* get_value_from_file(const char* key, const char* file);

// LDAP filters
static char samaccount_filter[BUFFER_SIZE] = ""; // To store the (sAMAccountName=%s) which we use all the time (samaccount is the (old) windows username)
static char* get_samaccount_filter(const char* samaccount) {
	char* filter = samaccount_filter;
	if (!*filter) {
		snprintf(filter, BUFFER_SIZE, "(sAMAccountName=%s)", samaccount);
		filter[BUFFER_SIZE -1] = 0;
	}

	return filter;
}

static char object_sid_filter[BUFFER_SIZE] = "";
static char* get_object_sid_filter(const char* object_sid_str) {
	char* filter = object_sid_filter;
	if (!*filter) {
		snprintf(filter, BUFFER_SIZE - 1, "(objectSid=%s)", object_sid_str);
		filter[BUFFER_SIZE -1] = 0;
	}

	return filter;
}

/*
 * The implementation is slightly different for VNC (C++) and SSH (C).
 * So they both must define (implement) it. As it is used by functions
 * here, we need to declare it.
 */
static char* base64_decode(char* userdn_pos) {
	char* userdn_decoded = (char *) calloc(BUFFER_SIZE, 1); // There used to me a malloc here but there were problems with ending the returned string. Setting the memory to zero with the calloc seems to make it work properly
	if (!userdn_decoded) {
		return 0;
	}
#ifdef __cplusplus
	base64::decoder decoder;
	decoder.decode(userdn_pos, strlen(userdn_pos), userdn_decoded);
#else
	base64_decodestate state;
	base64_init_decodestate(&state);
	base64_decode_block(userdn_pos, strlen(userdn_pos), userdn_decoded, &state);
#endif
	return userdn_decoded;
}

/*
 * The caller is responsible for passing a poiner to a char[] with enough space for the decoded string
 */
static int base64_decode_no_alloc(const char* encoded, char **decoded_ptr) {
	char* decoded = *decoded_ptr;
	memset(decoded, 0, BUFFER_SIZE);
#ifdef __cplusplus
	base64::decoder decoder;
	decoder.decode(encoded, strlen(encoded), decoded);
#else
	base64_decodestate state;
	base64_init_decodestate(&state);
	base64_decode_block(encoded, strlen(encoded), decoded, &state);
#endif
	return 1;
}

/* This modifies the original str and the returned value *MAY NOT* be freed.
 */
char* trim_str(char* str) {
	if (!str) {
		return NULL;
	}

	char* trimmed = str;
	while (*trimmed && isspace(*trimmed)) {
		trimmed++;
	}
	char* end = trimmed + strlen(trimmed) - 1;
	while(end > trimmed && isspace(*end)) {
		end--;
	}
	end[1] = 0;

	return trimmed;
}

/**
 * Escape characters for active directory (specifically for search filters):
 * See: https://docs.microsoft.com/de-de/windows/desktop/ADSI/search-filter-syntax#special-characters
 * We cannot really escape the NULL chars, as it will mean the end of the char* for us.
 */
static char* escape_distinguished_names_special_chars(char* userdn) {
// Example string: CN=Juan\, García (jg),CN=Users,DC=sso,DC=m-privacy,DC=jua
	char* tmp1 = str_replace(userdn, "\\", "\\5c");
	char* tmp2 = str_replace(tmp1, "*", "\\2a");
	char* tmp3 = str_replace(tmp2, "(", "\\28");
	char* tmp4 = str_replace(tmp3, ")", "\\29");
	char* escaped_userdn = str_replace(tmp4, "/", "\\2f");

	free(tmp1);
	free(tmp2);
	free(tmp3);
	free(tmp4);

	return escaped_userdn;
}

static void escape_distinguished_names_special_chars_no_alloc(char* dn, char (*dn_escaped)[BUFFER_SIZE]) {
// Example string: CN=Juan\, García (jg),CN=Users,DC=sso,DC=m-privacy,DC=jua

	char* tmp1 = str_replace(dn, "\\", "\\5c");
	char* tmp2 = str_replace(tmp1, "*", "\\2a");
	char* tmp3 = str_replace(tmp2, "(", "\\28");
	char* tmp4 = str_replace(tmp3, ")", "\\29");
	char* tmp5 = str_replace(tmp4, "/", "\\2f");

	char* result = *dn_escaped;
	strncpy(result, tmp5, BUFFER_SIZE - 1);
	result[BUFFER_SIZE - 1] = 0;

	free(tmp1);
	free(tmp2);
	free(tmp3);
	free(tmp4);
	free(tmp5);
}

static void portable_setenv (const char * var_name, const char * value)
{
//#ifdef HAVE_SETENV
#if 1
	setenv(var_name, value, 1);
#else
	int len = strlen(var_name) + strlen(value) + 2;
	char * str = (char *)malloc(len);
	sprintf(str, "%s=%s", var_name, value);
	putenv(str);
	// No free(str) here because putenv() needs allocated memory, which stays allocated (on glibc).
#endif
}

static void portable_unsetenv (const char * var_name)
{
//#ifdef HAVE_SETENV
#if 1
	unsetenv(var_name);
#else
	portable_setenv(var_name, "");
#endif
}

static char* get_domain(const char* realm, const char* realms_file)
{
	FILE * f;
	char line[BUFFER_SIZE];
	char * domain = NULL;
	int realm_len;

	f = fopen(realms_file, "r");
	if (!f)
		return NULL;
	if (realm)
		realm_len = strlen(realm);
	else
		realm_len = 0;
	while (!feof(f))
	{
		if (!fgets(line, sizeof(line), f))
			break;
		if (!realm || ( !strncmp(line, realm, realm_len) && (line[realm_len] == ' ')) )
		{
			char * p;
			char * q;

			p = line;
			q = NULL;
			while (*p)
			{
				if(*p == '\n')
				{
					*p = 0;
					break;
				}
				if(*p == ' ')
					q = p + 1;
				p++;
			}
			if (q && *q) {
				domain = strdup(q);
				mplog_debug("get_domain(): found domain %s for realm %s",
							domain ? domain : "NULL",
							realm ? realm : "NULL");
				break;
			}
		}
	}
	fclose(f);
	return domain;
}

static int check_ldap_config_values(const char* realm, const char* domain, const char* ldap_base, const char* ldap_server_1, const char* ldap_server_2, const char* ldap_auto) {
	if (!strcmp(ldap_base, "auto")) {
		mplog_debug("LDAP base auto");
		if (!realm || !realm[0]) {
			mplog_error("LDAP base auto, but no realm from displayname (ldap_principal)!");
			syslog(LOG_ERR, "LDAP base auto, but no realm from displayname (ldap_principal)!");
			return MP_LDAP_BASE_AUTO_ERROR;
		}
		if (!domain) {
			mplog_error("LDAP base auto, but no domain!");
			syslog(LOG_ERR, "LDAP base auto, but no domain!");
			return MP_LDAP_BASE_AUTO_ERROR;
		}
		mplog_debug("LDAP base auto settings okay (actually, plausible)");
	}

	if ((ldap_server_1 && !strcmp(ldap_server_1, "auto")) || (ldap_server_2 && !strcmp(ldap_server_2, "auto"))) {
		mplog_debug("LDAP server auto");
		if (!realm || !realm[0]) {
			mplog_error("LDAP server auto, but no realm from displayname!");
			syslog(LOG_ERR, "LDAP server auto, but no realm from displayname!");
			return MP_LDAP_SERVER_AUTO_ERROR;
		}
		if (!domain) {
			mplog_error("LDAP server auto, but no domain!");
			syslog(LOG_ERR, "LDAP server auto, but no domain!");
			return MP_LDAP_SERVER_AUTO_ERROR;
		}
		mplog_debug("LDAP server auto settings okay (actually, plausible)");
	}

	return MP_LDAP_OK;
}

static int generate_ldap_base(char** ldap_base, char* user_domain)
{
	/* The main domain includes more than the user domain. If you use the base of a subdomain
	   but ldapsearch for objects that actually belong to the main domain, you won't find them.
	   This would be the case if you have a user in an intern.example.org domain, but the tg groups
	   are only defined in example.org. If you used a base like dc=intern,dc=example,dc=org, you
	   wouldn't be able to find the groups. The other way around is possible: if a group existed only
	   in intern.example.org, you would find it querying the intern.example.org server using a
	   dc=example,dc=org base. So it's (apparently) better to use the highest known level. I hope.
	*/
	char* domain = get_value_from_file("CU_ACTIVEDIRECTORY_MAIN_DOMAIN", GCS_AND_DCS);
	if (!domain) {
		/* This will happen for manual for example. In this case, this is good, as we don't do
		   the optimized search and to the samaccount search instead of the tg* groups optimized search.
		   In this case, we need the 'precisest' subdomain.
		 */
		if (!user_domain || !*user_domain) {
			mplog_error("generate_ldap_base called with NULL or empty user_domain");
			return MP_LDAP_INVALID_ARG;
		}

		mplog_debug("Couldn't read main domain from '%s'. Using 'user_domain' to generate LDAP base", GCS_AND_DCS);
		domain = strdup(user_domain);
		if (!domain) {
			mplog_error("Couldn't strdup(user_domain) to generate LDAP base")
			return MP_LDAP_MALLOC_ERROR;
		}
	}
	if (strlen(domain) > 1023) {
		mplog_error("LDAP base auto, but domain name is too long!");
		syslog(LOG_ERR, "LDAP base auto, but domain is too long!");
		return MP_LDAP_MALLOC_ERROR;
	}

	*ldap_base = (char *) malloc(BUFFER_SIZE);
	if (!*ldap_base) {
		mplog_error("LDAP base auto, but cannot allocate memory!");
		return MP_LDAP_MALLOC_ERROR;
	}
	memset(*ldap_base, 0, BUFFER_SIZE);
	char* p = (char *) *ldap_base;
	char* p2 = domain;
	p += sprintf(p, "dc=");
	while (*p2) {
		if (*p2 == '.') {
			if (*(p2+1))
				p += sprintf(p, ", dc=");
		} else {
			*p = *p2;
			p++;
		}
		p2++;
	}
	free(domain);
	return MP_LDAP_OK;
}

static void free_ldap_server_list(char** ldap_server_list, int no_of_elements) {
	for (unsigned int i=0; i<no_of_elements; i++) {
		free(ldap_server_list[i]);
	}
	free(ldap_server_list);
}

void cut_string_at_line_feed(char* userdn_pos) {
	char* p = userdn_pos;
	while (*p) {
		if (*p == '\n') {
			*p = 0;
			break;
		}
		p++;
	}
}

/**
 * Returns a value from a key=value pair from config file.
 * It only works iwh files where key and value are separated by an '=' sign.
 *
 * Remember to free() the returned char*.
 * Returns NULL in if there is no key or no config file.
 */
char* get_value_from_file(const char* key, const char* file) {
	char* value = NULL;
	FILE* f;
	if ((f = fopen(file, "r"))) {
		char text_line[BUFFER_SIZE];
		while (!value && fgets(text_line, BUFFER_SIZE - 1, f)) {
			cut_string_at_line_feed(text_line);
			char* equal_sign = strchr(text_line, '=');
			if (equal_sign) {
				equal_sign[0] = '\0';
				char* current_key = text_line;
				char* current_value = equal_sign + 1;
				if (!strcmp(key, current_key)) {
					value = strdup(current_value);
					break;
				}
			}
		}
		fclose(f);
	}

	if (!value) {
		mplog_error("Couldn't find key '%s'", key);
	}

	return value;
}

static int get_first_empty_index(char** ldap_server_list)
{
	int i = 0;
	while (i < LDAP_SERVER_MAX_LIST_SIZE) {
		if (ldap_server_list && ldap_server_list[i] && *ldap_server_list[i]) {
			i++;
		} else {
			break;
		}
	}

	return i;
}

/**
 * Add ldap servers from list_of_candidates that match the given domain to ldap_server_list
 * This function assumes that memory was already allocated for the list.
 *
 */
static void add_servers_that_match_domain_to_list(char** ldap_server_list, const char* domain, const char* _list_of_candidates)
{
	char* list_of_candidates = strdup(_list_of_candidates); // strdup because strtok tends to do nasty stuff on the original strings
	mplog_verbose("add_servers_that_match_domain_to_list(%s, %s)", domain, list_of_candidates);
	int i = get_first_empty_index(ldap_server_list);

	char* delimiter = " ";
	char* candidate = strtok(list_of_candidates, delimiter);
	while (candidate != NULL) {
		if (i >= LDAP_SERVER_MAX_LIST_SIZE) {
			mplog_error("Unable to add more active directory servers to list because the max. capacity (%d) has been exceeded", LDAP_SERVER_MAX_LIST_SIZE);
			break;
		}
		char* candidate_domain = strchr(candidate, '.') + 1; /* Remove host name (until first dot) */
		char* colon_pos = strchr(candidate_domain, ':');
		colon_pos[0] = '\0'; /* Cut the port part of the domain */
		char* port = colon_pos + 1;
		char* ldap_protocol = "ldap://";
		if (!strcmp(port, "636") || !strcmp(port, "3269")) {
			ldap_protocol = "ldaps://";
		}
		unsigned int ldap_server_len = strlen(ldap_protocol) + strlen(candidate) + 1 + strlen(port); /* ldap://server:port - something like ldap://win2012.m-privacy.ad:389 */
		char ldap_server[ldap_server_len + 1];
		sprintf(ldap_server, "%s%s:%s", ldap_protocol, candidate, port);
		ldap_server[ldap_server_len] = '\0';

		if (ldap_server_len > LDAP_SERVER_BIND_STRING_MAX_LEN) {
			mplog_error("Unable to save active directory server '%s' as its name is longer than 64 chars which is not allowed by the Microsoft specification. If you still want this to happen, you will have to change the amount of alloced memory in the code.", ldap_server);
		} else {
			if (!strcasecmp(candidate_domain, domain)) {
				mplog_debug("Found matching ldap server in specified domain (%s): %s)", domain, candidate);
				ldap_server_list[i] = (char*) calloc(strlen(ldap_server) + 1, 1);
				strcpy(ldap_server_list[i], ldap_server);
				i++;
			}
		}

		candidate = strtok(NULL, delimiter); /* To continue tokenizing the previous string, we have to pass NULL */
	}
	free(list_of_candidates);
}

static char** gen_ldap_server_list(const char * domain) {
	mplog_debug("gen_ldap_server_list: Searching for Active Directory Global Catalogs and Domain Controllers. My domain is: %s", domain);

	if (!file_exists(GCS_AND_DCS)) {
		mplog_error("gen_ldap_server_list: file %s does not exist. Cannot read list of Global Catalogs and Domain Controllers. Aborting ldapsearch.", GCS_AND_DCS);
		return NULL;
	}

	char* mainDomain = get_value_from_file("CU_ACTIVEDIRECTORY_MAIN_DOMAIN", GCS_AND_DCS);
        char* globalCatalogList = get_value_from_file("CU_ACTIVEDIRECTORY_GCS", GCS_AND_DCS);
        char* domainControllerList = get_value_from_file("CU_ACTIVEDIRECTORY_DCS", GCS_AND_DCS);

	mplog_debug("gen_ldap_server_list: read mainDomain from /etc/cu/discovered-active-directory-gcs-and-dcs: %s", mainDomain);
	mplog_debug("gen_ldap_server_list: found globalCatalogList from /etc/cu/discovered-active-directory-gcs-and-dcs: %s", globalCatalogList);
	mplog_debug("gen_ldap_server_list: found domainControllerList from /etc/cu/discovered-active-directory-gcs-and-dcs: %s", domainControllerList);

	/* FQDN Length Limitations (from https://technet.microsoft.com/en-us/library/active-directory-maximum-limits-scalability%28v=ws.10%29.aspx#BKMK_FQDN)

	   Fully qualified domain names (FQDNs) in Active Directory cannot exceed 64 characters in total length, including hyphens and periods (.).
	   For example, the following host name has 65 characters; therefore, it is not valid in an Active Directory domain:

	   server10.branch-15.southaz.westernregion.northamerica.contoso.com
	*/
	/*
	  I am storing the GCs and DCs in order of preference for the ldap searches:
	  The first items are GC in my (sub)domain or in the 'main' domain
	  The next items are DC in my (sub)domain or in the 'main' domain
	 */

	char** ldap_server_list = (char** ) malloc(LDAP_SERVER_MAX_LIST_SIZE * sizeof(char *));
	if (!ldap_server_list) {
		mplog_error("gen_ldap_server_list: Could not allocate memory for list of GCs and DCs");
		syslog(LOG_ERR, "gen_ldap_server_list: Could not allocate memory for list of GCs and DCs");
		return NULL;
	}

	for (int i=0; i<LDAP_SERVER_MAX_LIST_SIZE; i++) {
		ldap_server_list[i] = NULL;
	}

	/* Loop through global catalogs list and look for ldap servers in my domain */
	mplog_debug("Looking for GCs in client's domain");
	add_servers_that_match_domain_to_list(ldap_server_list, domain, globalCatalogList);
	mplog_debug("Looking for DCs in client's domain");
	/* Loop through domain controllers list and look for ldap servers in my domain */
	add_servers_that_match_domain_to_list(ldap_server_list, domain, domainControllerList);

	if (strcmp(domain, mainDomain)) { // Only add more servers if client's domain is different from the main domain
		/* Loop through global catalogs list and look for ldap servers in main domain */
		mplog_debug("Looking for GCs in main domain");
		add_servers_that_match_domain_to_list(ldap_server_list, mainDomain, globalCatalogList);
		/* Loop through domain controllers list and look for ldap servers in main domain */
		mplog_debug("Looking for DCs in main domain");
		add_servers_that_match_domain_to_list(ldap_server_list, mainDomain, domainControllerList);
	}

	free(mainDomain);
        free(globalCatalogList);
        free(domainControllerList);

	mplog_verbose("Complete list of active directory servers in order of preference that we might query");

	for (int i=0; i<LDAP_SERVER_MAX_LIST_SIZE; i++) {
		char* ldap_server = ldap_server_list[i];
		if (!ldap_server || !*ldap_server) {
			break;
		}
		mplog_verbose("-> %s", ldap_server);
	}

	return ldap_server_list;
}

/*
 * Reallocates memory for a response buffer to accommodate new content.
 *
 * Parameters:
 *	 buffer: A pointer to a pointer to the buffer. The buffer must be allocated and null-terminated.
 *	 buffer_size: A pointer to the size of the buffer.
 *	 offset: The current offset within the buffer.
 *	 line: The line of text to be added to the buffer.
 *
 * Returns:
 *	 A pointer to the reallocated response buffer if successful, or NULL if memory reallocation fails.
 */
static void* realloc_buffer(char** buffer, size_t* buffer_size, size_t offset, size_t new_content_len) {
	size_t new_buffer_size = *buffer_size + BUFFER_SIZE;

	if (new_buffer_size > MAX_ALLOCATABLE_SIZE) {
		mplog_error("%s: Error: The buffer needs too much space (more than %zu).", __func__, MAX_ALLOCATABLE_SIZE);
		return NULL;
	}

	size_t needed_buffer_size = offset + new_content_len + 1; // +1 for null terminator
	if (needed_buffer_size > *buffer_size) {
		char* tmp_buf = (char*) realloc(*buffer, new_buffer_size);
		if (!tmp_buf) {
			mplog_error("%s: Error: Unable to reallocate memory for buffer.", __func__);
			// free(*buffer); // The caller should free it
			return NULL;
		}

		mplog_debug("%s: reallocating memory (current buffer size: %zu bytes; expected new size of response: %zu bytes). New buffer size: %zu", __func__, *buffer_size, needed_buffer_size, new_buffer_size);

		*buffer = tmp_buf;
		*buffer_size = new_buffer_size;
	}

	return *buffer;
}

// We only expect one result (like reading dn or a primary groupid for an account)
static int ldap_search_simple_attr(const char* ldap_server, const char* ldap_base, const char* filter, const char* attr, const int decode_base64, char (*attr_value)[BUFFER_SIZE]) {
	char* result = *attr_value;
	memset(result, 0, BUFFER_SIZE);

	char * attr_match = (char *) malloc(strlen(attr) + 3);
	if (!attr_match) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		return MP_LDAP_ERROR;
	}
	sprintf(attr_match, "%s: ", attr);

	char * attr_match_base64 = (char *) malloc(strlen(attr) + 4);
	if (!attr_match_base64) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		free(attr_match);
		return MP_LDAP_ERROR;
	}
	sprintf(attr_match_base64, "%s:: ", attr);

	char * command = (char *) malloc(BUFFER_SIZE);
	if (!command) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		free(attr_match);
		free(attr_match_base64);
		return MP_LDAP_ERROR;
	}
	snprintf(command, BUFFER_SIZE, "/usr/bin/timeout -k 10 60 /usr/bin/ldapsearch -LLL -Y GSSAPI -H '%s' -b '%s' -Omaxssf=256 -oldif-wrap=no \"%s\" %s 2>&1", ldap_server, ldap_base, filter, attr);
	command[BUFFER_SIZE - 1] = 0;

	mplog_debug("%s: running ldapsearch for attribute: %s", __func__, command);
	FILE* ppipe = popen(command, "r");
	if (!ppipe) {
		mplog_error("%s: Failed to execute ldapsearch for %s!", __func__, attr);
		syslog(LOG_ERR, "%s: Failed to execute ldapsearch for %s!", __func__, attr);
		free(command);
		free(attr_match);
		free(attr_match_base64);
		return MP_LDAP_QUERY_EXEC_ERROR;
	}

	const char * result_pos = NULL;
	unsigned int offset = 0;

	char * result_pos_decoded = (char *) malloc(BUFFER_SIZE);
	if (!result_pos_decoded) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		free(command);
		free(attr_match);
		free(attr_match_base64);
		return MP_LDAP_ERROR;
	}

	char * response_buf = (char *) malloc(BUFFER_SIZE);
	size_t response_size = BUFFER_SIZE;
	if (!response_buf) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		free(command);
		free(result_pos_decoded);
		free(attr_match);
		free(attr_match_base64);
		return MP_LDAP_ERROR;
	}
	memset(response_buf, 0, BUFFER_SIZE);

	char * line = (char *) malloc(BUFFER_SIZE);
	if (!line) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		free(command);
		free(result_pos_decoded);
		free(response_buf);
		free(attr_match);
		free(attr_match_base64);
		return MP_LDAP_ERROR;
	}
	memset(line, 0, BUFFER_SIZE);
	while (fgets(line, BUFFER_SIZE - 1, ppipe)) {
		void* realloc_result = realloc_buffer(&response_buf, &response_size, offset, strlen(line));
		if (!realloc_result) {
			mplog_error("%s: Error: Unable to reallocate memory! Aborting dynamic resizing of buffer...", __func__);
			break;
		}

		int bytes_read = snprintf(response_buf + offset, BUFFER_SIZE - 1 - offset, "%s", line);
		if (bytes_read > 0)
			offset += bytes_read;

		trim_str(line);
		result_pos = strcasestr(line, attr_match_base64); /* base64? */
		if (result_pos && *result_pos) {
			result_pos += strlen(attr_match_base64);
			mplog_verbose("Found base64-encoded attribute %s in reponse with value: %s", attr, result_pos);
			if (decode_base64) {
				memset(result_pos_decoded, 0, BUFFER_SIZE);
				int decode_succeeded = base64_decode_no_alloc(result_pos, &result_pos_decoded);
				if (decode_succeeded) {
					result_pos = result_pos_decoded;
					mplog_verbose("base64-decoding attr %s succeeded: %s", attr, result_pos);
				} else {
					mplog_error("Error! base64-decoding attr %s failed: %s", attr, result_pos);
					syslog(LOG_DEBUG, "%s: Error! base64-decoding attr %s failed: %s", __func__, attr, result_pos);
				}
			}
			break;
		}

		result_pos = strcasestr(line, attr_match);
		if (result_pos) {
			result_pos = line + (result_pos - line) + strlen(attr_match);
			mplog_verbose("%s: found attribute %s in reponse with value: %s", __func__, attr, result_pos);
			break;
		}
		memset(line, 0, BUFFER_SIZE);
	}
	response_buf[response_size - 1] = '\0';
	int err = pclose(ppipe) >> 8;

	if (!err && result_pos) {
		strncpy(result, result_pos, BUFFER_SIZE - 1);
		result[BUFFER_SIZE - 1] = 0;
	}
	free(result_pos_decoded);
	free(line);
	free(attr_match);
	free(attr_match_base64);

	if (err) {
		mplog_error("ldapsearch call for %s failed with error %d, command was %s, output was %s", attr, err, command, response_buf);
		syslog(LOG_ERR, "ldapsearch call for %s failed with error %d, command was %s, output was %s", attr, err, command, response_buf);
		add_error_to_list("ldapsearch call for %s failed with error %d, command was %s, output was %s", attr, err, command, response_buf);
		free(command);
		free(response_buf);
		return MP_LDAP_ERROR;
	} else if (!result || !*result) {
		mplog_debug("ldapsearch call for %s returned no result, command was %s, output was %s", attr, command, response_buf);
		// syslog(LOG_DEBUG, "ldapsearch call for %s returned no result, command was %s, output was %s", attr, command, response_buf);
		add_error_to_list("ldapsearch call for %s returned no result, command was %s, output was %s", attr, command, response_buf);
		free(command);
		free(response_buf);
		return MP_LDAP_EMPTY_RESPONSE_ERROR;
	} else {
		free(command);
		// Everything went well!
		mplog_debug("%s: Succeeded ldapsearching for attribute '%s' with filter '%s'", __func__, attr, filter);
		mplog_debug("%s: Result:", __func__);
		mplog_debug("%s: >> %s: %s", __func__, attr, result);

		mplog_verbose("%s: Complete LDAP response\n%s", __func__, response_buf);
		free(response_buf);

		return MP_LDAP_OK;
	}
}

static char* parse_usergroup(char* usergroup_line) {
	char* p = usergroup_line;
	while (*p) {
		*p = tolower(*p);
		p++;
	}
	const char* regex = ".*cn=[^,]*"
		"("
		"tgprouser|"
		"tgaudio|"
		"tgtransferspool|"
		"tgtransferauto|"
		"tgtransfer[0-9]*|"
		"tgprivileged|"
		"tgnoidletimeout|"
		"tgbandwidthhigh|"
		"tgbandwidth|"
		"tgfiltergroup[0-9]*|"
		"tgunfiltered|"
		"tgbebpoicon|"
		"tgmailicon|"
		"tgsignalicon|"
		"tgtoricon|"
		"tgchromeicon|"
		"tgstartpdf|"
		"tgmaxfilesize|"
		"tgadmin(?:config|maint|update|backuser|security|root)|"
		"tgopswat[0-9]*|"
		"tgvaithex"
		")"
		"[^,]*";

	return match(usergroup_line, regex, 1);
}

static int ldap_search_get_tg_groups(char* ldap_server, const char* ldap_base, const char* userdn, char** usergroups_ptr, size_t* usergroups_size_ptr) {
	char * command = (char *) malloc(BUFFER_SIZE);
	if (!command) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		return MP_LDAP_ERROR;
	}
	snprintf(command, BUFFER_SIZE - 1,
		 "/usr/bin/timeout -k 10 60 /usr/bin/ldapsearch -LLL -Y GSSAPI -H '%s' -b '%s' -Omaxssf=256 -oldif-wrap=no \""
			"(&"
				"(objectClass=group)"
				"(|"
					"(cn=*tgprouser*)"
					"(cn=*tgaudio*)"
					"(cn=*tgtransfer*)"
					"(cn=*tgprivileged*)"
					"(cn=*tgnoidletimeout*)"
					"(cn=*tgbandwidth*)"
					"(cn=*tgbandwidthhigh*)"
					"(cn=*tgtransferspool*)"
					"(cn=*tgfiltergroup*)"
					"(cn=*tgunfiltered*)"
					"(cn=*tgtransfer*)"
					"(cn=*tgbebpoicon*)"
					"(cn=*tgmailicon*)"
					"(cn=*tgtoricon*)"
					"(cn=*tgchromeicon*)"
					"(cn=*tgsignalicon*)"
					"(cn=*tgstartpdf*)"
					"(cn=*tgmaxfilesize*)"
					"(cn=*tgtransferauto*)"
					"(cn=*tgadmin*)"
					"(cn=*tgopswat*)"
					"(cn=*vaithex*)"
				")"
				"(member:1.2.840.113556.1.4.1941:=%s)"
			")"
			"\" memberOf 2>&1",
		 ldap_server, ldap_base, userdn);
	command[BUFFER_SIZE - 1] = 0;

	mplog_debug("%s: running ldapsearch for TGGROUPS: %s", __func__, command);
	FILE* ppipe = popen(command, "r");
	if (!ppipe) {
		mplog_error("%s: Failed to execute ldapsearch for TGGROUPS!", __func__);
		syslog(LOG_ERR, "%s: Failed to execute ldapsearch for TGGROUPS!", __func__);
		free(command);
		return MP_LDAP_QUERY_EXEC_ERROR;
	}

	char* usergroup_pos = NULL;

	char * usergroup_pos_decoded = (char *) malloc(BUFFER_SIZE);
	if (!usergroup_pos_decoded) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		free(command);
		return MP_LDAP_ERROR;
	}

	char * response_buf = (char *) malloc(BUFFER_SIZE);
	size_t response_size = BUFFER_SIZE;
	if (!response_buf) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		free(command);
		free(usergroup_pos_decoded);
		return MP_LDAP_ERROR;
	}
	response_buf[0] = 0;

	unsigned int offset = 0;

	(*usergroups_ptr)[0] = '\0';

	char line[BUFFER_SIZE];
	while (fgets(line, BUFFER_SIZE - 1, ppipe)) {
		void* realloc_result = realloc_buffer(&response_buf, &response_size, offset, strlen(line));
		if (!realloc_result) {
			mplog_error("%s: Error: Unable to reallocate memory! Aborting dynamic resizing of buffer...", __func__);
			break;
		}

		int bytes_read = snprintf(response_buf + offset, BUFFER_SIZE - 1 - offset, "%s", line);
		if (bytes_read > 0)
			offset += bytes_read;

		trim_str(line);
		mplog_verbose("%s: response line:\n%s", __func__, line);
		usergroup_pos = strcasestr(line, "dn:: ");
		if (usergroup_pos && *usergroup_pos) { /* base64? */
			usergroup_pos += strlen("dn:: ");
			mplog_verbose("Found base64-encoded tggroupin reponse with value: %s. Decoding...", usergroup_pos);
			int decode_succeeded = base64_decode_no_alloc(usergroup_pos, &usergroup_pos_decoded);
			if (decode_succeeded) {
				usergroup_pos = usergroup_pos_decoded;
			}
		} else {
			usergroup_pos = strcasestr(line, "dn: ");
			if (usergroup_pos && *usergroup_pos) {
				usergroup_pos += strlen("dn: ");
				mplog_verbose("%s: found tggroup in reponse with value: %s", __func__, usergroup_pos);
			}
		}

		if (usergroup_pos && *usergroup_pos) {
			char* usergroup = parse_usergroup(usergroup_pos);
			if (usergroup) {
				mplog_verbose("%s: parsed tggroup in reponse: %s", __func__, usergroup);
				void* realloc_result = realloc_buffer(usergroups_ptr, usergroups_size_ptr, strlen((*usergroups_ptr)) + 1, strlen(usergroup) + 1); // first +1 for '\0', second +1 for " "
				if (!realloc_result) {
					mplog_error("%s: Error: Unable to reallocate memory! Aborting dynamic resizing of buffer...", __func__);
					break;
				}
				strncat((*usergroups_ptr), usergroup, (*usergroups_size_ptr) - 2 - strlen((*usergroups_ptr)));
				strcat((*usergroups_ptr), " ");
				free(usergroup);
			}
		}
	}

	free(usergroup_pos_decoded);
	response_buf[response_size - 1] = 0;

	int err = pclose(ppipe) >> 8;
	if (err) {
		mplog_error("ldapsearch call for TGGROUPS failed with error %d, command was %s, output was %s", err, command, response_buf);
		syslog(LOG_ERR, "ldapsearch call for TGGROUPS failed with error %d, command was %s, output was %s", err, command, response_buf);
		add_error_to_list("ldapsearch call for TGGROUPS failed with error %d, command was %s, output was %s", err, command, response_buf);
		free(command);
		free(response_buf);
		return MP_LDAP_ERROR;
	} else if (!(*usergroups_ptr) || !*(*usergroups_ptr)) {
		mplog_debug("ldapsearch call for TGGROUPS returned no result, command was %s, output was %s", command, response_buf);
		// syslog(LOG_DEBUG, "ldapsearch call for TGGROUPS returned no result, command was %s, output was %s", command, response_buf);
		add_error_to_list("ldapsearch call for TGGROUPS returned no result, command was %s, output was %s", command, response_buf);
		free(command);
		free(response_buf);
		return MP_LDAP_NO_GROUPS_ERROR;
	} else {
		// Everything went well!
		(*usergroups_ptr)[(*usergroups_size_ptr) - 1] = '\0';
		mplog_debug("%s: Succeeded ldapsearching for TGGROUPS for user account '%s'!", __func__, userdn);
		mplog_debug("%s: Result:", __func__);
		mplog_debug("%s: >> TGGROUPS: %s", __func__, (*usergroups_ptr));

		free(command);
		free(response_buf);
		return MP_LDAP_OK;
	}
}

static int convert_binary_sid_to_string(const unsigned char* sid_bytes, char** sid_str)
{
	/*
	 * Very heavily inspired by a.k.a. copied from:
	 * https://gist.github.com/miromannino/04be6a64ea0b5f4d4254bb321e09d628
	 * Thank you very much, Miro Mannino!
	 *
	 * More about the SID structure: https://technet.microsoft.com/en-us/library/cc962011.aspx
	 * This bash script is also very interesting: https://bgstack15.wordpress.com/2018/02/26/get-sid-from-linux-ldapsearch-in-active-directory/
	 *
	 */

	char* p = *sid_str; // TODO this could be a problem: are we really writing where we should?
	int offset, size;

	// sid_bytes[0] is the Revision, we allow only version 1, because it's the only that exists right now.
	if (sid_bytes[0] != 1) {
		mplog_error("%s: SID revision must be 1", __func__);
		return 0;
	}

	p += sprintf(p, "S-1-");

	// The next byte specifies the numbers of sub authorities (number of dashes minus two)
	int sub_authority_count = sid_bytes[1];
	mplog_verbose("%s: sub_authority_count: %d", __func__, sub_authority_count);

	// IdentifierAuthority (6 bytes starting from the second) (big endian)
	long identifier_authority = 0;
	offset = 2;
	size = 6;
	for (int i = 0; i < size; i++) {
		identifier_authority |= (long) (sid_bytes[offset + i]) << (8 * (size - 1 - i));
	}

	mplog_verbose("%s: identifier_authority: %lu", __func__, identifier_authority);

	if (identifier_authority < pow(2, 32)) {
		p += sprintf(p, "%lu", identifier_authority);
	} else {
		p += sprintf(p, "%0X", identifier_authority);
	}

	// Iterate all the SubAuthority (little-endian)
	offset = 8;
	size = 4; // 32-bits (4 bytes) for each SubAuthority
	for (int i = 0; i < sub_authority_count; i++, offset += size) {
		long sub_authority = 0;
		for (int j = 0; j < size; j++) {
			sub_authority |= (long) (sid_bytes[offset + j]) << (8 * j);
			mplog_verbose("%s: sub-subauthority: %lu", __func__, sub_authority);
		}
		p += sprintf(p, "-%lu", sub_authority);
	}

	mplog_verbose("%s: SID as string: %s", __func__, *sid_str);

	return 1;
}

static int get_object_sid(char* ldap_server, const char* ldap_base, const char* ldap_login, char (*object_sid_ptr)[BUFFER_SIZE]) {
	int result;

	char* sid_str = *object_sid_ptr;
	sid_str[0] = 0;

	char object_sid_base64[BUFFER_SIZE];
	object_sid_base64[0] = 0;
	const char* filter = get_samaccount_filter(ldap_login);
	result = ldap_search_simple_attr(ldap_server, ldap_base, filter, "objectSid", 0, &object_sid_base64);

	if (result != MP_LDAP_OK) {
		return result;
	}

	size_t no_of_bytes = (strlen(object_sid_base64) * 6) / 8;
	char* object_sid_bytes = base64_decode(object_sid_base64);
	convert_binary_sid_to_string((unsigned char*) object_sid_bytes, (char**) &sid_str);
	mplog_debug("%s: object sid for user %s as string: %s", __func__, ldap_login, sid_str);

	return MP_LDAP_OK;
}

static int ldap_search_get_primary_tg_groups(char* ldap_server, const char* ldap_base, const char* ldap_login, char** usergroups_ptr, size_t* usergroups_size_ptr)
{
	/*
	 * The basic question is: is the user's primary group member of our tggroups?
	 * This question is not answered with the query we have done until now. We can try and get the answer to this using the following work-around:
	 *
	 * We need to get the groups that the primary group is a member of.
	 *
	 * The only reference we have to the user's primary group is its primaryGroupId (the default value for "Domain Users" is 513).
	 *
	 * Unfortunately, we cannot query the ldap server using the primary group id as a parameter
	 * We need to somehow get the primary group's dn. Once we have the dn (e.g. CN=Domänen-Benutzer,CN=Users,DC=M-PRIVACY,DC=DEV), we can query for groups as we would do for a normal user.
	 *
	 * If we had the objectSid of the primary group, we could do something like:
	 *
	 * /usr/bin/ldapsearch -H '%s' -b '%s' -Omaxssf=%u -oldif-wrap=no \"(objectSid=S-1-5-21-3573158993-1889095830-1552970993-513)\" dn
	 * Fortunately (I am not sure this is always the case), the objectSid only differs from the one of the current user in the number after the last dash.
	 *
	 * So we could get the get objectSid for user (e.g. S-1-5-21-3573158993-1889095830-1552970993-6784637242) and replace the number after the last dash with the primary group, resulting
	 * in a (hopefully valid) objectSid for the primary group (e.g. S-1-5-21-3573158993-1889095830-1552970993-513).
	 *
	 * This should actually work.
	 *
	 * There is another small difficulty in this work-around. The ldap server stores object sid's in a binary format and returns them encoded in base64 but we can only query using the string
	 * so we will have to transform the bytes to strings:
	 * See following java example:
	 * https://gist.github.com/miromannino/04be6a64ea0b5f4d4254bb321e09d628
	 * More about the SID structure: https://technet.microsoft.com/en-us/library/cc962011.aspx
	 * This bash script is also very interesting: https://bgstack15.wordpress.com/2018/02/26/get-sid-from-linux-ldapsearch-in-active-directory/
	 */


	int result = MP_LDAP_OK;

	// Get primary_group_id for user
	char user_primary_group_id[BUFFER_SIZE];

	const char* filter = get_samaccount_filter(ldap_login);
	result = ldap_search_simple_attr(ldap_server, ldap_base, filter, "primaryGroupId", 1, &user_primary_group_id);

	if (result != MP_LDAP_OK) {
		return result;
	}

	mplog_debug("%s: primary group id for user %s is %s", __func__, ldap_login, user_primary_group_id);

	char user_sid[BUFFER_SIZE];
	result = get_object_sid(ldap_server, ldap_base, ldap_login, &user_sid);

	if (result != MP_LDAP_OK) {
		return result;
	}
	mplog_debug("%s: object_sid for user %s is %s", __func__, ldap_login, user_sid);

	char primary_group_sid[BUFFER_SIZE];
	strncpy(primary_group_sid, user_sid, BUFFER_SIZE - 1);
	primary_group_sid[BUFFER_SIZE - 1] = 0;
	char* p = primary_group_sid + strlen(user_sid);
	while (*p !='-') {
		p--;
	}
	*(p + 1) = 0;
	strcat(primary_group_sid, user_primary_group_id);
	mplog_debug("%s: object_sid for user's primary group: %s", __func__, primary_group_sid);
	// Now ldap search like this /usr/bin/ldapsearch -Y GSSAPI -H '%s' -b '%s' -Omaxssf=%u -oldif-wrap=no \"(objectSid=S-1-5-21-3573158993-1889095830-1552970993-513)\" dn

	char primary_group_dn[BUFFER_SIZE];
	const char* sid_filter = get_object_sid_filter(primary_group_sid);
	result = ldap_search_simple_attr(ldap_server, ldap_base, sid_filter, "dn", 1, &primary_group_dn);
	if (result != MP_LDAP_OK) {
		return result;
	}
	mplog_debug("%s: primary group dn: %s", __func__, primary_group_dn);

	char* usergroups = (char*) malloc(BUFFER_SIZE);
	if (!usergroups) {
		mplog_error("%s: failed to allocate memory for usergroups!", __func__);
		return MP_LDAP_MALLOC_ERROR;
	}
	usergroups[0] = '\0';
	size_t usergroups_size = BUFFER_SIZE;

	result = ldap_search_get_tg_groups(ldap_server, ldap_base, primary_group_dn, &usergroups, &usergroups_size);
	if (result != MP_LDAP_OK) {
		return result;
	}

	(*usergroups_ptr)[0] = '\0';
	strncat((*usergroups_ptr), usergroups, usergroups_size - 2 - strlen(usergroups));
	strcat((*usergroups_ptr), " ");

	mplog_debug("%s: Succeeded ldapsearching for TGGROUPS in primary group for user account '%s'!", __func__, ldap_login);
	mplog_debug("%s: Result:", __func__);
	mplog_debug("%s: >> TGGROUPS (from primary group): %s", __func__, (*usergroups_ptr));

	return MP_LDAP_OK;
}

static int ldap_search_get_tg_groups_unoptimized(char* ldap_server, const char* ldap_base, const char* ldap_login, char** usergroups_ptr, size_t* usergroups_size_ptr)
{
	/*
	  Original call (with a huge response depending on the Active Directory group config):
	  - get dn using samaaccountname
	  snprintf(command, 4094,
	  ("/usr/bin/timeout -k 10 60 /usr/bin/ldapsearch -H '%s' -b '%s' -Omaxssf=%u%s -oldif-wrap=no \"(sAMAccountName=%s)\" dn 2>&1", krbldapbind, krbldapbase, 256, ldapextra, constuser);

	  - and with the dn:
	  snprintf(command, 4094,
	  "/usr/bin/timeout -k 10 60 /usr/bin/ldapsearch -H '%s' -b '%s' -Omaxssf=%u%s -oldif-wrap=no \"(|(member:1.2.840.113556.1.4.1941:=%s)(sAMAccountName=%s))\" memberOf 2>&1", krbldapbind, krbldapbase, 256, ldapextra, escaped_userdn, constuser);
	 */

	char userdn[BUFFER_SIZE];
	char userdn_escaped[BUFFER_SIZE];
	const char* filter = get_samaccount_filter(ldap_login);
	int result = ldap_search_simple_attr(ldap_server, ldap_base, filter, "dn", 1, &userdn);
	if (result == MP_LDAP_OK) {
		escape_distinguished_names_special_chars_no_alloc(userdn, &userdn_escaped);
	}

	char * command = (char *) malloc(BUFFER_SIZE);
	if (!command) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		return MP_LDAP_ERROR;
	}
	if (userdn_escaped && *userdn_escaped) {
		snprintf(command, BUFFER_SIZE - 1,
			 "/usr/bin/timeout -k 10 60 /usr/bin/ldapsearch -LLL -Y GSSAPI -H '%s' -b '%s' -Omaxssf=256 -oldif-wrap=no \"(|(sAMAccountName=%s)(member:1.2.840.113556.1.4.1941:=%s))\" memberOf 2>&1",
			 ldap_server, ldap_base, ldap_login, userdn_escaped);
	} else {
		snprintf(command, BUFFER_SIZE - 1,
			 "/usr/bin/timeout -k 10 60 /usr/bin/ldapsearch -LLL -Y GSSAPI -H '%s' -b '%s' -Omaxssf=256 -oldif-wrap=no \"(sAMAccountName=%s)\" memberOf 2>&1",
			 ldap_server, ldap_base, ldap_login);

	}
	command[BUFFER_SIZE - 1] = 0;

	mplog_debug("%s: running ldapsearch for TGGROUPS: %s", __func__, command);
	FILE* ppipe = popen(command, "r");
	if (!ppipe) {
		mplog_error("%s: Failed to execute ldapsearch for TGGROUPS!", __func__);
		syslog(LOG_ERR, "%s: Failed to execute ldapsearch for TGGROUPS!", __func__);
		free (command);
		return MP_LDAP_QUERY_EXEC_ERROR;
	}

	char* usergroup_pos = NULL;
	char * usergroup_pos_decoded = (char *) malloc(BUFFER_SIZE);
	if (!usergroup_pos_decoded) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		free (command);
		return MP_LDAP_ERROR;
	}

	char * response_buf = (char *) malloc(BUFFER_SIZE);
	size_t response_size = BUFFER_SIZE;
	if (!response_buf) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		free (command);
		free (usergroup_pos_decoded);
		return MP_LDAP_ERROR;
	}
	response_buf[0] = 0;

	unsigned int offset = 0;

	(*usergroups_ptr)[0] = '\0';

	char line[BUFFER_SIZE];
	line[BUFFER_SIZE - 1] = 0;
	while (fgets(line, BUFFER_SIZE - 1, ppipe)) {
		void* realloc_result = realloc_buffer(&response_buf, &response_size, offset, strlen(line));
		if (!realloc_result) {
			mplog_error("%s: Error: Unable to reallocate memory! Aborting dynamic resizing of buffer...", __func__);
			break;
		}

		int bytes_read = snprintf(response_buf + offset, BUFFER_SIZE - 1 - offset, "%s", line);
		if (bytes_read > 0)
			offset += bytes_read;

		trim_str(line);
		usergroup_pos = strcasestr(line, "memberOf:: ");
		if (usergroup_pos && *usergroup_pos) { /* base64? */
			usergroup_pos += strlen("memberOf:: ");
			mplog_verbose("Found base64-encoded tggroupin reponse with value: %s. Decoding...", usergroup_pos);
			int decode_succeeded = base64_decode_no_alloc(usergroup_pos, &usergroup_pos_decoded);
			if (decode_succeeded) {
				usergroup_pos = usergroup_pos_decoded;
			}
		} else {
			usergroup_pos = strcasestr(line, "memberOf: ");
			if (usergroup_pos && *usergroup_pos) {
				usergroup_pos += strlen("memberOf: ");
				mplog_verbose("%s: found tggroup in reponse with value: %s", __func__, usergroup_pos);
			}
		}

		if (usergroup_pos && *usergroup_pos) {
			char* usergroup = parse_usergroup(usergroup_pos);
			if (usergroup) {
				void* realloc_result = realloc_buffer(usergroups_ptr, usergroups_size_ptr, strlen((*usergroups_ptr)) + 1, strlen(usergroup) + 1); // first +1 for '\0', second +1 for " "
				if (!realloc_result) {
					mplog_error("%s: Error: Unable to reallocate memory! Aborting dynamic resizing of buffer...", __func__);
					break;
				}
				mplog_verbose("%s: parsed tggroup in reponse: %s", __func__, usergroup);
				strncat((*usergroups_ptr), usergroup, (*usergroups_size_ptr) - 2 - strlen((*usergroups_ptr)));
				strcat(*usergroups_ptr, " ");
				free(usergroup);
			}
		}
	}

	free (usergroup_pos_decoded);
	response_buf[response_size - 1] = 0;

	int err = pclose(ppipe) >> 8;
	if (err) {
		mplog_error("ldapsearch call (unoptimized) for TGGROUPS failed with error %d, command was %s, output was %s", err, command, response_buf);
		// syslog(LOG_ERR, "ldapsearch (old) call for TGGROUPS failed with error %d, command was %s, output was %s", err, command, response_buf);
		add_error_to_list("ldapsearch (old) call for TGGROUPS failed with error %d, command was %s, output was %s", err, command, response_buf);
		free (command);
		free (response_buf);
		return MP_LDAP_ERROR;
	} else if (!(*usergroups_ptr) || !*(usergroups_ptr)) {
		mplog_debug("ldapsearch call (unoptimized) for TGGROUPS returned no result, command was %s, output was %s", command, response_buf);
		// syslog(LOG_DEBUG, "ldapsearch call for TGGROUPS returned no result, command was %s, output was %s", command, response_buf);
		add_error_to_list("ldapsearch call for TGGROUPS returned no result, command was %s, output was %s", command, response_buf);
		free (command);
		free (response_buf);
		return MP_LDAP_NO_GROUPS_ERROR;
	} else {
		// Everything went well!
		(*usergroups_ptr)[(*usergroups_size_ptr) - 1] = '\0';
		mplog_debug("%s: Succeeded ldapsearching (unoptimized) for TGGROUPS for user account '%s'!", __func__, ldap_login);
		mplog_debug("%s: Result:", __func__);
		mplog_debug("%s: >> TGGROUPS: %s", __func__, (*usergroups_ptr));

		mplog_verbose("%s: Complete LDAP response\n%s", __func__, response_buf);

		free (command);
		free (response_buf);
		return MP_LDAP_OK;
	}
}

void get_ad_display_name(char* ldap_server, const char* ldap_base, const char* ldap_login) {
	char displayname[BUFFER_SIZE] = ""; // The displayName AD attribute. Just so that prepareuser has the "full name" for new users.

	const char* filter = get_samaccount_filter(ldap_login);

	mplog_debug("%s: Trying to get displayName for %s from AD server", __func__, ldap_login);
	int result = ldap_search_simple_attr(ldap_server, ldap_base, filter, "displayName", 1, &displayname);
	if (result == MP_LDAP_OK) {
		if (displayname[0]) {
			mplog_debug("%s: Setting ADDISPLAYNAME to %s", __func__, displayname);
			portable_setenv("ADDISPLAYNAME", displayname);
		} else {
			mplog_error("%s: Managed to get displayName from AD server *but* it is empty", __func__);
		}
	} else {
		mplog_error("%s: Failed to get displayName from AD server", __func__);
	}
}

static int perform_ldap_search(char* ldap_server, const char* ldap_base, const char* ldap_login)
{
	/* WARNING: there might be a problem while querying DC's without an ldap_base, so the optimized query might not work.
	 * We might have to add it again (somewhere) or revert to unoptimized search if a specific error was returned
	 * (until now I have seen an error 32). With GC's there doesn't seem to be a problem.
	 */
	char* usergroups = NULL; // to avoid compiler complaining about non-initializiation. It *is* being initialized in the if/else
	int result;

	char* tmp = ldap_server + strlen("ldap://"); // This should also work for ldaps:// because it would be enough to get rid of the ':'
	if (strchr(tmp, ':')) {
		// 'auto' or 'realm' (the port is always included, so we use this to distinguish it from the pure IP address)
		mplog_debug("%s: ldap server autodetection is set to 'auto' or 'realm'. Using optimized ldap search", __func__);

		char userdn[BUFFER_SIZE];
		const char* filter = get_samaccount_filter(ldap_login);
		result = ldap_search_simple_attr(ldap_server, ldap_base, filter, "dn", 1, &userdn);

		if (result != MP_LDAP_OK) {
			mplog_error("Failed to get 'dn' for user %s (result %d)", ldap_login, result);
		}

		if (result == MP_LDAP_OK) {
			get_ad_display_name(ldap_server, ldap_base, ldap_login); // Just for the full name while preparing user. Not really important.

			char userdn_escaped[BUFFER_SIZE];
			escape_distinguished_names_special_chars_no_alloc(userdn, &userdn_escaped);
			char* normal_group_tg_groups = (char*) malloc(BUFFER_SIZE);
			if (!normal_group_tg_groups) {
				mplog_error("%s: failed to allocate memory for 'normal' usergroups!", __func__);
				return MP_LDAP_MALLOC_ERROR;
			}
			normal_group_tg_groups[0] = '\0';
			size_t normal_group_tg_groups_size = BUFFER_SIZE;
			int normal_groups_query_result = ldap_search_get_tg_groups(ldap_server, ldap_base, userdn_escaped, &normal_group_tg_groups, &normal_group_tg_groups_size);

			// Experimental
			char* primary_group_tg_groups = (char*) malloc(BUFFER_SIZE);
			if (!primary_group_tg_groups) {
				mplog_error("%s: failed to allocate memory for 'primary' usergroups!", __func__);
				return MP_LDAP_MALLOC_ERROR;
			}
			primary_group_tg_groups[0] = '\0';
			size_t primary_group_tg_groups_size = BUFFER_SIZE;
			int primary_groups_query_result = ldap_search_get_primary_tg_groups(ldap_server, ldap_base, ldap_login, &primary_group_tg_groups, &primary_group_tg_groups_size);

			if (normal_groups_query_result == MP_LDAP_OK ||
			    primary_groups_query_result == MP_LDAP_OK) {
				if (normal_groups_query_result != MP_LDAP_OK) {
					normal_group_tg_groups[0] = 0;
				}
				if (primary_groups_query_result != MP_LDAP_OK) {
					primary_group_tg_groups[0] = 0;
				}
				size_t usergroups_size = strlen(normal_group_tg_groups) + strlen(primary_group_tg_groups) + 1;
				usergroups = (char*) malloc(usergroups_size);
				if (!usergroups) {
					mplog_error("%s: failed to allocate memory for usergroups!", __func__);
					result = MP_LDAP_MALLOC_ERROR;
				} else {
					snprintf(usergroups, usergroups_size, "%s %s", normal_group_tg_groups, primary_group_tg_groups);
					usergroups[usergroups_size - 1] = 0;
					result = MP_LDAP_OK;
				}
			} else if (normal_groups_query_result == MP_LDAP_QUERY_EXEC_ERROR ||
				   primary_groups_query_result == MP_LDAP_QUERY_EXEC_ERROR) {
				result = MP_LDAP_QUERY_EXEC_ERROR;
			} else {
				result = MP_LDAP_NO_GROUPS_ERROR;
			}
			free(normal_group_tg_groups);
			free(primary_group_tg_groups);
		}
	} else {
		// 'manual'. It only has an IP address and no port (after we have removed ldap://). Use old request (unoptimized) with sAMAccountName
		mplog_debug("%s: ldap server autodetection is set to 'manual'. Cannot use optimized ldap search. If %s belongs to a lot of groups, the authorization might take a long time. It is highly recommended to switch to 'auto' or to 'realm'", __func__, ldap_login);

		size_t usergroups_size = BUFFER_SIZE;
		// We don't need the user dn for the unoptimized search. We just get all the available info for the samaccount
		result = ldap_search_get_tg_groups_unoptimized(ldap_server, ldap_base, ldap_login, &usergroups, &usergroups_size);
	}

	if (result == MP_LDAP_OK) {
		portable_unsetenv("KRB5CCNAME");
		if (usergroups[0]) {
			portable_setenv("TGGROUPS", usergroups);
		} else {
			portable_setenv("TGGROUPS", "empty");
		}
	} else if (result == MP_LDAP_NO_GROUPS_ERROR) {
		portable_unsetenv("KRB5CCNAME");
		portable_setenv("TGGROUPS", "empty");
	}

	free(usergroups);

	return result;
}

static int perform_ldap_searches(const char* domain, char* ldap_server, const char* ldap_base, const char* ldap_login, const char* ldap_auto)
{
	// Tell the compiler I know we are ignoring these (don't want to change the API)
	(void) ldap_server;
	(void) ldap_auto;

	char** ldap_servers_list;
	int no_of_ldap_servers = 0;

	init_error_list();

	ldap_servers_list = gen_ldap_server_list(domain);
	no_of_ldap_servers = LDAP_SERVER_MAX_LIST_SIZE;
	if (!ldap_servers_list || !ldap_servers_list[0]) {
		mplog_error("LDAP bind auto: generating URI failed");
		syslog(LOG_ERR, "LDAP bind auto: generating URI failed");
		return MP_LDAP_ERROR;
	}

	int rc = MP_LDAP_ERROR;
	for (int i = 0; i < no_of_ldap_servers; i ++) {
		char* current_server = ldap_servers_list[i];
		if (!current_server || !current_server[0]) {
			mplog_debug("Reached end of list. Not querying any more ldap servers");
			break;
		}
		mplog_debug("Performing ldapsearch (try no. %d) with: %s", i + 1, current_server);
		mplog_debug("perform_ldap_search(ldap_server, ldap_base, ldap_login): (%s, %s, %s)", current_server, ldap_base, ldap_login);
		rc = perform_ldap_search(current_server, ldap_base, ldap_login);
		if (rc == MP_LDAP_OK) { // perform_ldap_search only sets GSASL_OK at the end if everything went well
			break;
		}
	}

	if (rc != MP_LDAP_OK) {
		// log all ldapsearch failues to auth.log only if the authentication fails (they are all logged to the vnc and/or ssh server anyway)
		for (unsigned int i=0; i<LDAP_SERVER_MAX_LIST_SIZE; i++) {
			if (!error_list || !error_list[i] || !*error_list[i]) {
				break; // reached end of list
			}
			syslog(LOG_ERR, "%s", error_list[i]);
		}
	}

	free_error_list();

	return rc;
}

static char* get_service_principal_name()
{
	char* service_principal_name = get_value_from_file("CU_ACTIVEDIRECTORY_SERVICE_PRINCIPAL_NAME", GCS_AND_DCS);

	if (!service_principal_name) {
		mplog_error("Could not get service_principal_name from '%s'.", GCS_AND_DCS);
		return NULL;
	}

	return strdup(service_principal_name);
}

static int run_kinit()
{
	char command[BUFFER_SIZE];
	snprintf(command, BUFFER_SIZE, "/usr/bin/kinit -k %s &>/dev/null", get_service_principal_name());
	command[BUFFER_SIZE - 1] = 0;
	mplog_debug("Trying to authenticate using keytab (%s)", command);
	int error = system(command);
	return !error;
}

static char* get_skel_value(const char* key)
{
	const char* config_file = "/etc/cu/skel";
	char* value = get_value_from_file(key, config_file);
	if (!value || !*value) {
		mplog_error("Warning! Failed to read %s from %s.", key, config_file);
		return NULL;
	}

	char* p = value;
	if (starts_with(p, "\"") || starts_with(p, "'")) {
		p++;
	}
	if (ends_with(p, "\"") || ends_with(p, "'")) {
		p[strlen(p) - 1] = 0;
	}

	if (!p || !*p) {
		mplog_error("Warning! Failed to read %s from %s.", key, config_file);
		return NULL;
	}

	char* skel_value = strdup(p);
	free(value);
	return skel_value;
}

static int is_check_ad_tg_groups()
{
	char* skel_value = get_skel_value("ADCHECKTGGROUPS");
	if (!skel_value || !*skel_value) {
		return 0;
	}
	int value = !strcmp("yes", skel_value);
	free(skel_value);
	return value;
}

static int is_ad_openldap()
{
	char* authmethod = get_skel_value("AUTHMETHOD");
	if (!authmethod || !*authmethod) {
		return 0;
	}
	int is_ad_open_ldap = !strcmp("AD-OpenLDAP", authmethod);
	free(authmethod);
	return is_ad_open_ldap;
}

static char* get_ldap_base_from_openldap()
{
	char* ldap_base = NULL;

	char * command = (char *) malloc(BUFFER_SIZE);
	if (!command) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		return NULL;
	}

	char* ldap_protocol = get_skel_value("LDAPPROTO1");
	if (!ldap_protocol) {
		free(command);
		mplog_error("%s: failed to read ldap_protocol. Aborting...", __func__);
		return NULL;
	}
	char* ldap_server = get_skel_value("LDAPSERVER1");
	if (!ldap_server) {
		free(command);
		free(ldap_protocol);
		mplog_error("%s: failed to read ldap_server. Aborting...", __func__);
		return NULL;
	}
	snprintf(command, BUFFER_SIZE - 1,
			 "/usr/bin/timeout -k 10 60 /usr/bin/ldapsearch -x -LLL -H %s//%s -s base -b \"\" namingContexts 2>&1",
			 ldap_protocol,
			 ldap_server);
	command[BUFFER_SIZE - 1] = 0;

	mplog_debug("%s: running ldapsearch for ldap base: %s", __func__, command);
	FILE* ppipe = popen(command, "r");
	if (!ppipe) {
		mplog_error("%s: Failed to execute ldapsearch for ldap base!", __func__);
		syslog(LOG_ERR, "%s: Failed to execute ldapsearch for ldap base!", __func__);
		free(command);
		return NULL;
	}
	free(command);
	free(ldap_protocol);
	free(ldap_server);

	char* naming_context_pos = NULL;

	char * response_buf = (char *) malloc(BUFFER_SIZE);
	size_t response_size = BUFFER_SIZE;
	if (!response_buf) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		return NULL;
	}
	response_buf[0] = 0;

	unsigned int offset = 0;

	char line[BUFFER_SIZE];
	while (fgets(line, BUFFER_SIZE - 1, ppipe)) {
		void* realloc_result = realloc_buffer(&response_buf, &response_size, offset, strlen(line));
		if (!realloc_result) {
			mplog_error("%s: Error: Unable to reallocate memory! Aborting dynamic resizing of buffer...", __func__);
			break;
		}

		int bytes_read = snprintf(response_buf + offset, BUFFER_SIZE - 1 - offset, "%s", line);
		if (bytes_read > 0)
			offset += bytes_read;

		trim_str(line);
		mplog_verbose("%s: response line:\n%s", __func__, line);

		naming_context_pos = strcasestr(line, "namingContexts: ");
		if (naming_context_pos && *naming_context_pos) {
			naming_context_pos += strlen("namingContexts: ");
			mplog_debug("%s: found namingContext in reponse with value: %s", __func__, naming_context_pos);
			ldap_base = strdup(naming_context_pos);
			break;
		}
	}

	free(response_buf);
	return ldap_base;
}

static int get_tg_groups_from_openldap(const char* username, char (*usergroups_ptr)[TGGROUPS_BUF_SIZE])
{
	init_error_list();

	char* usergroups = *usergroups_ptr;
	usergroups[0] = 0;

	char * command = (char *) malloc(BUFFER_SIZE);
	if (!command) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		return MP_LDAP_ERROR;
	}

	char * command_redacted = (char *) malloc(BUFFER_SIZE);
	if (!command_redacted) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		free(command);
		return MP_LDAP_ERROR;
	}

	char* ldap_protocol = get_skel_value("LDAPPROTO1");
	if (!ldap_protocol) {
		free(command);
		free(command_redacted);
		mplog_error("%s: failed to read ldap_protocol. Aborting...", __func__);
		return MP_LDAP_ERROR;
	}
	char* ldap_server = get_skel_value("LDAPSERVER1");
	if (!ldap_server) {
		free(command);
		free(command_redacted);
		free(ldap_protocol);
		mplog_error("%s: failed to read ldap_server. Aborting...", __func__);
		return MP_LDAP_ERROR;
	}
	char* ldap_bind_dn_user = get_skel_value("LDAPBINDDNUSER");
	if (!ldap_bind_dn_user) {
		free(command);
		free(command_redacted);
		free(ldap_protocol);
		free(ldap_server);
		mplog_error("%s: failed to read ldap_bind_dn_user. Aborting...", __func__);
		return MP_LDAP_ERROR;
	}
	char* ldap_bind_dn_password = get_skel_value("LDAPBINDDNPASSWD");
	if (!ldap_bind_dn_password) {
		free(command);
		free(command_redacted);
		free(ldap_protocol);
		free(ldap_server);
		free(ldap_bind_dn_user);
		mplog_error("%s: failed to read ldap_bind_dn_password. Aborting...", __func__);
		return MP_LDAP_ERROR;
	}
	char* ldap_base = get_skel_value("LDAPBASE");
	if (ldap_base && !strcmp(ldap_base, "auto")) {
		mplog_debug("%s: ldap base is set to 'auto'. Trying to figure it out automatically...", __func__);
		free(ldap_base);
		ldap_base = get_ldap_base_from_openldap();
	}
	if (!ldap_base) {
		free(command);
		free(command_redacted);
		free(ldap_protocol);
		free(ldap_server);
		free(ldap_bind_dn_user);
		free(ldap_bind_dn_password);
		mplog_error("%s: failed to read or figure out ldap_base automatically. Aborting...", __func__);
		return MP_LDAP_ERROR;
	}
	mplog_debug("%s: ldap base is <%s>", __func__, ldap_base);

	// First successfull attempt: "/usr/bin/timeout -k 10 60 /usr/bin/ldapsearch -x -LLL -b \"%s\" -H %s//%s -D \"%s\" -w \"%s\" -b \"uid=%s,ou=people,dc=m-privacy,dc=dev\" dn memberOf 2>&1",
	// With a filter, it should work too: "/usr/bin/timeout -k 10 60 /usr/bin/ldapsearch -x -LLL -b \"%s\" -H %s//%s -D \"%s\" -w \"%s\"  \"(uid=%s)\" dn memberOf 2>&1",
	// In the client's OpenLDAP, they don't use uid but cn: "/usr/bin/timeout -k 10 60 /usr/bin/ldapsearch -x -LLL -b \"%s\" -H %s//%s -D \"%s\" -w \"%s\"  \"(cn=%s)\" dn memberOf 2>&1",
	snprintf(command, BUFFER_SIZE - 1,
			 "/usr/bin/timeout -k 10 60 /usr/bin/ldapsearch -x -LLL -b \"%s\" -H %s//%s -D \"%s\" -w \"%s\" \"(|(cn=%s)(uid=%s)(sAMAccountName=%s))\" dn memberOf 2>&1",
			 ldap_base,
			 ldap_protocol,
			 ldap_server,
			 ldap_bind_dn_user,
			 ldap_bind_dn_password,
			 username,
			 username,
			 username);
	command[BUFFER_SIZE - 1] = 0;

	snprintf(command_redacted, BUFFER_SIZE - 1,
			 "/usr/bin/timeout -k 10 60 /usr/bin/ldapsearch -x -LLL -b \"%s\" -H %s//%s -D \"%s\" -w \"*******\" \"(|(cn=%s)(uid=%s)(sAMAccountName=%s))\" dn memberOf 2>&1",
			 ldap_base,
			 ldap_protocol,
			 ldap_server,
			 ldap_bind_dn_user,
			 username,
			 username,
			 username);
	command_redacted[BUFFER_SIZE - 1] = 0;

	mplog_debug("%s: running ldapsearch for TGGROUPS: %s", __func__, command_redacted);
	FILE* ppipe = popen(command, "r");
	if (!ppipe) {
		mplog_error("%s: Failed to execute ldapsearch for TGGROUPS!", __func__);
		syslog(LOG_ERR, "%s: Failed to execute ldapsearch for TGGROUPS!", __func__);
		free(command);
		free(command_redacted);
		return MP_LDAP_ERROR;
	}
	free(command);
	free(ldap_base);
	free(ldap_protocol);
	free(ldap_server);
	free(ldap_bind_dn_user);
	free(ldap_bind_dn_password);

	char* usergroup_pos = NULL;

	char * usergroup_pos_decoded = (char *) malloc(BUFFER_SIZE);
	if (!usergroup_pos_decoded) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		free(command_redacted);
		return MP_LDAP_ERROR;
	}

	char * response_buf = (char *) malloc(BUFFER_SIZE);
	size_t response_size = BUFFER_SIZE;
	if (!response_buf) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		free(command_redacted);
		free(usergroup_pos_decoded);
		return MP_LDAP_ERROR;
	}
	response_buf[0] = 0;

	unsigned int offset = 0;

	char line[BUFFER_SIZE];
	while (fgets(line, BUFFER_SIZE - 1, ppipe)) {
		void* realloc_result = realloc_buffer(&response_buf, &response_size, offset, strlen(line));
		if (!realloc_result) {
			mplog_error("%s: Error: Unable to reallocate memory! Aborting dynamic resizing of buffer...", __func__);
			break;
		}

		int bytes_read = snprintf(response_buf + offset, BUFFER_SIZE - 1 - offset, "%s", line);
		if (bytes_read > 0)
			offset += bytes_read;

		trim_str(line);
		mplog_verbose("%s: response line:\n%s", __func__, line);
		usergroup_pos = strcasestr(line, "memberOf:: ");
		if (usergroup_pos && *usergroup_pos) { /* base64? */
			usergroup_pos += strlen("memberOf:: ");
			mplog_verbose("Found base64-encoded tggroup in reponse with value: %s. Decoding...", usergroup_pos);
			int decode_succeeded = base64_decode_no_alloc(usergroup_pos, &usergroup_pos_decoded);
			if (decode_succeeded) {
				usergroup_pos = usergroup_pos_decoded;
			}
		} else {
			usergroup_pos = strcasestr(line, "memberOf: ");
			if (usergroup_pos && *usergroup_pos) {
				usergroup_pos += strlen("memberOf: ");
				mplog_verbose("%s: found tggroup in reponse with value: %s", __func__, usergroup_pos);
			}
		}

		if (usergroup_pos && *usergroup_pos) {
			char* usergroup = parse_usergroup(usergroup_pos);
			if (usergroup) {
				mplog_verbose("%s: parsed tggroup in reponse: %s", __func__, usergroup);
				strncat(usergroups, usergroup, TGGROUPS_BUF_SIZE - 2 - strlen(usergroups));
				strcat(usergroups, " ");
				free(usergroup);
			}
		}
	}

	free(usergroup_pos_decoded);
	response_buf[response_size - 1] = 0;

	int err = pclose(ppipe) >> 8;
	if (err) {
		mplog_error("ldapsearch call for TGGROUPS failed with error %d, command was %s, output was %s", err, command_redacted, response_buf);
		syslog(LOG_ERR, "ldapsearch call for TGGROUPS failed with error %d, command was %s, output was %s", err, command_redacted, response_buf);
		add_error_to_list("ldapsearch call for TGGROUPS failed with error %u, command was %s, output was %s", err, command_redacted, response_buf);
		free(command_redacted);
		free(response_buf);
		return MP_LDAP_ERROR;
	} else if (!usergroups || !*usergroups) {
		mplog_debug("ldapsearch call for TGGROUPS returned no result, command was %s, output was %s", command_redacted, response_buf);
		add_error_to_list("ldapsearch call for TGGROUPS returned no result, command was %s, output was %s", command_redacted, response_buf);
		free(command_redacted);
		free(response_buf);
		return MP_LDAP_ERROR;
	} else {
		usergroups[TGGROUPS_BUF_SIZE - 1] = 0;
		mplog_debug("%s: Succeeded ldapsearching for TGGROUPS for user account '%s'!", __func__, username);
		mplog_debug("%s: Result:", __func__);
		mplog_debug("%s: >> TGGROUPS: %s", __func__, usergroups);
		free(command_redacted);
		free(response_buf);
		return MP_LDAP_OK;
	}
}

static int get_tg_groups_from_openldap_new(const char* username, char** usergroups_ptr, size_t* usergroups_size_ptr)
{
	init_error_list();

	(*usergroups_ptr)[0] = '\0';

	char * command = (char *) malloc(BUFFER_SIZE);
	if (!command) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		return MP_LDAP_ERROR;
	}

	char * command_redacted = (char *) malloc(BUFFER_SIZE);
	if (!command_redacted) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		free(command);
		return MP_LDAP_ERROR;
	}

	char* ldap_protocol = get_skel_value("LDAPPROTO1");
	if (!ldap_protocol) {
		free(command);
		free(command_redacted);
		mplog_error("%s: failed to read ldap_protocol. Aborting...", __func__);
		return MP_LDAP_ERROR;
	}
	char* ldap_server = get_skel_value("LDAPSERVER1");
	if (!ldap_server) {
		free(command);
		free(command_redacted);
		free(ldap_protocol);
		mplog_error("%s: failed to read ldap_server. Aborting...", __func__);
		return MP_LDAP_ERROR;
	}
	char* ldap_bind_dn_user = get_skel_value("LDAPBINDDNUSER");
	if (!ldap_bind_dn_user) {
		free(command);
		free(command_redacted);
		free(ldap_protocol);
		free(ldap_server);
		mplog_error("%s: failed to read ldap_bind_dn_user. Aborting...", __func__);
		return MP_LDAP_ERROR;
	}
	char* ldap_bind_dn_password = get_skel_value("LDAPBINDDNPASSWD");
	if (!ldap_bind_dn_password) {
		free(command);
		free(command_redacted);
		free(ldap_protocol);
		free(ldap_server);
		free(ldap_bind_dn_user);
		mplog_error("%s: failed to read ldap_bind_dn_password. Aborting...", __func__);
		return MP_LDAP_ERROR;
	}
	char* ldap_base = get_skel_value("LDAPBASE");
	if (ldap_base && !strcmp(ldap_base, "auto")) {
		mplog_debug("%s: ldap base is set to 'auto'. Trying to figure it out automatically...", __func__);
		free(ldap_base);
		ldap_base = get_ldap_base_from_openldap();
	}
	if (!ldap_base) {
		free(command);
		free(command_redacted);
		free(ldap_protocol);
		free(ldap_server);
		free(ldap_bind_dn_user);
		free(ldap_bind_dn_password);
		mplog_error("%s: failed to read or figure out ldap_base automatically. Aborting...", __func__);
		return MP_LDAP_ERROR;
	}
	mplog_debug("%s: ldap base is <%s>", __func__, ldap_base);

	// First successfull attempt: "/usr/bin/timeout -k 10 60 /usr/bin/ldapsearch -x -LLL -b \"%s\" -H %s//%s -D \"%s\" -w \"%s\" -b \"uid=%s,ou=people,dc=m-privacy,dc=dev\" dn memberOf 2>&1",
	// With a filter, it should work too: "/usr/bin/timeout -k 10 60 /usr/bin/ldapsearch -x -LLL -b \"%s\" -H %s//%s -D \"%s\" -w \"%s\"  \"(uid=%s)\" dn memberOf 2>&1",
	// In the client's OpenLDAP, they don't use uid but cn: "/usr/bin/timeout -k 10 60 /usr/bin/ldapsearch -x -LLL -b \"%s\" -H %s//%s -D \"%s\" -w \"%s\"  \"(cn=%s)\" dn memberOf 2>&1",
	snprintf(command, BUFFER_SIZE - 1,
			 "/usr/bin/timeout -k 10 60 /usr/bin/ldapsearch -x -LLL -b \"%s\" -H %s//%s -D \"%s\" -w \"%s\" \"(|(cn=%s)(uid=%s)(sAMAccountName=%s))\" dn memberOf 2>&1",
			 ldap_base,
			 ldap_protocol,
			 ldap_server,
			 ldap_bind_dn_user,
			 ldap_bind_dn_password,
			 username,
			 username,
			 username);
	command[BUFFER_SIZE - 1] = 0;

	snprintf(command_redacted, BUFFER_SIZE - 1,
			 "/usr/bin/timeout -k 10 60 /usr/bin/ldapsearch -x -LLL -b \"%s\" -H %s//%s -D \"%s\" -w \"*******\" \"(|(cn=%s)(uid=%s)(sAMAccountName=%s))\" dn memberOf 2>&1",
			 ldap_base,
			 ldap_protocol,
			 ldap_server,
			 ldap_bind_dn_user,
			 username,
			 username,
			 username);
	command_redacted[BUFFER_SIZE - 1] = 0;

	mplog_debug("%s: running ldapsearch for TGGROUPS: %s", __func__, command_redacted);
	FILE* ppipe = popen(command, "r");
	if (!ppipe) {
		mplog_error("%s: Failed to execute ldapsearch for TGGROUPS!", __func__);
		syslog(LOG_ERR, "%s: Failed to execute ldapsearch for TGGROUPS!", __func__);
		free(command);
		free(command_redacted);
		return MP_LDAP_ERROR;
	}
	free(command);
	free(ldap_base);
	free(ldap_protocol);
	free(ldap_server);
	free(ldap_bind_dn_user);
	free(ldap_bind_dn_password);

	char* usergroup_pos = NULL;

	char * usergroup_pos_decoded = (char *) malloc(BUFFER_SIZE);
	if (!usergroup_pos_decoded) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		free(command_redacted);
		return MP_LDAP_ERROR;
	}

	char * response_buf = (char *) malloc(BUFFER_SIZE);
	size_t response_size = BUFFER_SIZE;
	if (!response_buf) {
		syslog(LOG_ERR, "%s: failed to alloc memory", __func__);
		free(command_redacted);
		free(usergroup_pos_decoded);
		return MP_LDAP_ERROR;
	}
	response_buf[0] = 0;

	unsigned int offset = 0;

	char line[BUFFER_SIZE];
	unsigned int is_response_empty = 1; // A completely empty response means there is no such user in the LDAP server and we should allow password and/or certificate logins
	while (fgets(line, BUFFER_SIZE - 1, ppipe)) {
		is_response_empty = 0; // The first fgets will return null if the response is completely empty
		void* realloc_result = realloc_buffer(&response_buf, &response_size, offset, strlen(line));
		if (!realloc_result) {
			mplog_error("%s: Error: Unable to reallocate memory! Aborting dynamic resizing of buffer...", __func__);
			break;
		}

		int bytes_read = snprintf(response_buf + offset, BUFFER_SIZE - 1 - offset, "%s", line);
		if (bytes_read > 0)
			offset += bytes_read;

		trim_str(line);
		mplog_verbose("%s: response line:\n%s", __func__, line);
		usergroup_pos = strcasestr(line, "memberOf:: ");
		if (usergroup_pos && *usergroup_pos) { /* base64? */
			usergroup_pos += strlen("memberOf:: ");
			mplog_verbose("Found base64-encoded tggroup in reponse with value: %s. Decoding...", usergroup_pos);
			int decode_succeeded = base64_decode_no_alloc(usergroup_pos, &usergroup_pos_decoded);
			if (decode_succeeded) {
				usergroup_pos = usergroup_pos_decoded;
			}
		} else {
			usergroup_pos = strcasestr(line, "memberOf: ");
			if (usergroup_pos && *usergroup_pos) {
				usergroup_pos += strlen("memberOf: ");
				mplog_verbose("%s: found tggroup in reponse with value: %s", __func__, usergroup_pos);
			}
		}

		if (usergroup_pos && *usergroup_pos) {
			char* usergroup = parse_usergroup(usergroup_pos);
			if (usergroup) {
				mplog_verbose("%s: parsed tggroup in reponse: %s", __func__, usergroup);
				void* realloc_result = realloc_buffer(usergroups_ptr, usergroups_size_ptr, strlen((*usergroups_ptr)) + 1, strlen(usergroup) + 1); // first +1 for '\0', second +1 for " "
				if (!realloc_result) {
					mplog_error("%s: Error: Unable to reallocate memory! Aborting dynamic resizing of buffer...", __func__);
					break;
				}
				strncat((*usergroups_ptr), usergroup, (*usergroups_size_ptr) - 2 - strlen((*usergroups_ptr)));
				strcat((*usergroups_ptr), " ");

				free(usergroup);
			}
		}
	}

	free(usergroup_pos_decoded);
	response_buf[response_size - 1] = 0;

	int err = pclose(ppipe) >> 8;
	if (err) {
		mplog_error("ldapsearch call for TGGROUPS failed with error %d, command was %s, output was %s", err, command_redacted, response_buf);
		syslog(LOG_ERR, "ldapsearch call for TGGROUPS failed with error %d, command was %s, output was %s", err, command_redacted, response_buf);
		add_error_to_list("ldapsearch call for TGGROUPS failed with error %u, command was %s, output was %s", err, command_redacted, response_buf);
		free(command_redacted);
		free(response_buf);
		return MP_LDAP_ERROR;
	} else if (is_response_empty) {
		syslog(LOG_ERR, "The response was completely empty (the user is not in the LDAP server). Continuing with password normally (also not applying any TG groups settings)");
		return MP_LDAP_EMPTY_RESPONSE_ERROR; // The response was empty. User not in LDAP server (still password and certificate should be possible!)
	} else if (!(*usergroups_ptr) || !*(*usergroups_ptr)) {
		mplog_debug("ldapsearch call for TGGROUPS returned no result, command was %s, output was %s", command_redacted, response_buf);
		add_error_to_list("ldapsearch call for TGGROUPS returned no result, command was %s, output was %s", command_redacted, response_buf);
		free(command_redacted);
		free(response_buf);
		return MP_LDAP_NO_GROUPS_ERROR; // The user is in the LDAP server but belongs to no TG groups!
	} else {
		(*usergroups_ptr)[(*usergroups_size_ptr) - 1] = '\0';
		mplog_debug("%s: Succeeded ldapsearching for TGGROUPS for user account '%s'!", __func__, username);
		mplog_debug("%s: Result:", __func__);
		mplog_debug("%s: >> TGGROUPS: %s", __func__, (*usergroups_ptr));
		free(command_redacted);
		free(response_buf);
		return MP_LDAP_OK;
	}
}

/*
 * This function is retrieves the TG groups for a given username for PAM auth (user wants to
 * authenticate with username and password) if authentication method is set to 'LDAP' or 'AD-OpenLDAP'
 * on the server. It takes these parameters:
 *   - username: The original username to fetch groups for (before modifying stuff like everything to lowercase
 *     and #-ending if it's a numeric user).
 *
 * Doesn't return anything, just sets TGGROUPS environment variable to "empty" or the list of
 * tg groups or doesn't set it at all if group-based authorization isn't set.
 */
static void get_tg_groups_and_set_env_if_necessary(const char* username)
{
	char* authmethod = get_skel_value("AUTHMETHOD");

	syslog(LOG_INFO, "Authentication method is %s and the user connected using username and password or a certificate", authmethod);
	mplog_debug("%s: Authentication method is %s and the user connected using username and password or a certificate", __func__, authmethod);
	if (!strcmp(authmethod, "LDAP") || !strcmp(authmethod, "AD-OpenLDAP")) {
		char* ldap_check_tg_groups = get_skel_value("LDAPCHECKTGGROUPS");
		if (!strcmp(ldap_check_tg_groups, "yes") || !strcmp(authmethod, "AD-OpenLDAP")) { // AD-OpenLDAP only makes sense to read the groups
			mplog_debug("%s: Trying to get TG groups with binddn and password...", __func__);
			portable_setenv("TGGROUPS", "empty");
			char* usergroups = (char*) malloc(BUFFER_SIZE);
			size_t usergroups_size = BUFFER_SIZE;
			if (!usergroups) {
				mplog_error("%s: Failed to allocate memory to store TG groups!", __func__);
			} else {
				int rc = get_tg_groups_from_openldap_new(username, &usergroups, &usergroups_size);
				if (rc == MP_LDAP_OK) {
					if (*usergroups) {
						portable_setenv("TGGROUPS", usergroups);
					} else {
						portable_setenv("TGGROUPS", "empty");
					}
				} else if (rc == MP_LDAP_EMPTY_RESPONSE_ERROR) {
					syslog(LOG_INFO, "LDAP response was empty (user not in LDAP server). Continuing with password or certificate auth");
					mplog_debug("%s: LDAP response was empty (user not in LDAP server). Continuing with password or certificate auth", __func__);
					portable_unsetenv("TGGROUPS"); // make sure that it's not "empty"
				} else if (rc == MP_LDAP_QUERY_EXEC_ERROR) {
					mplog_error("Failed to execute LDAP query. Assuming empty list of TG groups");
					mplog_debug("%s: Failed to execute LDAP query. Assuming empty list of TG groups", __func__);
					portable_setenv("TGGROUPS", "empty");
				} else if (rc == MP_LDAP_NO_GROUPS_ERROR) {
					syslog(LOG_INFO, "LDAP query returned no groups. Assuming empty list of TG groups");
					mplog_debug("%s: LDAP query returned no groups. Assuming empty list of TG groups", __func__);
					portable_setenv("TGGROUPS", "empty");
				}
			}
		} else {
			syslog(LOG_INFO, "NOT trying to get TG groups from LDAP server... (LDAPCHECKTGGROUPS=\"no\")");
			mplog_debug("%s: NOT trying to get TG groups from LDAP server... (LDAPCHECKTGGROUPS=\"no\")", __func__);
		}
	} else if (!strcmp(authmethod, "AD")) {
		char* ad_check_tg_groups = get_skel_value("ADCHECKTGGROUPS");
		if (!strcmp(ad_check_tg_groups, "no")) {
			syslog(LOG_INFO, "NOT trying to get TG groups with kerberos ticket... (ADCHECKTGGROUPS=\"no\")");
			mplog_debug("%s: NOT trying to get TG groups with kerberos ticket... (ADCHECKTGGROUPS=\"no\")", __func__);
		} else {
			mplog_debug("%s: Authentication method is AD and the user connected using username and password or a certificate. Trying to get TG groups with kerberos ticket...", __func__);
			if (run_kinit()) {
				mplog_debug("%s: run_kinit() successful, try to get TGGROUPS", __func__);
				char* domain = get_domain(NULL, "/etc/cu/krbrealmdomains");
				if (domain) {
					char* ldap_base = NULL;
					generate_ldap_base(&ldap_base, domain);
					if (ldap_base) {
						perform_ldap_searches(domain, NULL, ldap_base, username, NULL); // This setenvs TGGROUPS appropriately
						free(ldap_base);
						portable_setenv("TGPAMKRB", username);
					} else {
						syslog(LOG_DEBUG, "%s: no LDAP base found for krb domain %s", __func__, domain);
					}
					free(domain);
				} else {
					mplog_syslog_debug("%s: no krb domain found", __func__);
				}
			} else {
				mplog_syslog_debug("%s: no krb domain found", __func__);
			}
		}
	}
}

#undef strcasestr strcasestrugly

#endif /* #if !defined(WIN32) && !defined(__APPLE__) */
