#ifdef __cplusplus
extern "C" {
#endif

#ifndef MP_UTILS_H
#define MP_UTILS_H

/**
 * Returns true if string starts with beginning
 * Inspired by https://stackoverflow.com/questions/15515088/how-to-check-if-string-starts-with-certain-string-in-c
 */
int starts_with(const char* string, const char* beginning);

/**
 * Returns true if string starts ends with ending
 * Inspired by https://stackoverflow.com/questions/10347689/how-can-i-check-whether-a-string-ends-with-csv-in-c
 */
int ends_with(const char* string, const char* ending);

const char* strstr_insensitive(const char *haystack, const char *needle);

/**
 * Inspired by http://c-programming-blog.blogspot.de/2015/05/pcre2-simple-sample-program.html
 *
 * Returns only the first match and NULL if no match was found.
 * Remeber to free the returned match!
 */
char* match(const char* string, const char* regex, const int group);

/**
 * Concatenates to strings and returns the result.
 *
 * Returns the concatenated string or NULL if there was an error allocation the memory.
 * Remeber to free the returned string!
 *
 * Inspired (copied might be more suitable here) by https://stackoverflow.com/questions/8465006/how-to-concatenate-2-strings-in-c
 */
char* concat(char* s1, char* s2);

/**
 * Returns a random string (just numbers and small letters!) or NULL if there was an error allocation the memory or length was NULL
 * Remeber to free the returned string!
 *
 * Inspired (copied might be more suitable here) by https://codereview.stackexchange.com/questions/29198/random-string-generator-in-c
 */
char* randomstring(size_t length);

/**
 * Replace all occurences of a string in another string
 *
 * From:
 * https://stackoverflow.com/questions/779875/what-is-the-function-to-replace-string-in-c
 * Thank you!
 *
 * Don't forget to free the returned char*
 */
char* str_replace(char* orig, char* rep, char* with);

/**
 * Removes the surrounding double quotes for a string. If it starts
 * and ends with a double quote.
 *
 * It will change the input char* so please beware. If the input was
 * allocated, it should still be safe to free afterwards.
 *
 * Make sure input is a null-terminated char*.
 */
void remove_quotes(char* input);

/**
 * Replace all occurences of a invalid chars for Windows paths
 * (<>:\"/\\|?*)
 *
 * Don't forget to free the returned char* (although it might
 * return NULL if the input is NULL or for linux)
 */
char* replace_invalid_windows_path_chars(const char* input);

unsigned int is_valid_utf8(const char* s);

#if defined(WIN32) || defined(WIN64)
/**
 * Transform a normal C/C++ string (char*) into a wchar_t* for Windows.
 * Mostly for _w functions like _wfopen (to write filenames with non-ASCII chars)
 *
 * Don't forget to free the returned wchar_t*
 */
wchar_t* utf8_char_to_wchar_t(const char* string);

/**
 * Does exactly the same as 'utf8_char_to_wchar_t'. The only
 * difference is, its name is nicer :)
 *
 * Don't forget to free the returned wchar_t*
 */
wchar_t* convert_utf8_char_to_utf16_wchar_t(const char* string);

/**
 * Transforms a Windows UTF16 wchar_t into a normal C/C++ UTF8 string (char*)
 *
 * Don't forget to free the returned wchar_t*
 */
char* convert_utf16_wchar_to_utf8_char(const wchar_t* wstring);

#endif /* defined(WIN32) || defined(WIN64) */

/**
 * Delete a file in Windows and Linux.
 *
 * No need to free anything after it.
 */
void delete_file(char* file);

/**
 * Move or rename a file in Windows and Linux with the
 * possibility to overwrite.
 *
 * No need to free anything after it.
 */
void mp_move(char* source, char* destination, unsigned int overwrite);

/**
 * Copy file in Windows and Linux with the
 * possibility to overwrite.
 *
 * No need to free anything after it.
 */
void mp_copy(char* source, char* destination, unsigned int overwrite);

/**
 * Resolve DNS name. Saves the ip address as string to the given ip_address array.
 *
 * Example:
 * 	char ip_address[16];
 * 	ip_address[0] = 0;
 * 	get_host_by_name("internet.intern.netz", &ip_address);
 */
unsigned int get_host_by_name(const char* hostname, char (*ip_address_ptr)[16]);

/**
 * Check if the dns resolution of the server is 255.255.255.255
 * which is the answer we get from a cluster with no available nodes.
 */
unsigned int is_cluster_node_available(const char* hostname);

/**
 * Read specific field from a CSV line.
 *
 * Returns NULL if there are not enough fields in the given line.
 *
 * It allocates memory for the returned string. You might want to free it
 * after using it.
 */
char* get_field(const char* line, char delimiter, const unsigned int field);

/**
 * Return descriptions of prepareuser error codes,
 * Returned strings are static, do not free()
 */

char * prepareuser_error(int errorcode);

#endif /* MP_UTILS_H */

#ifdef __cplusplus
}
#endif
