#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>
#include <ctype.h>
#include <wchar.h>
#if defined(WIN32) || defined(WIN64)
#include <windows.h>
#include <tchar.h>
#include <psapi.h>
#include <direct.h>
#include <lmcons.h>
#include <tlhelp32.h>
#include <shlobj.h>
#include <shobjidl.h>
#else
#include "pwd.h"
#endif
#include <sys/stat.h>
#include "log.h"
#include "tgpro_environment.h"
#include "mp_utils.h"

// #if defined(WIN32) || defined(WIN64)
// #define errno GetLastError()
//#else
#include <errno.h>
//#endif

wchar_t* error_box_title = NULL;
char* appdata_path = NULL;
char* tgpro_path = NULL;
char* u_tmp_dir = NULL;
char* winscp_exe_path = NULL;
char* winscp_com_path = NULL;
char* browserchoice_exe_path = NULL;
char* schleuse_exe_path = NULL;
char* putty_exe_path = NULL;
char* tgpro_cfg_path = NULL;
char* tgpro_vnc_path = NULL;
char* sumatrapdf_exe_path = NULL;
char* krb_host_name = NULL;
char* host_name = NULL;
char* sectypes = NULL;
char* putty_settings_file = NULL;
char* putty_cfg_path = NULL;
char* url_whitelist_path_system = NULL;
char* url_whitelist_path_cwd = NULL;
char* start_path = NULL;
char* user_name = NULL;
int quick_print = 0;
#if defined(WIN32) || defined(WIN64)
HWND viewer_main_window_handle = NULL;
#endif /* defined(WIN32) || defined(WIN64) */
char* vncviewer_exe_path = NULL;
char* browserchoice_cfg_path = NULL;
char* tmp_magicurl_pipe_name_file_path = NULL;
char* viewer_log_file_path = NULL;
char* viewer_trace_file_path = NULL;
char* viewer_instream_trace_file_path = NULL;
char* viewer_outstream_trace_file_path = NULL;

FILE* log_fp = NULL;

size_t copy_start_path(char* start_path_array, size_t max_len) {
	size_t path_len;
#if defined(WIN32) || defined(WIN64)
	path_len = GetModuleFileName(NULL, start_path_array, max_len -1);
#else
	char sz_tmp[32];
	snprintf(sz_tmp, 31, "/proc/%d/exe", getpid());
	const size_t rl_len = readlink(sz_tmp, start_path_array, max_len);
	path_len = rl_len < max_len -1 ? rl_len : max_len - 1;
	if (path_len >= 0)
		start_path_array[path_len] = '\0';
#endif /* defined(WIN32) || defined(WIN64) */
	if (path_len > 0) {
#if defined(WIN32) || defined(WIN64)
		static const char seperator = '\\';
#else
		static const char seperator = '/';
#endif /* defined(WIN32) || defined(WIN64) */
		char* rev_search = start_path_array + strlen(start_path_array);
		while (rev_search > start_path_array && *rev_search != seperator)
			--rev_search;
		if (*rev_search == seperator)
			*++rev_search = '\0';
	}
	return path_len;
}


const char* get_start_path() {
	if (start_path)
		return start_path;
	char start_path_arr[8192];
	size_t spa_len = copy_start_path(start_path_arr, 8192);
	error_check(spa_len < 0, L"get_start_path(): Fehler beim Suchen des Startverzeichnisses.");
	return (start_path = strdup(start_path_arr));
}


void set_quick_print(int new_qp) {
	quick_print = new_qp;
}


/**
 * Sets the title for the window of fatal error messages. Should be something
 * about the application's name and “error”.
 */
void set_error_box_title(wchar_t* new_error_box_title) {
	error_check(!new_error_box_title, L"Fehler: error_box_title sollte zu NULL gesetzt werden.");
	// printf("realloc for error_box_title mit Laenge: %zu\n", wcslen(new_error_box_title) * sizeof(wchar_t) + 2);
	error_box_title = realloc(error_box_title, wcslen(new_error_box_title) * sizeof(wchar_t) + 2);
	// printf("kopiere jetzt hinein: %ls\n", new_error_box_title);
	wcscpy(error_box_title, new_error_box_title);
}


/**
 * If error_case is non-zero this will display error_text and exit
 * the program. Should only be used for fatal errors.
 */
void error_check(const int error_case, const wchar_t* error_text) {
	if (!error_case)
		return;
	if (!error_box_title)
		error_box_title = L"TightGate-Pro Error";

	int dw = errno;

#if defined(WIN32) || defined(WIN64)
	char* lpMsgBuf;

	FormatMessageW(FORMAT_MESSAGE_ALLOCATE_BUFFER |
		       FORMAT_MESSAGE_FROM_SYSTEM |
		       FORMAT_MESSAGE_IGNORE_INSERTS,
		       NULL,
		       dw,
		       MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
		       (LPWSTR) &lpMsgBuf,
		       0, NULL);

	if (dw) {
		char* lpDisplayBuf = (LPVOID)LocalAlloc(LMEM_ZEROINIT,
							 (wcslen((LPCWSTR)lpMsgBuf) + wcslen((LPCWSTR)error_text) + 40 + 1) * sizeof(WCHAR));
		snwprintf((LPWSTR)lpDisplayBuf,
			  LocalSize(lpDisplayBuf) / sizeof(WCHAR),
			  L"%ls\n\nSystemfehlercode %d: %ls",
			  error_text, (int)dw, (wchar_t*)lpMsgBuf);
		MessageBoxW(NULL, (LPCWSTR)lpDisplayBuf, error_box_title, MB_OK);
		LocalFree(lpDisplayBuf);
	} else
		MessageBoxW(NULL, error_text, error_box_title, MB_OK);

	LocalFree(lpMsgBuf);
#else
	log_error("Scherwiegender Fehler: %ls", error_text);
#endif /* defined(WIN32) || defined(WIN64) */
	exit(dw ? dw : error_case);
}

/**
 * Checks for dir existence.
 */
int dir_exists(const char *path) {
	struct stat stats;
	stat(path, &stats);
	if (S_ISDIR(stats.st_mode)) {
		return 1;
	} else {
		return 0;
	}
}

#ifndef WIN32
int symlink_exists(const char* path) {
    struct stat buf;
    return lstat(path, &buf) == 0 ? 1 : 0;
}
#endif /* WIN32 */

/**
 * Checks for file existence.
 */
int file_exists(const char* filename) {
	if (!filename)
		return 0;
	char* fn = strdup(filename);
#if 0
/* This used to remove all the back slashes at the beginning,
   which lead to non-working UNC paths (windows share \\Server\Share\shared_file).
   I don't really now why this code was here so I'm just leaving it 'undefined'
   in case we need it in the future
*/
	while (*fn != '\0' && *fn == '\\')
		++fn;
#endif /* 0 */

	int exists = 0;
	FILE* file;
#if defined(WIN32) || defined(WIN64)
	wchar_t* wfn = utf8_char_to_wchar_t(fn);
	if ((file = _wfopen(wfn, L"r"))) {
		fclose(file);
		exists = 1;
	}
	free(wfn);
#else
	if ((file = fopen(fn, "r"))) {
		fclose(file);
		exists = 1;
	}
#endif /* defined(WIN32) || defined(WIN64) */
	free(fn);
	return exists;
}


/**
 * Remove the beginning and ending blanks of string text_line.
 * Returns a pointer to the result.
 * Warning: changes the original given string text_line and returns
 * a pointer to a position somewhere inside.
 * Replaces line end signs by string terminators.
 */
char* str_trim(char* text_line) {
	char* cr_killer = text_line;
	while (*cr_killer != '\0') {
		if (*cr_killer == '\r' || *cr_killer == '\n') {
			*cr_killer = '\0';
			break;
		}
		++cr_killer;
	}
	while (isspace(*text_line))
		++text_line;
	if (*text_line == '\0')
		return text_line;
	char* text_line_end = text_line + strlen(text_line) - 1;
	while (text_line_end > text_line && isspace(*text_line_end))
		--text_line_end;
	*(text_line_end + 1) = '\0';
	return text_line;
}

/**
 * Skips the UTF-8 Byte Order Mark (BOM) at the beginning of a line.
 *
 * If a BOM is found, returns a pointer to the position after the BOM;
 * otherwise, returns the original input pointer.
 *
 * It might "break" your original char.
 */
char* skip_bom(char* line) {
    // UTF-8 BOM is 0xEF 0xBB 0xBF
    if ((unsigned char) line[0] == 0xEF &&
        (unsigned char) line[1] == 0xBB &&
        (unsigned char) line[2] == 0xBF) {
        return line + 3; // Skip BOM by moving 3 bytes further
    }
    return line; // No BOM, return the original pointer
}

/**
 * Checks if the given file_name's final 3 characters are pdf.
 */
int check_pdf(const char* file_name) {
	size_t fn_length = strlen(file_name);
	if (fn_length < 4)
		return -1;
	return strcmp(file_name + fn_length - 4, ".pdf");
}

void kill_process(int pid) {
#if defined(WIN32) || defined(WIN64)
	log_debug("Killing process with pid %u", pid);
	UINT ecode = 0;
	HANDLE process = OpenProcess(SYNCHRONIZE | PROCESS_TERMINATE, FALSE, pid);
	TerminateProcess(process, ecode);
	CloseHandle(process);
#endif
}

/**
 * Runs a child program. If wait_for_child is non-zero this function waits
 * until the child program finishes. After the child program's start a
 * window called window_name is raised to focus. If window_name is NULL
 * nothing is raised.
 */
int launch_program(char* prog_cmd, const int show_window, const int bring_to_front, const int wait_for_child, const char* working_dir) {
#if defined(WIN32) || defined(WIN64)
	STARTUPINFO si;
	ZeroMemory(&si, sizeof(si));
	si.cb = sizeof(si);
	si.dwFlags = STARTF_USESTDHANDLES | STARTF_USESHOWWINDOW;
	si.hStdInput = GetStdHandle(STD_INPUT_HANDLE);
	si.hStdOutput =  GetStdHandle(STD_OUTPUT_HANDLE);
	si.hStdError = GetStdHandle(STD_ERROR_HANDLE);
	si.wShowWindow = show_window? SW_SHOWDEFAULT : 0;

	PROCESS_INFORMATION pi;
	ZeroMemory(&pi, sizeof(pi));

	if (!CreateProcess(NULL,             // No module name
			   prog_cmd,
			   NULL,             // Process handle not inheritable
			   NULL,             // Thread handle not inheritable
			   FALSE,            // Set handle inheritance to FALSE
			   0, //CREATE_NO_WINDOW, // creation flags
			   NULL,             // Use parent's environment block
			   working_dir ? working_dir : get_tgpro_path(), // working_dir
			   &si,              // Pointer to STARTUPINFO structure
			   &pi)) {           // Pointer to PROCESS_INFORMATION structure
		error_check(1, L"Fataler Fehler! Der Start eines notwendigen Programms ist fehlgeschlagen.");
	}

	log_trace("Wait for child program to open a window... ");
	WaitForInputIdle(pi.hProcess, INFINITE); /* TODO: don't wait infinite time. */
	log_trace("OK, looks ready.");

	if (bring_to_front) {
		log_trace("Trying to move window to foreground");
		bring_window_to_front((LPARAM) pi.dwProcessId);
	}

	/* Wait until child process exits. */
	if (wait_for_child) {
		WaitForSingleObject(pi.hProcess, INFINITE);
		log_trace("Child program has just left the building");
	}

	/* Close process and thread handles. */
	CloseHandle(pi.hProcess);
	CloseHandle(pi.hThread);

	return pi.dwProcessId;
#else
	return 0;
#endif /* defined(WIN32) || defined(WIN64) */
}


char* file_exists_in_path(const char* path_1, const char* path_2) {
	if (!path_1 || !path_2)
		return NULL;
	size_t path_len = strlen(path_1) + strlen(path_2) + 2;
	char* full_path = malloc(path_len);
	if (strlen(path_1))
		snprintf(full_path, path_len, "%s\\%s", path_1, path_2);
	else
		snprintf(full_path, path_len, "%s", path_2);
	log_trace("Looking for file <%s> -- ", full_path);
	if (!file_exists(full_path)) {
		log_trace("it is not there!");
		free(full_path);
		full_path = NULL;
	} else
		log_trace("THERE IT IS!");
	return full_path;
}


/**
 * Returns a pointer to a string containing the path to the SumatraPDF.exe file.
 * After you have called this function you should exit_tgpro_environment()
 * to free the memory.
 */
const char* get_sumatrapdf_exe_path() {
	if (sumatrapdf_exe_path)
		return sumatrapdf_exe_path;

	if ((sumatrapdf_exe_path = file_exists_in_path(get_start_path(), SUMATRAPDF_EXE)))
		return sumatrapdf_exe_path;
	if ((sumatrapdf_exe_path = file_exists_in_path(get_start_path(), SUMATRAPDF_EXE_SUBDIR)))
		return sumatrapdf_exe_path;
	if ((sumatrapdf_exe_path = file_exists_in_path(get_tgpro_path(), SUMATRAPDF_EXE)))
		return sumatrapdf_exe_path;
	if ((sumatrapdf_exe_path = file_exists_in_path(get_tgpro_path(), SUMATRAPDF_EXE_SUBDIR)))
		return sumatrapdf_exe_path;
	if (file_exists(SUMATRAPDF_EXE_SUBDIR))
		return (sumatrapdf_exe_path = strdup(SUMATRAPDF_EXE_SUBDIR));
	if (file_exists(SUMATRAPDF_EXE))
		return (sumatrapdf_exe_path = strdup(SUMATRAPDF_EXE));

	return NULL;
}

#if defined(WIN32) || defined(WIN64)
static int showed_no_sumatra_error = 0;
#endif
/**
 * Tells the system to print the given pdf file.
 */
void print_pdf(const char* pdf_file) {
#if defined(WIN32) || defined(WIN64)
	if (!get_sumatrapdf_exe_path()) {
		if (!showed_no_sumatra_error)
			MessageBoxW(NULL, L"Drucken ist leider nicht möglich, da ich das\nDruckprogramm SumatraPDF nicht finden kann.", L"Druckspooler-Fehler", MB_OK);
		showed_no_sumatra_error = 1;
		return;
	}

	const char* params = quick_print ? SUMATRA_QUICK_PRINT_PARAMS : SUMATRA_PARAMS;
	const size_t cmd_len = strlen(get_sumatrapdf_exe_path()) + strlen(params) + strlen(pdf_file) + 5;
	char pdf_printer_cmd[cmd_len];
	snprintf(pdf_printer_cmd, cmd_len, "%s %s \"%s\"", get_sumatrapdf_exe_path(), params, pdf_file);
	log_trace("print cmd: %s", pdf_printer_cmd);

	launch_program(pdf_printer_cmd, 1, 1, 1, NULL);
#else

#endif /* defined(WIN32) || defined(WIN64) */
}

/**
 * Print the given pdf file using Adobe Acrobat Reader.
 */
int adobe_print_pdf(const char* pdf_file) {
#if defined(WIN32) || defined(WIN64)
	char* adobe_reader_exe_path = "C:\\Program Files\\Adobe\\Acrobat DC\\Acrobat\\Acrobat.exe";
	if (!file_exists(adobe_reader_exe_path)) {
		adobe_reader_exe_path = "C:\\Program Files (x86)\\Adobe\\Acrobat Reader 2020\\Reader\\AcroRd32.exe";
	}
	if (!file_exists(adobe_reader_exe_path)) {
		adobe_reader_exe_path = "C:\\Program Files (x86)\\Adobe\\Acrobat DC\\Acrobat\\Acrobat.exe";
	}
	if (!file_exists(adobe_reader_exe_path)) {
		log_error("%s: couldn't find Adobe Acrobat Reader exe file. Looked here <C:\\Program Files\\Adobe\\Acrobat DC\\Acrobat\\Acrobat.exe>, here <C:\\Program Files (x86)\\Adobe\\Acrobat Reader 2020\\Reader\\AcroRd32.exe> and here <C:\\Program Files (x86)\\Adobe\\Acrobat DC\\Acrobat\\Acrobat.exe> ", __func__);
		return 0;
	}
	const char* params = "/p";
	const size_t cmd_len = strlen(adobe_reader_exe_path) + strlen(params) + strlen(pdf_file) + 9;
	char pdf_printer_cmd[cmd_len];
	snprintf(pdf_printer_cmd, cmd_len - 1, "\"%s\" %s \"%s\"", adobe_reader_exe_path, params, pdf_file);
	log_debug("Trying to start Adobe Acrobat Reader with following command: %s", pdf_printer_cmd);

	int process_id = launch_program(pdf_printer_cmd, 0, 0, 1, NULL);
	if (!process_id) {
		log_error("%s: failed to launch Adobe Acrobat Reader print dialog", __func__);
		return 0;
	}
	return 1;
#else
	log_info("%s: can't print using Adobe Acrobat Reader print dialog on Linux", __func__);
	return 0;
#endif /* defined(WIN32) || defined(WIN64) */
}


/**
 * Returns a pointer to the the string right of the equals sign in
 * text_line. Returns NULL if there is no equals sign in text_line,
 * otherwise a pointer to the right-side value. Trims off the
 * right-side value's leading and final blanks.
 */
char* get_right_value(char* text_line) {
	text_line = strchr(text_line, '=');
	if (!text_line)
		return NULL;
	++text_line;
	while (*text_line == ' ')
		++text_line;
	char* final_letter = text_line - 1;
	char* it;
	for (it = text_line; *it != '\0' && *it != '\n' && *it != '\r'; it++)
		if (*it != ' ')
			final_letter = it;
	++final_letter;
	*final_letter = '\0';
	return text_line;
}


/**
 * Returns a pointer to the string left of the equals sign in
 * text_line. Returns NULL if there is no equals sign in text_line,
 * otherwise a pointer to the left-side value. Trims off the left-side
 * value's leading and final blanks. Changes the string text_line
 * (replaces the equals sign with the string termination symbol 0).
 */
char* get_left_value(char* text_line) {
	char* equals_sign = strchr(text_line, '=');
	if (!equals_sign)
		return NULL;
	while (equals_sign-1 >= text_line && *(equals_sign-1) == ' ')
		--equals_sign;
	*equals_sign = '\0';
	while (text_line < equals_sign && *text_line == ' ')
		++text_line;
	return text_line;
}


/**
 * Returns TRUE if this TG-Pro installation uses SSO / Active Directory
 * for authentication. Returns FALSE if it uses SSO without AD.
 */
int uses_sso_ad_authentication() {
	log_debug("Checking if it is an AD installation (SecurityTypes=*krb*)");
	sectypes = get_tgpro_cfg_value("SecurityTypes");
	if (!sectypes) {
		return 0;
	}
	int ad_installation = strstr_insensitive(sectypes, "krb") != 0;
	log_debug("uses AD: %d (SecurityTypes=%s)", ad_installation, sectypes);
	return ad_installation;
}

int uses_plain_authentication() { // tlsplain or x509plain
	log_debug("Checking if it uses a 'plain' authentication");
	sectypes = get_tgpro_cfg_value("SecurityTypes");
	if (!sectypes) {
		return 0;
	}
	int uses_plain_auth = strstr_insensitive(sectypes, "tlsplain") || strstr_insensitive(sectypes, "x509plain");
	log_debug("uses plain authentication: %d (SecurityTypes=%s)", uses_plain_auth, sectypes);
	return uses_plain_auth;
}

/**
 * Checks tgpro.cfg for line HostName=… and returns the value on the
 * right side of the equals sign.
 * Danger: Memory is allocated by this function. Remember to free it.
 * Returns NULL if there is no KrbHostName found in tgpro.cfg or if there is
 * no tgpro.cfg.
 */
char* get_host_name() {
	if (!host_name) {
		host_name = get_tgpro_cfg_value("ServerName");
	}
	if (!host_name) {
		host_name = get_tgpro_cfg_value("Host");
	}
	if (!host_name) {
		host_name = get_tgpro_cfg_value("KrbHostName");
	}
	return host_name;
}

/**
 * Checks tgpro.cfg for line KrbHostName=… and returns the value on the
 * right side of the equals sign.
 * Danger: Memory is allocated by this function. Remember to free it.
 * Returns NULL if there is no KrbHostName found in tgpro.cfg or if there is
 * no tgpro.cfg.
 */
char* get_krb_host_name() {
	if (!krb_host_name) {
		krb_host_name = get_tgpro_cfg_value("KrbHostName");
	}
	if (!krb_host_name) {
		krb_host_name = get_tgpro_cfg_value("ServerName");
	}
	if (!krb_host_name) {
		krb_host_name = get_tgpro_cfg_value("Host");
	}
	return krb_host_name;
}

/**
 * Returns a value from a key=value pair from config file.
 * It only works iwh files where key and value are separated by an '=' sign.
 *
 * Remember to free() the returned char*.
 * Returns NULL in if there is no key or no config file.
 */
char* get_cfg_value_from_file(const char* key, const char* cfg_file) {
	char* value = NULL;
	FILE* cfg_file_file;
	if ((cfg_file_file = fopen(cfg_file, "r"))) {
		log_trace("Looking for key '%s' in config file '%s'", key, cfg_file);
		char text_line[4096];
		while (!value && fgets(text_line, 4095, cfg_file_file)) {
			char* right_val = get_right_value(text_line);
			if (!right_val || (strlen(right_val) < 1)) {
				continue;
			}
			char* left_val = get_left_value(text_line);
			if (!left_val || (strlen(left_val) < 1)) {
				continue;
			}
			if (strcasecmp(left_val, key) == 0) {
				log_trace("I found the key '%s': '%s'", key, right_val);
				value = strdup(right_val);
			}
		}
		fclose(cfg_file_file);
	}

	if (!value) {
		log_warn("Couldn't find key '%s' in config file '%s'", key, cfg_file);
	}

	return value;
}

/**
 * Deprecated. Use get_cfg_value_from_file instead.
 * Returns a value from a key/value pair from tgpro config file.
 * Remember to free() the returned char*.
 * Returns NULL in if there is no key or no config file.
 */
char* get_tgpro_cfg_value_from_file(const char* key, const char* cfg_file) {
	return get_cfg_value_from_file(key, cfg_file);
}

/**
 * Returns a value from a key/value pair from tgpro config file.
 * Remember to free() the returned char*.
 * Returns NULL in if there is no key or no config file.
 */
char* get_tgpro_cfg_value(const char* key) {
	const char* cfg_file = get_tgpro_vnc_path();
	if (!cfg_file)
		cfg_file = get_tgpro_cfg_path();
	if (!cfg_file) {
		log_warn("Failed to get key %s (couldn't find a tgpro config file.", key);
		return NULL;
	}
	return get_tgpro_cfg_value_from_file(key, cfg_file);
}


/**
 * Returns a pointer to a string containing the path to the tgpro temp directory.
 * This is a subdirectory of the operating system's temp dir. Normally windows
 * gives us a per-user temp dir.
 * After you have called this function you should exit_tgpro_environment()
 * to free the memory.
 */
const char* get_tgpro_tmp_path() {
	if (u_tmp_dir)
		return u_tmp_dir;

	char tmp_tmp_dir[4096];
#if defined(WIN32) || defined(WIN64)
	GetTempPathA(4095 - strlen(TGPRO_TEMP_SUBDIR), tmp_tmp_dir);
	strcat(tmp_tmp_dir, TGPRO_TEMP_SUBDIR);
 	strcat(tmp_tmp_dir, "\\");
	if (mkdir(tmp_tmp_dir) && errno != EEXIST) {
		snprintf(tmp_tmp_dir, 4095, "%s\\%s\\", get_appdata_path(), TGPRO_TEMP_SUBDIR);
		if (mkdir(tmp_tmp_dir) && errno != EEXIST) {
			return NULL;
		}
	}

    // Now get the long path name (without the legacy ~ in path name)
    char long_path[4096];
    DWORD long_len = GetLongPathNameA(tmp_tmp_dir, long_path, sizeof(long_path));
    if (long_len == 0 || long_len >= sizeof(long_path)) {
        // Failed to resolve, fallback to original
        u_tmp_dir = strdup(tmp_tmp_dir);
    } else {
        u_tmp_dir = strdup(long_path);
    }

#else
	snprintf(tmp_tmp_dir, 4089 - strlen(TGPRO_TEMP_SUBDIR), "/tmp/%s/", TGPRO_TEMP_SUBDIR);
	if (mkdir(tmp_tmp_dir, 0700))
		error_check(errno != EEXIST, L"Die Erzeugung des temporären Verzeichnisses ist fehlgeschlagen.");
#endif /* defined(WIN32) || defined(WIN64) */
	u_tmp_dir = strdup(tmp_tmp_dir);

	return u_tmp_dir;
}


/**
 * Returns a pointer to a string containing the path to the WinSCP.exe file.
 * After you have called this function you should exit_tgpro_environment()
 * to free the memory.
 */
const char* get_winscp_exe_path() {
	if (winscp_exe_path)
		return winscp_exe_path;

	if ((winscp_exe_path = file_exists_in_path(get_start_path(), WINSCP_EXE)))
		return winscp_exe_path;
	if ((winscp_exe_path = file_exists_in_path(get_start_path(), WINSCP_EXE_SUBDIR)))
		return winscp_exe_path;
	if ((winscp_exe_path = file_exists_in_path(get_tgpro_path(), WINSCP_EXE)))
		return winscp_exe_path;
	if ((winscp_exe_path = file_exists_in_path(get_tgpro_path(), WINSCP_EXE_SUBDIR)))
		return winscp_exe_path;
	if (file_exists(WINSCP_EXE))
		return (winscp_exe_path = strdup(WINSCP_EXE));
	if (file_exists(WINSCP_EXE_SUBDIR))
		return (winscp_exe_path = strdup(WINSCP_EXE_SUBDIR));

	return NULL;
}


/**
 * Returns a pointer to a string containing the path to the Browserchoice.exe file.
 * After you have called this function you should exit_tgpro_environment()
 * to free the memory.
 */
const char* get_browserchoice_exe_path() {
	if (browserchoice_exe_path)
		return browserchoice_exe_path;

	if ((browserchoice_exe_path = file_exists_in_path(get_start_path(), BROWSERCHOICE_EXE)))
		return browserchoice_exe_path;
	if ((browserchoice_exe_path = file_exists_in_path(get_tgpro_path(), BROWSERCHOICE_EXE)))
		return browserchoice_exe_path;
	if (file_exists(BROWSERCHOICE_EXE))
		return (browserchoice_exe_path = strdup(BROWSERCHOICE_EXE));

	return NULL;
}

/**
 * Returns a pointer to a string containing the path to the Schleuse.exe file.
 * After you have called this function you should exit_tgpro_environment()
 * to free the memory.
 */
const char* get_schleuse_exe_path() {
	if (schleuse_exe_path)
		return schleuse_exe_path;

	if ((schleuse_exe_path = file_exists_in_path(get_start_path(), SCHLEUSE_EXE)))
		return schleuse_exe_path;
	if ((schleuse_exe_path = file_exists_in_path(get_tgpro_path(), SCHLEUSE_EXE)))
		return schleuse_exe_path;
	if (file_exists(SCHLEUSE_EXE))
		return (schleuse_exe_path = strdup(SCHLEUSE_EXE));

	return NULL;
}

/**
 * Returns a pointer to a string containing the path to the vncviewer.exe file.
 * After you have called this function you should exit_tgpro_environment()
 * to free the memory.
 */
const char* get_vncviewer_exe_path() {
	if (vncviewer_exe_path)
		return vncviewer_exe_path;

	if ((vncviewer_exe_path = file_exists_in_path(get_start_path(), VNCVIEWER_EXE)))
		return vncviewer_exe_path;
	if ((vncviewer_exe_path = file_exists_in_path(get_tgpro_path(), VNCVIEWER_EXE)))
		return vncviewer_exe_path;
	if (file_exists(VNCVIEWER_EXE))
		return (vncviewer_exe_path = strdup(VNCVIEWER_EXE));

	return NULL;
}

const char* get_tgpro_cfg_path() {
	if (tgpro_cfg_path)
		return tgpro_cfg_path;

#if defined(WIN32) || defined(WIN64)
	if ((tgpro_cfg_path = file_exists_in_path(get_start_path(), TGPRO_CFG)))
		return tgpro_cfg_path;
	if ((tgpro_cfg_path = file_exists_in_path(get_start_path(), TGPRO_CFG_SUBDIR)))
		return tgpro_cfg_path;
	if ((tgpro_cfg_path = file_exists_in_path(get_tgpro_path(), TGPRO_CFG)))
		return tgpro_cfg_path;
	if ((tgpro_cfg_path = file_exists_in_path(get_tgpro_path(), TGPRO_CFG_SUBDIR)))
		return tgpro_cfg_path;
	if (file_exists(TGPRO_CFG))
		return (tgpro_cfg_path = strdup(TGPRO_CFG));
	if (file_exists(TGPRO_CFG_SUBDIR))
		return (tgpro_cfg_path = strdup(TGPRO_CFG_SUBDIR));
#else
	const size_t tg_p_len = strlen(TGPRO_CFG) + 10;
	char tg_p[tg_p_len];
	snprintf(tg_p, tg_p_len - 1, "/etc/%s", TGPRO_CFG);
	tgpro_cfg_path = file_exists(tg_p) ? strdup(tg_p) : NULL;
#endif /* defined(WIN32) || defined(WIN64) */

	return tgpro_cfg_path;
}


const char* get_browserchoice_cfg_path() {
	if (browserchoice_cfg_path) {
		return browserchoice_cfg_path;
	}

#if defined(WIN32) || defined(WIN64)
	if ((browserchoice_cfg_path = file_exists_in_path(get_start_path(), BROWSERCHOICE_CFG)))
		return browserchoice_cfg_path;
	if ((browserchoice_cfg_path = file_exists_in_path(get_start_path(), BROWSERCHOICE_CFG_SUBDIR)))
		return browserchoice_cfg_path;
	if ((browserchoice_cfg_path = file_exists_in_path(get_tgpro_path(), BROWSERCHOICE_CFG)))
		return browserchoice_cfg_path;
	if ((browserchoice_cfg_path = file_exists_in_path(get_tgpro_path(), BROWSERCHOICE_CFG_SUBDIR)))
		return browserchoice_cfg_path;
	if (file_exists(BROWSERCHOICE_CFG))
		return (browserchoice_cfg_path = strdup(BROWSERCHOICE_CFG));
	if (file_exists(BROWSERCHOICE_CFG_SUBDIR))
		return (browserchoice_cfg_path = strdup(BROWSERCHOICE_CFG_SUBDIR));
#endif /* defined(WIN32) || defined(WIN64) */

	return browserchoice_cfg_path;
}

const char* get_tgpro_vnc_path() {
	if (tgpro_vnc_path)
		return tgpro_vnc_path;

#if defined(WIN32) || defined(WIN64)
	if ((tgpro_cfg_path = file_exists_in_path(get_start_path(), TGPRO_VNC)))
		return tgpro_vnc_path;
	if ((tgpro_cfg_path = file_exists_in_path(get_start_path(), TGPRO_VNC_SUBDIR)))
		return tgpro_vnc_path;
	if ((tgpro_cfg_path = file_exists_in_path(get_appdata(0), TGPRO_VNC_SUBDIR)))
		return tgpro_vnc_path;
	if ((tgpro_cfg_path = file_exists_in_path(get_appdata(0), TGPRO_VNC)))
		return tgpro_vnc_path;
	if (file_exists(TGPRO_VNC))
		return (tgpro_vnc_path = strdup(TGPRO_VNC));
	if (file_exists(TGPRO_VNC_SUBDIR))
		return (tgpro_vnc_path = strdup(TGPRO_VNC_SUBDIR));
#else
	const size_t tg_p_len = strlen(TGPRO_VNC) + 10;
	char tg_p[tg_p_len];
	snprintf(tg_p, tg_p_len - 1, "~/.vnc/%s", TGPRO_VNC);
	tgpro_cfg_path = file_exists(tg_p) ? strdup(tg_p) : NULL;
#endif /* defined(WIN32) || defined(WIN64) */

	return tgpro_vnc_path;
}


/**
 * Returns a pointer to a string containing the path to the putty.exe file.
 * After you have called this function you should exit_tgpro_environment()
 * to free the memory.
 */
const char* get_putty_exe_path() {
	if (putty_exe_path)
		return putty_exe_path;

	if ((putty_exe_path = file_exists_in_path(get_start_path(), PUTTY_EXE)))
		return putty_exe_path;
	if ((putty_exe_path = file_exists_in_path(get_start_path(), PUTTY_EXE_SUBDIR)))
		return putty_exe_path;
	if ((putty_exe_path = file_exists_in_path(get_tgpro_path(), PUTTY_EXE)))
		return putty_exe_path;
	if ((putty_exe_path = file_exists_in_path(get_tgpro_path(), PUTTY_EXE_SUBDIR)))
		return putty_exe_path;
	if (file_exists(PUTTY_EXE))
		return (putty_exe_path = strdup(PUTTY_EXE));
	if (file_exists(PUTTY_EXE_SUBDIR))
		return (putty_exe_path = strdup(PUTTY_EXE_SUBDIR));

	return NULL;
}


/**
 * Returns a pointer to a string containing the path to the WinSCP.com file.
 * After you have called this function you should exit_tgpro_environment()
 * to free the memory.
 */
const char* get_winscp_com_path_spool() {
	if (winscp_com_path)
		return winscp_com_path;

	if ((winscp_com_path = file_exists_in_path(get_start_path(), WINSCP_COM)))
		return winscp_com_path;
	if ((winscp_com_path = file_exists_in_path(get_start_path(), WINSCP_COM_SUBDIR_SPOOL)))
		return winscp_com_path;
	if ((winscp_com_path = file_exists_in_path(get_tgpro_path(), WINSCP_COM)))
		return winscp_com_path;
	if ((winscp_com_path = file_exists_in_path(get_tgpro_path(), WINSCP_COM_SUBDIR_SPOOL)))
		return winscp_com_path;
	if (file_exists(WINSCP_COM))
		return (winscp_com_path = strdup(WINSCP_COM));
	if (file_exists(WINSCP_COM_SUBDIR_SPOOL))
		return (winscp_com_path = strdup(WINSCP_COM_SUBDIR_SPOOL));

	return NULL;
}


/**
 * Returns a pointer to a string containing the path to the WinSCP.com file.
 * After you have called this function you should exit_tgpro_environment()
 * to free the memory.
 */
const char* get_winscp_com_path_transfer() {
	return get_winscp_com_path_spool();
}


/**
 * Returns a pointer to a string containing the APPDATA path.
 * Do not try to free the memory of the string you got.
 * You can choose whether to create a vnc subdir.
 */
const char* get_appdata(int create_vnc_subdir) {
	if (!appdata_path) {
		const char* raw_appdata = getenv("APPDATA");
		error_check(raw_appdata == NULL, L"Die Systemvariable APPDATA wurde nicht gefunden.");

#if defined(WIN32) || defined(WIN64)
		// Now get the long path name (without the legacy ~ in path name)
		char long_appdata[MAX_PATH];
		DWORD len = GetLongPathNameA(raw_appdata, long_appdata, sizeof(long_appdata));
		if (len == 0 || len >= sizeof(long_appdata)) {
			// Failed to resolve, fallback to original
			appdata_path = strdup(raw_appdata);
		} else {
			appdata_path = strdup(long_appdata);
		}
#else
		appdata_path = strdup(raw_appdata);
#endif

		if (create_vnc_subdir) {
			char* vnc_path = malloc(strlen(appdata_path) + strlen("\\vnc") + 1);
			strcpy(vnc_path, appdata_path);
			strcat(vnc_path, "\\vnc");
#if defined(WIN32) || defined(WIN64)
			if (mkdir(vnc_path))
#else
			if (mkdir(vnc_path, 0700))
#endif
				error_check(errno != EEXIST, L"Die Erstellung des vnc-Unterverzeichnisses in %APPDATA% ist fehlgeschlagen.");
			free(vnc_path);
		}
	}
	return appdata_path;
}

/**
 * Returns a pointer to a string containing the APPDATA path.
 * Do not try to free the memory of the string you got.
 *
 * Creates a vnc subdir or historic reasons.
 */
const char* get_appdata_path() {
	/* true so that we don't break the existing API. I don't really know exactly when the vnc subfolder should be created.
	   I prefer to 'shut it off' when I know it needn't be used. This is the reason for the 'new' function. */
	return get_appdata(1);
}


/**
 * Returns a pointer to a string containing TG-Pro's installation path
 * It looks for the TightGate-Pro in the main PROGRAMFILES directory.
 * If there is also nothing to find we take the EXEs start path.
 * Do not try to free the memory of the string you got.
 */
const char* get_tgpro_path() {
	int i;
	char* programfiles_path = getenv("PROGRAMFILES");
	for (i=0; !tgpro_path && i<2; i++) {
		if (programfiles_path) {
			char* possible_tgpro_path = malloc(strlen(programfiles_path) +
							   strlen(TGPRO_SUBDIR) + 2);
			strcpy(possible_tgpro_path, programfiles_path);
			strcat(possible_tgpro_path, "\\");
			strcat(possible_tgpro_path, TGPRO_SUBDIR);
			struct stat pos_tg_stat;
			int err = stat(possible_tgpro_path, &pos_tg_stat);
			if ((err != -1) && (S_ISDIR(pos_tg_stat.st_mode)))
				tgpro_path = possible_tgpro_path;
			else
				free(possible_tgpro_path);
		}
		if (!tgpro_path)
			programfiles_path = getenv("PROGRAMFILES(X86)");
	}
	if (!tgpro_path)
		tgpro_path = strdup(get_start_path());

	if (tgpro_path)
		log_trace("TG-Pro installation path: %s", tgpro_path);

	return tgpro_path;
}

/**
 * Returns a pointer to a string containing TG-Selector's installation.
 * If there is also nothing to find we take the EXEs start path.
 * Do not try to free the memory of the string you got.
 */
const char* get_tgselector_path() {
	char* path = NULL;
	int i;
	char* programfiles_path = getenv("PROGRAMFILES");
	for (i=0; !path && i<2; i++) {
		if (programfiles_path) {
			char* possible_path = malloc(strlen(programfiles_path) + strlen(TGSELECTOR_SUBDIR) + 2);
			strcpy(possible_path, programfiles_path);
			strcat(possible_path, "\\");
			strcat(possible_path, TGSELECTOR_SUBDIR);
			struct stat pos_tg_stat;
			int err = stat(possible_path, &pos_tg_stat);
			if ((err != -1) && (S_ISDIR(pos_tg_stat.st_mode))) {
				path = possible_path;
			} else {
				free(possible_path);
			}
		}
		if (!path) {
			programfiles_path = getenv("PROGRAMFILES(X86)");
		}
	}

	if (!path)
		path = strdup(get_start_path());

	return path;
}

void fwrite_str(FILE* file, char* str) {
/*	char* concat(const char* str_a, const char* str_b) {
		char* ret = malloc(strlen(str_a) +
				   strlen(str_b) + 1);
		strcpy(ret, str_a);
		strcat(ret, str_b);
		return ret;
	}
*/
	size_t str_len = strlen(str);
	size_t written_bytes = fwrite(str, sizeof(char), str_len, file);
	error_check(written_bytes != str_len * sizeof(char), L"Ich konnte in eine Konfigurationsdatei nicht schreiben.");
}


const char* get_putty_cfg_path() {
	if (putty_cfg_path)
		return putty_cfg_path;

	putty_cfg_path =  malloc(strlen(get_appdata_path()) +
				 strlen(PUTTY_CFG_SUBDIR) + 1);

	strcpy(putty_cfg_path, get_appdata_path());
	strcat(putty_cfg_path, PUTTY_CFG_SUBDIR);

#if defined(WIN32) || defined(WIN64)
	if (mkdir(putty_cfg_path))
#else
	if (mkdir(putty_cfg_path, 0700))
#endif
		error_check(errno != EEXIST, L"Die Erstellung der Putty-Konfigurationsdatei in %APPDATA% ist fehlgeschlagen.");

	return putty_cfg_path;
}

char* construct_path(const char* base, const char* filename) {
	size_t size = strlen(base) + strlen(filename) + 2; // +1 for \, +1 for null terminator
	char* path = (char*) malloc(size);
	if (!path) {
		log_error("%s: Failed to allocate memory for path", __func__);
		return NULL;
	}
#ifdef WIN32
	snprintf(path, size, "%s\\%s", base, filename);
#else
	snprintf(path, size, "%s/%s", base, filename);
#endif
	path[size - 1] = '\0';
	return path;
}

char* try_get_whitelist_path(const char* dir, const char* primary_filename, const char* fallback_filename) {
	if (!dir) {
		log_error("%s: Directory is NULL. Cannot build whitelist path.", __func__);
		return NULL;
	}

	char* path = construct_path(dir, primary_filename);
	if (!path) {
		log_error("%s: Failed to allocate memory for whitelist path", __func__);
		return NULL;
	}

	if (!file_exists(path)) {
		log_trace("%s: The whitelist file does not exist: %s. Trying fallback file...", __func__, path);
		free(path);

		path = construct_path(dir, fallback_filename);
		if (!path) {
			log_error("%s: Failed to allocate memory for fallback whitelist path", __func__);
			return NULL;
		}

		if (!file_exists(path)) {
			log_trace("%s: The fallback whitelist file also does not exist: %s", __func__, path);
			free(path);
			return NULL;
		}
	}

	return path;
}

char* get_url_whitelist_path_startpath() {
	const char* startpath = get_start_path();
	if (!startpath) {
		log_error("%s: Failed to get startpath. Can't build 'url_whitelist_path_startpath'", __func__);
		return NULL;
	}

	return try_get_whitelist_path(startpath, URL_WHITELIST_FILENAME2, URL_WHITELIST_FILENAME);
}

char* get_url_whitelist_path_viewer_config_dir(const char* viewer_config_dir) {
	if (!viewer_config_dir) {
		log_error("%s: viewer_config_dir is NULL. Cannot build URL whitelist path.", __func__);
		return NULL;
	}

	return try_get_whitelist_path(viewer_config_dir, URL_WHITELIST_FILENAME2, URL_WHITELIST_FILENAME);
}

char* get_url_whitelist_path_user() {
#ifdef WIN32
	const char* appdata_path = get_appdata_path();
	if (!appdata_path) {
		log_error("%s: Failed to get appdata path. Can't build 'url_whitelist_path_user'", __func__);
		return NULL;
	}

	char* appdata_vnc_path = construct_path(appdata_path, "vnc");
#else
	char* appdata_path = getenv("HOME"); // appdata_path isn't a very good name, as it's actually the user's home
	if (!appdata_path) {
		const uid_t uid = getuid();
		const struct passwd* passwd = getpwuid(uid);
		if (!passwd) {
			log_error("%s: Failed to read home dir for user (!passwd)", __func__);
			return NULL;
		}
		appdata_path = passwd->pw_dir;
		if (!appdata_path) {
			log_error("%s: Failed to read home dir for user (!passwd->pw_dir)", __func__);
			return NULL;
		}
	}
	char* appdata_vnc_path = construct_path(appdata_path, ".vnc");
#endif
	if (!appdata_vnc_path) {
		log_error("%s: Failed to construct appdata_vnc_path", __func__);
		return NULL;
	}

	char* path = try_get_whitelist_path(appdata_vnc_path, URL_WHITELIST_FILENAME2, URL_WHITELIST_FILENAME);
	free(appdata_vnc_path);
	return path;
}

char* get_url_whitelist_path_system() {
	const char* tgpro_path = get_tgpro_path();
	if (!tgpro_path) {
		log_error("%s: Failed to get tgpro path. Can't build 'url_whitelist_path_system'", __func__);
		return NULL;
	}

	return try_get_whitelist_path(tgpro_path, URL_WHITELIST_FILENAME2, URL_WHITELIST_FILENAME);
}

char* strdup_and_lowercase(const char* old_str) {
	char* new_str = strdup(old_str);
	char* lower_caser;
	for (lower_caser = new_str; *lower_caser != '\0'; lower_caser++)
		*lower_caser = tolower(*lower_caser);
	return new_str;
}


char* jump_over_str(char* text_str, const char* to_jump_str) {
	size_t jump_str_len = strlen(to_jump_str);
	if (strncasecmp(text_str, to_jump_str, jump_str_len) == 0) {
		return text_str + jump_str_len;
	} else
		return text_str;
}


/* Converts an integer value to its hex character*/
char to_hex(char code) {
	static char hex[] = "0123456789abcdef";
	return hex[code & 15];
}

/* Returns a url-encoded version of str */
/* IMPORTANT: be sure to free() the returned string after use */
char* url_encode(char* str) {
	char* pstr = str;
	char* buf = malloc(strlen(str) * 3 + 1);
	char* pbuf = buf;

	while (*pstr) {
		if (isalnum(*pstr) || *pstr == '-' || *pstr == '_' || *pstr == '.' || *pstr == '~')
			*pbuf++ = *pstr;
		else if (*pstr == ' ')
			*pbuf++ = '+';
		else
			*pbuf++ = '%', *pbuf++ = to_hex(*pstr >> 4), *pbuf++ = to_hex(*pstr & 15);
		pstr++;
	}
	*pbuf = '\0';
	return buf;
}

/* Returns a url-encoded version of str */
/* The difference with url_semi_encode is that it adds
   a couple of exceptions to the chars that shouldn't
   be encoded so that Firefox understands the url. */
/* IMPORTANT: be sure to free() the returned string after use */
char* url_semi_encode(const char* url) {
	char* pstr = (char*) url;
	char* buf = malloc(strlen(url) * 3 + 1);
	char* pbuf = buf;

	while (*pstr) {
		if (isalnum(*pstr) || *pstr == '-' || *pstr == '_' || *pstr == '.' || *pstr == '~' || *pstr == ':' || *pstr == '/' || *pstr == '?' || *pstr == '=' || *pstr == '&'
				   || *pstr == ';' || *pstr == '@' || *pstr == '$' || *pstr == '+' || *pstr == '!' || *pstr == '*' || *pstr == '\\' || *pstr == '\''
				   || *pstr == '(' || *pstr == ')' || *pstr == ',' || *pstr == '%' || *pstr == '#' || *pstr == '|')
			*pbuf++ = *pstr;
		else if (*pstr == ' ')
			*pbuf++ = '+';
		else
			*pbuf++ = '%', *pbuf++ = to_hex(*pstr >> 4), *pbuf++ = to_hex(*pstr & 15);
		pstr++;
	}
	*pbuf = '\0';
	return buf;
}

/**
 * url_encode_non_ascii - Encodes a string to URL encoding format.
 *
 * This function takes an input string and encodes it in URL encoding format,
 * converting non-ASCII characters and optionally handling specified characters
 * based on user-defined rules. Characters that are explicitly marked for exclusion
 * will remain unchanged in the output, while characters that are to be forcefully encoded
 * will be encoded even if they are ASCII characters. Non-ASCII characters are always encoded.
 *
 * @param str: The input string to be encoded.
 * @param exclude: A string containing characters that should **not** be URL encoded.
 *                 These characters will appear unchanged in the encoded string.
 * @param force_encode: A string containing characters that should always be encoded.
 *                      Even if they are ASCII, they will be encoded as percent-encoded values.
 *
 * @return: A dynamically allocated string containing the URL-encoded result.
 *          The caller is responsible for freeing the memory.
 *
 * This function performs the following encoding rules:
 * - ASCII characters (values 0 to 127) are not encoded unless they are in the `force_encode` list.
 * - Non-ASCII characters (values 128 and above) are always encoded.
 * - Characters found in the `exclude` list are **not** encoded.
 * - Characters found in the `force_encode` list **are always encoded**, even if they are ASCII.
 *
 * Example:
 *   const char* test_str = "Hello World! C:\\Users\\Me\\File.txt";
 *   const char* exclude_chars = "\\:!";   // Exclude '\' and ':' and '!'
 *   const char* force_encode_chars = " "; // Force encode space (' ') as '%20'
 *
 *   The output will be: "Hello%20World! C:\Users\Me\File.txt"
 *   (Note that the space is encoded, while '\' and ':' remain unchanged.)
 *
 * Errors:
 *   If memory allocation fails, the function returns NULL.
 */
char* url_encode_non_ascii(const char* str, const char* exclude, const char* force_encode) {
	if (!str) {
		return NULL;
	}

	size_t len = strlen(str);
	char* buf = malloc(len * 3 + 1);  // Allocate max possible size
	if (!buf) {
		fprintf(stderr, "Failed to allocate memory for encoded string\n");
		return NULL;
	}

	const char* pstr = str;
	char* pbuf = buf;

	while (*pstr) {
		unsigned char c = (unsigned char) *pstr;

		// If character is in 'exclude' list, keep it unchanged
		if (exclude && strchr(exclude, c)) {
			*pbuf++ = c;
		}
		// If character is in 'force_encode' list, always encode it
		else if (force_encode && strchr(force_encode, c)) {
			sprintf(pbuf, "%%%02X", c);
			pbuf += 3;
		}
		// Keep ASCII characters unchanged
		else if (c < 128) {
			*pbuf++ = c;
		}
		// Encode non-ASCII characters
		else {
			sprintf(pbuf, "%%%02X", c);
			pbuf += 3;
		}
		pstr++;
	}
	*pbuf = '\0';

	return buf;
}

/**
 * Looks if the string-with-asterisks ast_str matches the string cmp_str.
 * ast_str is interpreted as LDE (Lewiocal Domainic Expression).
 *
 * Example: *.m-privacy.de/foo * matches m-privacy.de/footer.html
 * Example: *.m-privacy.de/ * matches blog.m-privacy.de/lew.html
 * Example: *.m-privacy.de/ * does NOT match baum-privacy.de/lew.html
 * Example: *.m-privacy.de/ * matches m-privacy.de
 */
int asterisk_strcmp(char* ast_str, char* cmp_str) {
	while (*ast_str != '\0' && *cmp_str != '\0') {
		if (*ast_str != '*') {
			if (*ast_str++ != *cmp_str++)          /* Asterisk-free normal comparison. */
				return 0;
		} else {
			while (*ast_str == '*')                /* Many asterisks in a row are useless but allowed. */
				++ast_str;
			if (*ast_str == '\0')                  /* Nothing behind the asterisk means the rest of cmp_str is ok. */
				return 1;
			char* next_ast = strchr(ast_str, '*'); /* Is there an asterisk after the asterisk? */
			if (next_ast)
				*next_ast = '\0';
			char* eq_str = strstr(cmp_str, ast_str);
			if (!eq_str) {
				if (*ast_str == '.') {
					eq_str = strstr(cmp_str, ++ast_str);
					if (eq_str != cmp_str) {
						if (next_ast)
							*next_ast = '*';
						return 0;
					}
				} else {
					if (next_ast)
						*next_ast = '*';
					return 0;
				}
			}
			size_t eq_len = strlen(ast_str);
			if (next_ast)
				*next_ast = '*';
			cmp_str = eq_str + eq_len;
			ast_str += eq_len;
		}
	}
	while (*ast_str == '/' || *ast_str == '*')
		++ast_str;
	while (*cmp_str == '/')
		++cmp_str;
	return *ast_str == '\0' && *cmp_str == '\0';    /* The LDE matches if both pointers point at their string's end. */
}

void free_url_match(struct url_match_t* url_match) {
	if (!url_match) {
		return;
	}

	if (url_match->url_expr) {
		free(url_match->url_expr);
	}
	if (url_match->dest) {
		free(url_match->dest);
	}
	if (url_match->browser){
		free(url_match->browser);
	}
	if (url_match->browser_profile) {
		free(url_match->browser_profile);
	}

	free(url_match);
}

/**
 * Parses a new cool line with '|' and writes it to a
 * struct url_match_t. This allows us to use the lists for remote
 * links too

 * This firstly appeared due to the need to send url's to tgpro
 * that use a specific firefox profile).

 * Example of a new cool line: *.m-privacy.de/foo|remote|firefox|my-cool-profile

 * It also understands old-school lines. These are always sent to
 * the configured local browser:
 * Example of an old-school line: *.m-privacy.de
 */
struct url_match_t* parse_line(const char* line) {
	struct url_match_t* url_match = (struct url_match_t*) malloc(sizeof(struct url_match_t));
	url_match->url_expr = NULL;
	url_match->dest = NULL;
	url_match->browser = NULL;
	url_match->browser_profile = NULL;

	char* url_expr = get_field(line, '|', 0);
	char* dest = get_field(line, '|', 1);
	char* browser = get_field(line, '|', 2);
	char* browser_profile = get_field(line, '|', 3);

	url_match->url_expr = url_expr;
	url_match->dest = dest ? dest : strdup("local"); // default to local for backwards compatibility (all links in the url list used to be opened with a local browser)
	url_match->browser = browser;
	url_match->browser_profile = browser_profile;

	return url_match;
}

struct url_match_t* is_in_url_file(char* lower_url_cut, FILE* wl_file)  {
	struct url_match_t* url_match = NULL;
	int ret = 0;
	char text_line_buffer[4096];
	unsigned int is_first_line = 1;

	while (fgets(text_line_buffer, 4095, wl_file) && !ret) {
		char* text_line = str_trim(text_line_buffer);

		if (is_first_line) {
			text_line = skip_bom(text_line);
			is_first_line = 0;
		}

		if (*text_line != '#' && *text_line != '\0') {
			char* lower_caser;
			for (lower_caser = text_line; *lower_caser != '\0'; lower_caser++)
				*lower_caser = tolower(*lower_caser);
			url_match = parse_line(text_line);
			log_trace(" --> url_expr: <%s>", url_match->url_expr);
		        log_trace(" --> dest: <%s>", url_match->dest);
		        log_trace(" --> browser: <%s>", url_match->browser);
			log_trace(" --> browser_profile: <%s>", url_match->browser_profile);

			char* text_line_cut = jump_over_str(url_match->url_expr, "https://");
			text_line_cut = jump_over_str(text_line_cut, "http://");
			char* final_char = text_line_cut + strlen(text_line_cut) - 1;
			if (*final_char == '/')
				*final_char = '\0';
			log_trace("Read line from whitelist: %s", text_line_cut);
			ret = asterisk_strcmp(text_line_cut, lower_url_cut);
			if (!ret && strlen(text_line_cut) > 2) {
				char* final_two_chars = text_line_cut + strlen(text_line_cut) - 2;
				log_trace("final chars: %s", final_two_chars);
				if (strcmp(final_two_chars, "/*") == 0) {
					*final_two_chars = '\0';
					ret = asterisk_strcmp(text_line_cut, lower_url_cut);
				}
			}
			if (!ret) {
				free_url_match(url_match);
				url_match = NULL;
			}
		}
	}

	return url_match;
}

struct url_match_t* is_in_specific_url_whitelist(const char* url, const char* whitelist_file) {
	error_check(!url, L"Der Funktion is_in_url_whitelist wurde kein URL übergeben.");
	log_debug("* Check URL '%s' in whitelist '%s'", url, whitelist_file);

	char* lower_url = strdup_and_lowercase(url);
	char* lower_url_cut = jump_over_str(lower_url, "https://");
	lower_url_cut = jump_over_str(lower_url_cut, "http://");
	log_debug("URL to compare is now: %s", lower_url_cut);

	FILE* wl_file = whitelist_file ? fopen(whitelist_file, "r") : NULL;
	struct url_match_t* url_match = is_in_url_file(lower_url_cut, wl_file);
	if (wl_file) {
		fclose(wl_file);
	}
	free(lower_url);
	return url_match;
}

struct url_match_t* is_in_any_default_url_whitelist(const char* url, const char* viewer_config_dir) { // Looks in Program Files, in %appdata%\vnc and next to browserchoice.exe and will open locally if it matches *any* of the whitelists
	struct url_match_t* url_match = NULL;

	if (viewer_config_dir) { // If we run browserchoice with -configdata, it should be able to look for a whitelist in configdir too
		char* viewer_config_dir_whitelist = get_url_whitelist_path_viewer_config_dir(viewer_config_dir);
		if (!viewer_config_dir_whitelist) {
			log_info("%s: Didn't find a whitelist in config dir '%s'", __func__, viewer_config_dir);
		} else {
			url_match = is_in_specific_url_whitelist(url, viewer_config_dir_whitelist);
			free(viewer_config_dir_whitelist);
			if (url_match) {
				return url_match;
			}
		}
	}

	char* system_whitelist = get_url_whitelist_path_system();
	if (!system_whitelist) {
		log_info("%s: Didn't find a whitelist in installation path", __func__);
	} else {
		url_match = is_in_specific_url_whitelist(url, system_whitelist);
		free(system_whitelist);
		if (url_match) {
			return url_match;
		}
	}

	char* user_whitelist = get_url_whitelist_path_user();
	if (!user_whitelist) {
		log_info("%s: Didn't find a user-specific whitelist", __func__);
	} else {
		url_match = is_in_specific_url_whitelist(url, user_whitelist);
		free(user_whitelist);
		if (url_match) {
			return url_match;
		}
	}

	char* startpath_whitelist = get_url_whitelist_path_startpath();
	if (!startpath_whitelist) {
		log_info("%s: Didn't find a whitelist in start path", __func__);
	} else  {
		url_match = is_in_specific_url_whitelist(url, startpath_whitelist);
		free(startpath_whitelist);
		if (url_match) {
			return url_match;
		}
	}

	log_info("%s: Didn't find any match (or couldn't find any whitelist)", __func__);
	return NULL;
}


const char* get_putty_settings_file() {
	if (putty_settings_file)
		return putty_settings_file;

	putty_settings_file = malloc(strlen(get_putty_cfg_path()) +
				     strlen(PUTTY_SESSIONS_SUBDIR) +
				     strlen(PUTTY_SETTINGS_FILE) + 1);

	strcpy(putty_settings_file, get_putty_cfg_path());
	strcat(putty_settings_file, PUTTY_SESSIONS_SUBDIR);
#if defined(WIN32) || defined(WIN64)
	if (mkdir(putty_settings_file))
#else
	if (mkdir(putty_settings_file, 0700))
#endif
		error_check(errno != EEXIST, L"Das Erstellen des Verzeichnisses für die Putty-Konfiguration in %APPDATA% ist fehlgeschlagen.");

	strcat(putty_settings_file, PUTTY_SETTINGS_FILE);
	if (!file_exists(putty_settings_file)) {
		FILE* p_set_f;
		error_check(!(p_set_f = fopen(putty_settings_file, "wb")), L"Ich konnte die Putty-Konfigurationsdatei in %APPDATA% nicht erzeugen.");
		fwrite_str(p_set_f, "WinTitle\\TG-Pro%20Admin-Login\\\n");
		fwrite_str(p_set_f, "ScrollBar\\0\\\n");
		fwrite_str(p_set_f, "GssapiFwd\\1\\\n");
		fwrite_str(p_set_f, "FontHeight\\12\\\n");
		fwrite_str(p_set_f, "FontCharSet\\0\\\n");
		fwrite_str(p_set_f, "FontIsBold\\0\\\n");
		fwrite_str(p_set_f, "Font\\Courier%20New\\\n");
		fwrite_str(p_set_f, "TermHeight\\40\\\n");
		fwrite_str(p_set_f, "TermWidth\\140\\\n");
		fwrite_str(p_set_f, "AgentFwd\\1\\\n");
		fwrite_str(p_set_f, "AuthGSSAPI\\1\\\n");
		fwrite_str(p_set_f, "AuthKI\\1\\\n");
		fwrite_str(p_set_f, "LineCodePage\\UTF-8\\\n");
		fwrite_str(p_set_f, "TerminalType\\linux\\\n");
		fclose(p_set_f);
	}

	return putty_settings_file;
}


/**
 * Returns the user's name.
 */
const char* get_user_name() {
	if (user_name)
		return user_name;

#if defined(WIN32) || defined(WIN64)
	DWORD max_uname_len = UNLEN + 1;
	user_name = calloc(max_uname_len, sizeof(TCHAR));
	error_check(!GetUserName(user_name, &max_uname_len), L"Ich konnte den Betriebssystem-Usernamen nicht feststellen.");
#else
	char* new_user_name = getlogin();
	error_check(!new_user_name, L"Ich konnte den Betriebssystem-Usernamen nicht feststellen.");
	user_name = strdup(new_user_name);
#endif /* defined(WIN32) || defined(WIN64) */

	return user_name;
}

char* get_current_session_id() {
#ifdef WIN32
	DWORD sessionId = WTSGetActiveConsoleSessionId();
	char* sessionIdStr = NULL;
	int bufferSize = snprintf(NULL, 0, "%lu", sessionId);

	if (bufferSize > 0) {
		sessionIdStr = (char*) malloc(bufferSize + 1);
		if (sessionIdStr != NULL) {
			snprintf(sessionIdStr, bufferSize + 1, "%lu", sessionId);
		}
	}
	return sessionIdStr;
#else
	return strdup("0");
#endif
}

#if defined(WIN32) || defined(WIN64)
#define MAX_NAME 256
BOOL get_username_and_domain_from_token(HANDLE hToken, char** username, char** domain) {
	DWORD dwSize = MAX_NAME;
	BOOL succeeded = FALSE;
	DWORD dwLength = 0;
	*username = "";
	*domain = "";
	PTOKEN_USER ptu = NULL;

	if (!hToken)
		goto Cleanup;

	if (!GetTokenInformation(
		    hToken,         // handle to the access token
		    TokenUser,      // get information about the token's groups
		    (LPVOID) ptu,   // pointer to PTOKEN_USER buffer
		    0,              // size of buffer
		    &dwLength       // receives required buffer size
		    ))
	{
		if (GetLastError() != ERROR_INSUFFICIENT_BUFFER)
			goto Cleanup;

		ptu = (PTOKEN_USER) HeapAlloc(GetProcessHeap(), HEAP_ZERO_MEMORY, dwLength);

		if (ptu == NULL)
			goto Cleanup;
	}

	if (!GetTokenInformation(
		    hToken,         // handle to the access token
		    TokenUser,      // get information about the token's groups
		    (LPVOID) ptu,   // pointer to PTOKEN_USER buffer
		    dwLength,       // size of buffer
		    &dwLength       // receives required buffer size
		    ))
	{
		goto Cleanup;
	}

	SID_NAME_USE SidType;
	char lpName[MAX_NAME];
	char lpDomain[MAX_NAME];

	if (!LookupAccountSid(NULL, ptu->User.Sid, lpName, &dwSize, lpDomain, &dwSize, &SidType)) {
		DWORD dwResult = GetLastError();
		if(dwResult == ERROR_NONE_MAPPED) {
			strcpy (lpName, "NONE_MAPPED");
		} else {
			log_error("LookupAccountSid Error %lu", GetLastError());
		}
	} else {
		*username = strdup(lpName);
		*domain = strdup(lpDomain);
		succeeded = TRUE;
	}

Cleanup:

	if (ptu != NULL) {
		HeapFree(GetProcessHeap(), 0, (LPVOID) ptu);
	}
	return succeeded;
}
#undef MAX_NAME
#endif /* defined(WIN32) || defined(WIN64) */

#if defined(WIN32) || defined(WIN64)
BOOL get_username_and_domain_from_process(const DWORD process_id,  char** username, char** domain) {
	HANDLE hProcess = OpenProcess(PROCESS_QUERY_INFORMATION, FALSE, process_id);
	if(hProcess == NULL) {
		log_error("Failed to OpenProcess (you probably don't have the rights to open this process because it doesn't belong to you)");
		return FALSE;
	}
	HANDLE hToken = NULL;

	if(!OpenProcessToken(hProcess, TOKEN_QUERY, &hToken)){
		log_error("Error: couldn't open process token");
		CloseHandle(hProcess);
		return FALSE;
	}
	BOOL succeeded = get_username_and_domain_from_token (hToken, username, domain);

	CloseHandle(hToken);
	CloseHandle(hProcess);
	return succeeded;
}
#endif /* defined(WIN32) || defined(WIN64) */

#if defined(WIN32) || defined(WIN64)
DWORD get_process_id_for_user(const char* processname, const char* username) {
	if (username) {
		log_trace("Looking for a process named '%s' for user '%s'", processname, username);
	} else {
		log_trace("Looking for a process named '%s' for any user", processname);
	}

	DWORD result = 0;

	// Take a snapshot of all processes in the system.
	HANDLE hProcessSnap = CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS, 0);
	if (hProcessSnap == INVALID_HANDLE_VALUE) {
		error_check(1, L"Ich konnte kein Snapshot der laufenden Prozesse machen. So kann ich nicht überprüfen, ob der VNC-Viewer schon läuft.");
		return 0;
	}

	PROCESSENTRY32 pe32;
	pe32.dwSize = sizeof(PROCESSENTRY32); // <----- IMPORTANT

	// Retrieve information about the first process,
	// and exit if unsuccessful
	if (!Process32First(hProcessSnap, &pe32)) {
		CloseHandle(hProcessSnap);          // clean the snapshot object
		error_check(1, L"Ich konnte den ersten Prozess aus der Prozessliste nicht auslesen.");
		return 0;
	}

	do {
		if (0 == strcmp(processname, pe32.szExeFile)) {
			result = pe32.th32ProcessID;
			if (!username) { // If no specific username was given, just return the first process with the specified name
				log_trace("I found a process (the first one) named '%s' and found out its process id: %ld", processname, result);
				break;
			} else {
				log_debug("I found a process named '%s' (process id: %ld)", processname, result);
				log_trace("Trying to find out whom it belongs to");
				char* process_username;
				char* process_user_domain;
				HRESULT read_user_from_process = get_username_and_domain_from_process(result, &process_username, &process_user_domain);
				if (read_user_from_process) {
					log_debug("Process' user: %s", process_username);
					log_debug("Specified user: %s", username);
					if (!strcmp(process_username, username)) {
						log_debug("Eureka! The process '%s' belongs to '%s'", processname, username);
						HWND window_handle = get_viewer_main_window_handle(result, 0);
						if (window_handle) {
							log_trace("It also is the owner of a TightGate-Pro Viewer window.");
							break;
						} else {
							log_trace("Unfortunately, its title doesn't contain the 'product name' so it probably isn't a TightGate-Viewer...");
							result = 0;
						}
					} else {
						log_debug("Unfortunately, the process doesn't belong to the speficied user '%s' but to '%s'", username, process_username);
						result = 0;
					}
				} else {
					log_error("Failed to get username from process. It probably doesn't belong to the current user");
					result = 0;
				}
			}
		}
	} while (Process32Next(hProcessSnap, &pe32));

	CloseHandle(hProcessSnap);
	return result;
}
#else
int get_process_id_for_user(const char* process_name, const char* username) {
	if (!process_name || !username) {
		log_error("%s: can't search for process without process name or username", __func__);
		return 0;
	}

	size_t command_size = 10 + strlen(process_name) + strlen(username) + 1;
	char* command = (char*) malloc(command_size);
	snprintf(command, command_size, "pgrep -u %s %s", username, process_name);
	command[command_size - 1] = '\0';

	FILE *pipe = popen(command, "r");
	if (!pipe) {
		log_error("%s: failed to open pipe to read pgrep's output", __func__);
		return -1;
	}

	pid_t my_pid = getpid();

	char buffer[20]; // Assume process id's have 20 chars max
	while (fgets(buffer, sizeof(buffer), pipe) != NULL) {
		log_debug("%s: found a running viewer with pid: %s", __func__, buffer);
		int process_id = atoi(buffer);
		if (process_id == my_pid) {
			log_debug("%s: found a tightgateviewer process, but it's ourselves. Looking for more tightgateviewer processe", __func__);
			continue;
		}
		pclose(pipe);
		return process_id;
	}

	log_error("%s: failed to find process '%s' for user '%s'", __func__, process_name, username);
	pclose(pipe);
	return 0;
}
#endif /* defined(WIN32) || defined(WIN64) */

#if defined(WIN32) || defined(WIN64)
unsigned int count_processes_for_user(const char* processname, const char* username) {
	if (username) {
		log_trace("Counting processes named '%s' for user '%s'", processname, username);
	} else {
		log_error("(Username can't be empty. Can't count processes named '%s' for all users. Aborting", processname);
		return 0;
	}

	unsigned int count = 0;

	// Take a snapshot of all processes in the system.
	HANDLE hProcessSnap = CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS, 0);
	if (hProcessSnap == INVALID_HANDLE_VALUE) {
		error_check(1, L"Ich konnte kein Snapshot der laufenden Prozesse machen. So kann ich nicht überprüfen, ob der VNC-Viewer schon läuft.");
		return 0;
	}

	PROCESSENTRY32 pe32;
	pe32.dwSize = sizeof(PROCESSENTRY32); // <----- IMPORTANT

	// Retrieve information about the first process,
	// and exit if unsuccessful
	if (!Process32First(hProcessSnap, &pe32)) {
		CloseHandle(hProcessSnap);          // clean the snapshot object
		error_check(1, L"Ich konnte den ersten Prozess aus der Prozessliste nicht auslesen.");
		return 0;
	}

	do {
		if (0 == strcmp(processname, pe32.szExeFile)) {
			DWORD pid = pe32.th32ProcessID;
			log_debug("I found a process named '%s' (process id: %ld)", processname, pid);
			log_trace("Trying to find out whom it belongs to");
			char* process_username;
			char* process_user_domain;
			HRESULT read_user_from_process = get_username_and_domain_from_process(pid, &process_username, &process_user_domain);
			if (read_user_from_process) {
				log_debug("Process' user: %s", process_username);
				log_debug("Specified user: %s", username);
				if (!strcmp(process_username, username)) {
					log_debug("Eureka! The process '%s' belongs to '%s'", processname, username);
					HWND window_handle = get_viewer_main_window_handle(pid, 0);
					if (window_handle) {
						log_trace("It also is the owner of a TightGate-Pro Viewer window. count++");
						count++;
					}
				} else {
					log_debug("Unfortunately, the process doesn't belong to the speficied user '%s' but to '%s'", username, process_username);
				}
			} else {
				log_debug("Failed to get username from process. It probably doesn't belong to the current user");
			}
		}
	} while (Process32Next(hProcessSnap, &pe32));

	CloseHandle(hProcessSnap);
	return count;
}
#endif /* defined(WIN32) || defined(WIN64) */

#if defined(WIN32) || defined(WIN64)
DWORD get_process_id(const char *processname) {
	return get_process_id_for_user(processname, NULL);
}
#endif /* defined(WIN32) || defined(WIN64) */

int check_running(const char* prog_name) {
#if defined(WIN32) || defined(WIN64)
	return get_process_id(prog_name) ? 1 : 0;
#endif
	return 0;
}

char* get_product_name_from_config_file() {
	char* value = NULL;
	value = get_tgpro_cfg_value("ProductName");
	if (!(value && *value)) {
		log_warn("%s: Couldn't get product name from tgpro config files. Using default product name 'TightGate-Pro'", __func__);
		value = strdup("TightGate-Pro");
	}
	return value;
}

#if defined(WIN32) || defined(WIN64)
// This gets called by winapi for every window on the desktop
BOOL CALLBACK EnumWindowsProc(HWND window_handle, LPARAM process_id)  {
	DWORD searched_process_id = (DWORD) process_id;  // This is the process ID we search for (passed from BringToForeground as process_id)
	DWORD window_process_id = 0;
	GetWindowThreadProcessId(window_handle, &window_process_id); // Get process ID of the window we just found
	if (searched_process_id == window_process_id)  {  // Is it the process we care about?
		WINDOWINFO* window_info = malloc(sizeof(WINDOWINFO));
		window_info->cbSize = sizeof(WINDOWINFO);
		GetWindowInfo(window_handle, window_info);
		free(window_info);

		char* window_title = malloc(128);
		GetWindowText(window_handle, window_title, 128);
		log_trace("%s: Window title: %s", __func__, window_title);
		char* product_name = get_product_name_from_config_file();
		log_trace("%s: Product name (to look for in window title): %s", __func__, product_name);
		if (strstr(window_title, "@") && strstr(window_title, "dpi")) { // TODO: Yes, this is a really dirty hack. I should really find a way to fond out what window I just found.
			log_trace("%s: Found a window with 'dpi' and '@' in its title (the old-hacky way). Will still try to find a better match using the title if possible.", __func__);
			viewer_main_window_handle = window_handle;
		}
		if (strstr(window_title, product_name)) {
			log_trace("%s: Yay! Product name '%s' found in window title: %s", __func__, product_name, window_title);
			viewer_main_window_handle = window_handle;
			free(product_name);
			free(window_title);
			return FALSE;  // Stop enumerating windows
		}
		free(product_name);
		free(window_title);
	}
	return TRUE;  // Continue enumerating
}
#endif /* defined(WIN32) || defined(WIN64) */

#if defined(WIN32) || defined(WIN64)
HWND get_viewer_main_window_handle(const DWORD process_id, const unsigned int force_reload) {
	if (force_reload) {
		viewer_main_window_handle = NULL;
		log_trace("%s: forcing reload of 'viewer_main_window_handle'", __func__);
	} else if (viewer_main_window_handle) {
		log_trace("%s: returning already available 'viewer_main_window_handle' (force reload was set to false)", __func__);
		return viewer_main_window_handle;
	}

	log_trace("%s: About to look for the window with EnumWindows", __func__);
	EnumWindows(&EnumWindowsProc, (LPARAM) process_id);
	if (!viewer_main_window_handle) {
		// If it finds the viewer main window, it stores it in viewer_main_window_handle
		log_error("Error while looking for the window.");
	}
	return viewer_main_window_handle;
}
#endif /* defined(WIN32) || defined(WIN64) */

#if defined(WIN32) || defined(WIN64)
// Callback function for EnumWindows
BOOL CALLBACK window_to_front_enum_windows_callback(HWND hwnd, LPARAM lParam) {
	DWORD processId;
	GetWindowThreadProcessId(hwnd, &processId);

	if (processId == (DWORD)lParam) {
		// Found the window for processId
		if (!IsZoomed(hwnd)) {
			log_trace("%s: window (pid=%lu) is not zoomed. Restoring...", __func__, processId);
			if (ShowWindow(hwnd, SW_RESTORE)) {
				log_trace("%s: managed to restore (SW_RESTORE) window (pid=%lu)", __func__, processId);
			} else {
				log_error("%s: failed to restore (SW_RESTORE) window (pid=%lu)", __func__, processId);
			}
		}

		if (SetForegroundWindow(hwnd)) {
			log_error("%s: managed to bring window (pid=%lu) to foreground (SetForegroundWindow(hwnd))", __func__, processId);		} else {
			log_error("%s: failed to bring window (pid=%lu) to foreground (SetForegroundWindow(hwnd))", __func__, processId);
		}
		return FALSE; // Stop enumerating windows
	}

	return TRUE; // Continue enumerating windows
}
#endif /* defined(WIN32) || defined(WIN64) */

#if defined(WIN32) || defined(WIN64)
void bring_window_to_front(const DWORD pid) {
	log_trace("%s: trying to bring window to front for app with pid: %d", __func__, pid);
	EnumWindows(window_to_front_enum_windows_callback, (LPARAM) pid);
}
#endif /* defined(WIN32) || defined(WIN64) */

#if defined(WIN32) || defined(WIN64)
/**
 * Restores the vncviewer window and brings it to the front (only on Windows)
 *
 */
void bring_viewer_window_to_front()  {
	DWORD process_id = get_process_id("vncviewer.exe");
	bring_window_to_front(process_id);
}
#endif /* defined(WIN32) || defined(WIN64) */

/**
 * Read configdir from command line if given. Returns a NULL if -configdir is not specified
 * Does not return the surrounding quotes but ALWAYS adds a trailing backslash!
 * Warning! The string has to be freed at the end (match allocates memory!)
 */
char* get_config_dir_parameter(char* command_line) {
#if defined(WIN32) || defined(WIN64)
	char* config_dir_param = match(command_line, "-configdir[= ]+((\".+?\")|([^ ]+))", 1);
	char* config_dir_param_with_trailing_backslash = NULL;
	if (config_dir_param) {
		char* p = config_dir_param;
		if (starts_with(p, "\"")) {
			p++;
		}
		if (ends_with(p, "\"")) {
			p[strlen(p) - 1] = 0;
		}
		while (ends_with(p, "\\")) { // Remove all trailing backslashes to make sure I have only one at the end
			p[strlen(p) - 1] = 0;
		}
		config_dir_param_with_trailing_backslash = malloc(strlen(p) + 2);
		sprintf(config_dir_param_with_trailing_backslash, "%s\\", p);
		free(config_dir_param);
		log_debug("Read configdir from command line: %s", config_dir_param_with_trailing_backslash);
	} else {
		log_error("Couldn't read configdir from command line!");
	}

	return config_dir_param_with_trailing_backslash;
#else
	return 0;
#endif

}

// TODO Put these two funtions into one with a parameter!!!
/**
 * Read vncviewer from command line if given. Returns a NULL if -vncviewer is not specified
 * Does not return the surrounding quotes but ALWAYS adds a trailing backslash!
 * Warning! The string has to be freed at the end (match allocates memory!)
 */
char* get_vncviewer_parameter(char* command_line) {
#if defined(WIN32) || defined(WIN64)
	char* vncviewer_param = match(command_line, "-vncviewer ((\".+?\")|([^ ]+))", 1);
	char* vncviewer_param_with_trailing_backslash = NULL;
	if (vncviewer_param) {
		if (starts_with(vncviewer_param, "\"")) {
			vncviewer_param++;
		}
		if (ends_with(vncviewer_param, "\"")) {
			vncviewer_param[strlen(vncviewer_param)-1] = 0;
		}
		while (ends_with(vncviewer_param, "\\")) { // Remove all trailing backslashes to make sure I have only one at the end
			vncviewer_param[strlen(vncviewer_param)-1] = 0;
		}
		vncviewer_param_with_trailing_backslash = malloc(strlen(vncviewer_param) + 2);
		sprintf(vncviewer_param_with_trailing_backslash, "%s\\", vncviewer_param);
		free(vncviewer_param);
		log_trace("Read vncviewer from command line: %s", vncviewer_param_with_trailing_backslash);
	} else {
		log_error("Couldn't read vncviewer from command line!");
	}

	return vncviewer_param_with_trailing_backslash;
#else
	return 0;
#endif /* if defined(WIN32) || defined(WIN64) */
}

#if defined(WIN32) || defined(WIN64)
wchar_t* get_running_exe() {
	log_trace("Get filename of (currently running) browserchoice");
	int len = 4096;
	wchar_t* filename = (wchar_t*) calloc(len, 2);
	int bytes = GetModuleFileNameW(NULL, filename, len);
	if (bytes == 0) {
		log_error("Failed to read filename");
		return NULL;
	} else {
		return filename;
	}
}
#endif /* defined(WIN32) || defined(WIN64) */

#if defined(WIN32) || defined(WIN64)
char* get_tmp_magicurl_pipe_name_file_path() {
	log_trace("Get filename where the magicurl named pipe name for this user is stored.");
	if (!tmp_magicurl_pipe_name_file_path) {
		tmp_magicurl_pipe_name_file_path = (char*) malloc(strlen(get_tgpro_tmp_path()) + strlen(MAGIC_URL_PIPE_NAME_FILE_NAME) + 1);
		strcpy(tmp_magicurl_pipe_name_file_path, get_tgpro_tmp_path());
		strcat(tmp_magicurl_pipe_name_file_path, MAGIC_URL_PIPE_NAME_FILE_NAME);
	}
	return tmp_magicurl_pipe_name_file_path;
}
#endif /* defined(WIN32) || defined(WIN64) */

char* get_viewer_log_file_path() {
	if (!viewer_log_file_path) {
		viewer_log_file_path = (char*) malloc(strlen(get_tgpro_tmp_path()) + strlen(VNCVIEWER_LOG_FILE_NAME) + 1);
		if (!viewer_log_file_path) {
			log_error("Error! Failed to allocate memory to store string for vncviewer log file path.");
			return NULL;
		}
		strcpy(viewer_log_file_path, get_tgpro_tmp_path());
		strcat(viewer_log_file_path, VNCVIEWER_LOG_FILE_NAME);
	}
	return viewer_log_file_path;
}

char* get_viewer_trace_file_path() {
	if (!viewer_trace_file_path) {
		viewer_trace_file_path = (char*) malloc(strlen(get_tgpro_tmp_path()) + strlen(VNCVIEWER_TRACE_FILE_NAME) + 1);
		if (!viewer_trace_file_path) {
			log_error("Error! Failed to allocate memory to store string for vncviewer trace file path.");
			return NULL;
		}
		strcpy(viewer_trace_file_path, get_tgpro_tmp_path());
		strcat(viewer_trace_file_path, VNCVIEWER_TRACE_FILE_NAME);
	}
	return viewer_trace_file_path;
}

char* get_viewer_instream_trace_file_path() {
	if (!viewer_instream_trace_file_path) {
		viewer_instream_trace_file_path = (char*) malloc(strlen(get_tgpro_tmp_path()) + strlen(VNCVIEWER_INSTREAM_TRACE_FILE_NAME) + 1);
		if (!viewer_instream_trace_file_path) {
			log_error("Error! Failed to allocate memory to store string for vncviewer instram trace file path.");
			return NULL;
		}
		strcpy(viewer_instream_trace_file_path, get_tgpro_tmp_path());
		strcat(viewer_instream_trace_file_path, VNCVIEWER_INSTREAM_TRACE_FILE_NAME);
	}
	return viewer_instream_trace_file_path;
}

char* get_viewer_outstream_trace_file_path() {
	if (!viewer_outstream_trace_file_path) {
		viewer_outstream_trace_file_path = (char*) malloc(strlen(get_tgpro_tmp_path()) + strlen(VNCVIEWER_OUTSTREAM_TRACE_FILE_NAME) + 1);
		if (!viewer_outstream_trace_file_path) {
			log_error("Error! Failed to allocate memory to store string for vncviewer instram trace file path.");
			return NULL;
		}
		strcpy(viewer_outstream_trace_file_path, get_tgpro_tmp_path());
		strcat(viewer_outstream_trace_file_path, VNCVIEWER_OUTSTREAM_TRACE_FILE_NAME);
	}
	return viewer_outstream_trace_file_path;
}

/**
 * Initializes a file logger. Mostly for Windows, as it's practically
 * impossible to see stdout. Always writes to %tmp% or /tmp.
 *
 * The filename parameter cannot include the path.
 *
 * The log_level parameter can be one of these values (see loglib/log.h):
 * LOG_TRACE, LOG_DEBUG, LOG_INFO, LOG_WARN, LOG_ERROR, LOG_FATAL
 */
int log_to_file_init(const char* filename, const int log_level) {
	const char* tmp_dir = get_tgpro_tmp_path();
	if (!tmp_dir) {
		fprintf(stderr, "Failed to initialize logging (failed to get tmp dir).\n");
		return 0;
	}

	char* log_file = (char*) malloc(4096);
	if (!log_file) {
		fprintf(stderr, "Failed to initialize logging (failed to allocate memory for log_file).\n");
		return 0;
	}
	snprintf(log_file, 4095, "%s/%s", tmp_dir, filename);
	log_file[4095] = '\0';

	char* log_file_old = (char*) malloc(4096);
	if (!log_file_old) {
		fprintf(stderr, "Failed to initialize logging (failed to allocate memory for log_file_old).\n");
		return 0;
	}
	snprintf(log_file_old, 4095, "%s.old", log_file);
	log_file[4095] = '\0';

	if (file_exists(log_file)) {
		mp_move(log_file, log_file_old, 1); // Renaming last log to log.old (overwriting). Only keeping one copy of log (like the viewer does)
	}

	log_fp = fopen(log_file, "ab");
	if(log_fp == NULL) {
		fprintf(stderr, "Failed to initialize logging (failed to open file to log to (%s)).\n", log_file);
		free(log_file);
		return 0;
	}
	free(log_file);
	log_add_fp(log_fp, log_level);

	return 1;
}

#if defined(WIN32) || defined(WIN64)
void log_version(const char* path_to_exe) {
	if (!path_to_exe) {
		log_error("Can't get file version of a program without its path.");
		return;
	}
	DWORD ver_handle = 0;
	UINT size = 0;
	LPBYTE buf = NULL;
	DWORD ver_size = GetFileVersionInfoSize(path_to_exe, &ver_handle);

	if (ver_size != 0) {
		char ver_data[ver_size];

		if (GetFileVersionInfo(path_to_exe, ver_handle, ver_size, (LPSTR)ver_data)) {
			if (VerQueryValue(ver_data, "\\", (VOID FAR* FAR*)&buf, &size)) {
				if (size) {
					VS_FIXEDFILEINFO *verInfo = (VS_FIXEDFILEINFO*) buf;
					if (verInfo->dwSignature == 0xfeef04bd) {
						log_info("%s version: %lu.%lu.%lu.%lu",
						       path_to_exe,
						       ( verInfo->dwFileVersionMS >> 16 ) & 0xffff,
						       ( verInfo->dwFileVersionMS >>  0 ) & 0xffff,
						       ( verInfo->dwFileVersionLS >> 16 ) & 0xffff,
						       ( verInfo->dwFileVersionLS >>  0 ) & 0xffff
							);
					}
				}
			}
		}
	}
}
#endif // defined(WIN32) || defined(WIN64)

#if defined(WIN32) || defined(WIN64)
void log_tgpro_environment() {
	log_info("Running as user: %s", get_user_name());

	const char* vncviewer_exe_path = get_vncviewer_exe_path();
	if (vncviewer_exe_path) {
		log_version(vncviewer_exe_path);
	} else {
		log_warn("Couldn't determine vncviewer.exe's version (unable to find .exe)");
	}
	const char* browserchoice_exe_path = get_browserchoice_exe_path();
	if (browserchoice_exe_path) {
		log_version(browserchoice_exe_path);
	} else {
		log_warn("Couldn't determine Browserchoice.exe's version (unable to find .exe)");
	}
	const char* schleuse_exe_path = get_schleuse_exe_path();
	if (schleuse_exe_path) {
		log_version(schleuse_exe_path);
	} else {
		log_warn("Couldn't determine Schleuse.exe's version (unable to find .exe)");
	}
	const char* winscp_exe_path = get_winscp_exe_path();
	if (winscp_exe_path) {
		log_version(winscp_exe_path);
	} else {
		log_warn("Couldn't determine WinSCP.exe's version (unable to find .exe)");
	}
}
#endif //defined(WIN32) || defined(WIN64)

#if defined(WIN32) || defined(WIN64)
char* get_absolute_path(const char* relative_path) {
	char* absolute_path = (char*) malloc(MAX_PATH + 1);
	if (!absolute_path) {
		log_error("%s: Failed to allocate memory for absolute path: %lu", GetLastError());
		return NULL;
	}
	DWORD result = GetFullPathName(relative_path, MAX_PATH, absolute_path, NULL);
	if (result == 0) {
		log_error("%s: Error getting absolute path: %lu", GetLastError());
		free(absolute_path);
		return NULL;
	}
	absolute_path[MAX_PATH] = '\0';
	log_debug("%s: absolute path: %s", __func__, absolute_path);
	return absolute_path;
}
#endif //defined(WIN32) || defined(WIN64)
#if defined(WIN32) || defined(WIN64)
int create_send_to_shortcut(const char* config_dir, const char* description) {

	const char* schleuse_path = get_schleuse_exe_path();
	if (!schleuse_path) {
		log_error("%s: failed to find Schleuse.exe. Can't create a link in shell:sendto!", __func__);
		return 0;
	}
	wchar_t* schleuse_path_w = utf8_char_to_wchar_t(schleuse_path);
	if (!schleuse_path_w) {
		log_error("%s: failed to transform schleuse_path const char* (%s) to wchar_t*", __func__, schleuse_path);
		return 0;
	}

	const char* working_dir = get_start_path();
	if (!working_dir) {
		log_error("%s: failed to get working dir. Can't create a link in shell:sendto!", __func__);
		return 0;
	}
	wchar_t* working_dir_w = utf8_char_to_wchar_t(working_dir);
	if (!working_dir_w) {
		log_error("%s: failed to transform working_dir const char* (%s) to wchar_t*", __func__, working_dir);
		return 0;
	}

	wchar_t* description_w = NULL;
	if (description && *description) {
		description_w = utf8_char_to_wchar_t(description);
		if (!description_w) {
			log_error("%s: failed to transform description const char* (%s) to wchar_t*", __func__, description);
			free(schleuse_path_w);
			return 0;
		}
	} else {
		description_w = wcsdup(L"Send file to TightGate-Pro"); // Datei an TightGate-Pro senden
	}

	char* arguments = NULL;
	size_t arguments_size = 0;
	if (config_dir && *config_dir) {
		arguments_size = strlen("-upload ") + strlen("-configdir ") + strlen(config_dir) + 2 + 1; // +2 for quotes surrounding path, +1 for null-termination
	} else {
		arguments_size = strlen("-upload") + 1; // +1 for null-termination
	}
	arguments = (char*) malloc(arguments_size);
	if (!arguments) {
		log_error("%s: failed to allocate memory for Schleuse call arguments", __func__);
		free(schleuse_path_w);
		if (description_w) {
			free(description_w);
		}
		return 0;
	}

	if (config_dir && *config_dir) {
		char* absolute_path = get_absolute_path(config_dir);
		if (absolute_path) {
			snprintf(arguments, arguments_size, "-configdir \"%s\" -upload", absolute_path); // Upload needs to be the last argument (so that the file to upload is the - actual - last)
			free(absolute_path);
		} else {
			snprintf(arguments, arguments_size, "-configdir \"%s\" -upload", config_dir); // Upload needs to be the last argument (so that the file to upload is the - actual - last)
		}
	} else {
		strncpy(arguments, "-upload", arguments_size);
	}
	arguments[arguments_size - 1] = '\0';

	wchar_t* arguments_w = utf8_char_to_wchar_t(arguments);
	if (!arguments_w) {
		log_error("%s: failed to transform arguments const char* (%s) to wchar_t*", __func__, arguments);
		return 0;
	}

	HRESULT hres;
	IShellLinkW* psl;
	WCHAR sendToPath[MAX_PATH];

	hres = CoInitialize(NULL);
	if (SUCCEEDED(hres)) {
		hres = CoCreateInstance(&CLSID_ShellLink, NULL, CLSCTX_INPROC_SERVER, &IID_IShellLinkW, (LPVOID*)&psl);
		if (SUCCEEDED(hres)) {
			IPersistFile* ppf;
			psl->lpVtbl->SetPath(psl, schleuse_path_w);
			psl->lpVtbl->SetArguments(psl, arguments_w);
			psl->lpVtbl->SetWorkingDirectory(psl, working_dir_w);
			psl->lpVtbl->SetDescription(psl, description_w);
			// Query IShellLink for the IPersistFile interface
			hres = psl->lpVtbl->QueryInterface(psl, &IID_IPersistFile, (void**)&ppf);
			if (SUCCEEDED(hres)) {
				// Get the SendTo folder path
				hres = SHGetFolderPathW(NULL, CSIDL_SENDTO, NULL, 0, sendToPath);
				if (SUCCEEDED(hres)) {
					swprintf(sendToPath + wcslen(sendToPath), MAX_PATH - wcslen(sendToPath), L"\\TightGate-Pro.lnk");
					// Save the link by calling IPersistFile::Save
					hres = ppf->lpVtbl->Save(ppf, sendToPath, TRUE);
					ppf->lpVtbl->Release(ppf);
				}
			}
			psl->lpVtbl->Release(psl);
		}
		CoUninitialize();
	}
	return 1;
}
#endif //defined(WIN32) || defined(WIN64)

/**
 * The functions in this file allocate memory when needed. Use this function
 * to free all allocated memory after use. It is not harmful to use other
 * functions again after calling exit_tgpro_environment(), but you have to
 * call exit_tgpro_environment() again after that.
 */
void exit_tgpro_environment() {
	if (sumatrapdf_exe_path) {
		free(sumatrapdf_exe_path);
		sumatrapdf_exe_path = NULL;
	}
	if (u_tmp_dir) {
		free(u_tmp_dir);
		u_tmp_dir = NULL;
	}
	if (winscp_exe_path) {
		free(winscp_exe_path);
		winscp_exe_path = NULL;
	}
	if (winscp_com_path) {
		free(winscp_com_path);
		winscp_com_path = NULL;
	}
	if (krb_host_name) {
		free(krb_host_name);
		krb_host_name = NULL;
	}
	if (putty_exe_path) {
		free(putty_exe_path);
		putty_exe_path = NULL;
	}
	if (user_name) {
		free(user_name);
		user_name = NULL;
	}
	if (putty_settings_file) {
		free(putty_settings_file);
		putty_settings_file = NULL;
	}
	if (putty_cfg_path) {
		free(putty_cfg_path);
		putty_cfg_path = NULL;
	}
	if (tgpro_cfg_path) {
		free(tgpro_cfg_path);
		tgpro_cfg_path = NULL;
	}
	if (tgpro_vnc_path) {
		free(tgpro_vnc_path);
		tgpro_vnc_path = NULL;
	}
	if (url_whitelist_path_system) {
		free(url_whitelist_path_system);
		url_whitelist_path_system = NULL;
	}
	if (url_whitelist_path_cwd) {
		free(url_whitelist_path_cwd);
		url_whitelist_path_cwd = NULL;
	}
	if (browserchoice_exe_path) {
		free(browserchoice_exe_path);
		browserchoice_exe_path = NULL;
	}
	if (schleuse_exe_path) {
		free(schleuse_exe_path);
		schleuse_exe_path = NULL;
	}
	if (vncviewer_exe_path) {
		free(vncviewer_exe_path);
		vncviewer_exe_path = NULL;
	}
	if (browserchoice_cfg_path) {
		free(browserchoice_cfg_path);
		browserchoice_cfg_path = NULL;
	}
	if (tmp_magicurl_pipe_name_file_path) {
		free(tmp_magicurl_pipe_name_file_path);
		tmp_magicurl_pipe_name_file_path = NULL;
	}
	if (viewer_log_file_path) {
		free(viewer_log_file_path);
		viewer_log_file_path = NULL;
	}
	if (viewer_trace_file_path) {
		free(viewer_trace_file_path);
		viewer_trace_file_path = NULL;
	}
	if (viewer_instream_trace_file_path) {
		free(viewer_instream_trace_file_path);
		viewer_instream_trace_file_path = NULL;
	}
	if (viewer_outstream_trace_file_path) {
		free(viewer_outstream_trace_file_path);
		viewer_outstream_trace_file_path = NULL;
	}
	if (log_fp) {
		fclose(log_fp);
		log_fp = NULL;
	}
}
