/*************************************************** */
/* Rule Set Based Access Control                     */
/*                                                   */
/* Author and (c) 1999-2014: Amon Ott <ao@rsbac.org> */
/*                                                   */
/* Last modified: 16/Oct/2014                        */
/*************************************************** */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <rsbac/types.h>
#include <rsbac/syscalls.h>
#include <rsbac/error.h>
#include <rsbac/helpers.h>
#include "nls.h"
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

char * progname;

void use(void)
    {
      printf(gettext("%s (RSBAC %s)\n***\n"), progname, VERSION);
      printf(gettext("Remove all groups and memberships of a user\n\n"));  
      printf(gettext("Use: %s [flags] user\n"), progname);  
      printf(gettext(" -h = this help, -- = no more flags,\n"));
      printf(gettext(" -z = do not also try numeric username as uid\n"));
      printf(gettext(" -y: remove without asking\n"));  
      printf(gettext(" -N ta = transaction number (default = value of RSBAC_TA, if set, or 0)\n"));
    }

int main(int argc, char ** argv)
{
  int res = 0;
  rsbac_list_ta_number_t ta_number = 0;
  int allyes = 0;
  int nonumeric = 0;
  u_int stopflags = FALSE;

  locale_init();
  progname = argv[0];
  
  {
    char * env = getenv("RSBAC_TA");

    if(env)
      ta_number = strtoul(env,0,0);
  }
  while((argc > 1) && (argv[1][0] == '-') && !stopflags)
    {
      char * pos = argv[1];
      pos++;
      while(*pos)
        {
          switch(*pos)
            {
              case '-':
                stopflags = TRUE;
                break;
              case 'h':
                use();
                return 0;
              case 'y':
                allyes=1;
                break;
              case 'z':
                nonumeric=1;
                break;

              case 'N':
                if(argc > 2)
                  {
                    ta_number = strtoul(argv[2], 0, 10);
                    argc--;
                    argv++;
                  }
                else
                  {
                    fprintf(stderr, gettext("%s: missing transaction number value for parameter %c\n"), progname, *pos);
                    exit(1);
                  }
                break;

              default:
                fprintf(stderr, gettext("%s: unknown parameter %c\n"), progname, *pos);
                exit(1);
            }
          pos++;
        }
      argv++;
      argc--;
    }
  if (argc > 1)
    {
      struct rsbac_acl_syscall_arg_t   arg;
      char            yn;

      if(rsbac_get_uid(ta_number, &arg.tid.user, argv[1], nonumeric))
        {
          fprintf(stderr, gettext("%s: Invalid User %s!\n"),
                  progname, argv[1]);
          exit(1);
        }
      arg.target = T_USER;
      arg.subj_type = ACLS_USER;
      arg.subj_id = arg.tid.user;
      if(!allyes)
        {
          if (RSBAC_UID_SET(arg.tid.user))
            printf(gettext("Remove all groups and memberships of user %u/%u '%s' [y/n]\n"),
                 RSBAC_UID_SET(arg.tid.user),
                 RSBAC_UID_NUM(arg.tid.user),
                 argv[1]);
          else
            printf(gettext("Remove all groups and memberships of user %u '%s' [y/n]\n"),
                 RSBAC_UID_NUM(arg.tid.user),
                 argv[1]);
          yn = getchar();
          if(yn != 'y')
            exit(0);
        }
      res = rsbac_acl(ta_number, ACLC_remove_user, &arg);
      error_exit(res);
      exit(0);
    }
  else
    {
      use();
      exit(1);
    }
}

