/*************************************************** */
/* Rule Set Based Access Control                     */
/*                                                   */
/* Author and (c) 1999-2024: Amon Ott <ao@rsbac.org> */
/*                                                   */
/* Last modified: 14/Nov/2024                        */
/*************************************************** */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <rsbac/types.h>
#include <rsbac/aci_data_structures.h>
#include <rsbac/getname.h>
#include <rsbac/syscalls.h>
#include <rsbac/error.h>
#include <rsbac/helpers.h>
#include "nls.h"
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

int verbose=0;
int recurse=0;
int printall=0;
union rsbac_attribute_value_t value;
enum rsbac_target_t target;
enum rsbac_attribute_t attr;
char * progname;
char * target_n;
rsbac_list_ta_number_t ta_number = 0;
char none_name[] = "FD";

void use(void)
    {
      printf(gettext("%s (RSBAC %s)\n***\n"), progname, VERSION);
      printf(gettext("Use: %s [flags] target-type file/dirname(s)\n"), progname);  
      printf(gettext(" -h = this help, -- = no more flags,\n"));
      printf(gettext(" -v = verbose, -r = recurse into subdirs,\n"));
      printf(gettext(" -p = print right names,\n"));
      printf(gettext(" -N ta = transaction number (default = value of RSBAC_TA, if set, or 0)\n"));
      printf(gettext(" target-type = FILE, DIR, FIFO, SYMLINK, DEV or FD\n"));
      printf(gettext(" (FD: let %s decide between FILE, DIR, FIFO and SYMLINK, no DEV),\n"), progname);
    }

int process(char * name)
  {
    int res = 0;
    char tmp1[RSBAC_MAXNAMELEN];
    struct stat buf;
    rsbac_rc_request_vector_t request_vector;
    rsbac_time_t ttl;

    if(verbose)
      printf(gettext("Processing %s '%s'\n"),
             target_n,
             name);
    res = rsbac_rc_get_eff_rights_n(ta_number, target, name, &request_vector, &ttl);
    if(res)
      {
        get_error_name(tmp1,res);
        fprintf(stderr, gettext("%s: error: %s\n"), name, tmp1);
      }
    else
      {
        if(ttl)
          printf("%s: %s (ttl: %us)\n",
                 name, u64tostrlog(tmp1, request_vector), ttl);
        else
          printf("%s: %s\n",
                 name, u64tostrlog(tmp1, request_vector));
        if(printall)
          {
            int i;

            for (i=0; i<R_NONE; i++)
              if(request_vector & ((rsbac_rc_request_vector_t) 1 << i))
                printf("  %s\n", get_request_name(tmp1,i));
          }
      }

    if(   !lstat(name,&buf)
       && S_ISDIR(buf.st_mode)
       && recurse)
      {
        DIR * dir_stream_p;
        struct dirent * dirent_p;
        char name2[PATH_MAX];

        if(S_ISLNK(buf.st_mode))
          return(0);
        if(!(dir_stream_p = opendir(name)))
          {
            fprintf(stderr, gettext("opendir for dir %s returned error: %s\n"),
                   name,
                   strerror(errno));
            return(-2);
          }
        while((dirent_p = readdir(dir_stream_p)))
          {
            if(   (strcmp(".",dirent_p->d_name))
               && (strcmp("..",dirent_p->d_name)) )
              {
                strcpy(name2,name);
                strcat(name2,"/");
                strcat(name2,dirent_p->d_name);
                process(name2);
              }
          }
        closedir(dir_stream_p);
      }
    return(0);
  }

int main(int argc, char ** argv)
{
  int res = 0;
  int i;
  u_int stopflags = FALSE;

  locale_init();
  
  progname = argv[0];
  {
    char * env = getenv("RSBAC_TA");

    if(env)
      ta_number = strtoul(env,0,0);
  }
  while((argc > 1) && (argv[1][0] == '-') && !stopflags)
    {
      char * pos = argv[1];
      pos++;
      while(*pos)
        {
          switch(*pos)
            {
              case '-':
                stopflags = TRUE;
                break;
              case 'h':
                use();
                return 0;
              case 'v':
                verbose=1;
                break;
              case 'r':
                recurse=1;
                break;
              case 'p':
                printall=1;
                break;
              case 'N':
                if(argc > 2)
                  {
                    ta_number = strtoul(argv[2], 0, 10);
                    argc--;
                    argv++;
                  }
                else
                  {
                    fprintf(stderr, gettext("%s: missing transaction number value for parameter %c\n"), progname, *pos);
                    exit(1);
                  }
                break;
              default:
                fprintf(stderr, gettext("%s: unknown parameter %c\n"), progname, *pos);
                exit(1);
            }
          pos++;
        }
      argv++;
      argc--;
    }
  if (argc > 1)
    {
      target = get_target_nr(argv[1]);
      if(   (target != T_DIR)
         && (target != T_FILE)
         && (target != T_FIFO)
         && (target != T_SYMLINK)
         && (target != T_DEV)
         && (target != T_FD)
        )
        {
          if(verbose)
            printf(gettext("%s: %i targets\n\n"), progname, argc - 1);
          fprintf(stderr, gettext("%s: No target type given, assuming FD\n"), progname);
          target = T_FD;
          target_n = none_name;
          for (i=1;i < (argc);i++)
            {
              process(argv[i]);
            }
        }
      else
        {
          target_n = argv[1];
          if(argc > 2)
            {
              if(verbose)
                printf(gettext("%s: %i targets\n\n"), progname, argc - 2);
              for (i=1;i < (argc-1);i++)
                {
                  process(argv[i+1]);
                }
            }
        }
    }
  else
    {
      use();
      return 1;
    }
  return (res);
}
