/*************************************************** */
/* Rule Set Based Access Control                     */
/*                                                   */
/* Author and (c) 1999-2017: Amon Ott <ao@rsbac.org> */
/*                                                   */
/* Last modified: 07/Nov/2017                        */
/*************************************************** */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <rsbac/types.h>
#include <rsbac/rc_data_structures.h>
#include <rsbac/getname.h>
#include <rsbac/rc_getname.h>
#include <rsbac/syscalls.h>
#include <rsbac/error.h>
#include <rsbac/helpers.h>
#include "nls.h"
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#define SETPROG "rc_set_item"

/* reserve list room for so many extra items - to avoid racing problems */
#define LISTROOM 10

rsbac_list_ta_number_t ta_number = 0;
char * progname;
char * htmltitle = NULL;

void use(void)
  {
    printf(gettext("%s (RSBAC %s)\n***\n"), progname, VERSION);
    printf(gettext("Use: %s [flags] rc-target-type id-nr item [sub-id-nr [right]]\n"), progname);  
    printf(gettext("     %s list_xxx\n"), progname);  
    printf(gettext("     %s list_unused_xxx  (_nr only)\n"), progname);  
    printf(gettext("     %s list_def_fd_ind_create_type{s|_nr|_values role-id\n"), progname);  
    printf(gettext("     %s backup\n"), progname);  
    printf(gettext("     %s print\n"), progname);  
    printf(gettext("     %s htmlprint\n"), progname);  
    printf(gettext(" -h = this help, -- = no more flags,\n"));
    printf(gettext(" -v = verbose, -p = print right names,\n"));
    printf(gettext(" -i = list items and values,\n"));
    printf(gettext(" -r = remove role before restore (backup only)\n"));
    printf(gettext(" -0 = explicitely set no rights (backup only)\n"));
    printf(gettext(" -N ta = transaction number (default = value of RSBAC_TA, if set, or 0)\n"));
    printf(gettext(" rc-target-type = ROLE or TYPE,\n"));
    printf(gettext(" id-nr = ROLE or TYPE number,\n"));
    printf(gettext(" item = entry line,\n"));
    printf(gettext(" sub-id-nr = use this sub-id (_comp items only),\n"));
    printf(gettext(" right = right name or number (type_comp items only),\n"));
    printf(gettext(" xxx = roles, fd_types, dev_types, ipc_types, user_types, process_types,\n"));
    printf(gettext("       scd_types, group_types, role_nr, fd_type_nr, dev_type_nr, ipc_type_nr,\n"));
    printf(gettext("       user_type_nr, process_type_nr, scd_type_nr, rights: print a list\n"));
    printf(gettext(" list_def_fd_ind_create_types etc.: print a list\n"));
  }

void print_html_rights(rsbac_rc_rights_vector_t rights, enum rsbac_target_t target)
{
  int k;
  rsbac_rc_rights_vector_t vector;
  char tmp1[RSBAC_MAXNAMELEN];

  switch(target) {
    case T_FD:
    case T_FILE:
    case T_DIR:
    case T_FIFO:
    case T_SYMLINK:
    case T_UNIXSOCK:
      vector = RSBAC_FD_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
      break;
    case T_DEV:
      vector = RSBAC_DEV_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
      break;
    case T_IPC:
      vector = RSBAC_IPC_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
      break;
    case T_SCD:
      vector = RSBAC_SCD_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
      break;
    case T_USER:
      vector = RSBAC_USER_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
      break;
    case T_GROUP:
      vector = RSBAC_GROUP_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
      break;
    case T_PROCESS:
      vector = RSBAC_PROCESS_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
      break;
    case T_NETDEV:
      vector = RSBAC_NETDEV_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
      break;
    case T_NETTEMP:
      vector = RSBAC_NETTEMP_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
      break;
    case T_NETOBJ:
      vector = RSBAC_NETTEMP_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
      break;
    default:
      vector = RSBAC_RC_ALL_REQUESTS;
  }
  for (k=0; k<R_NONE; k++)
    {
      vector = RSBAC_RC_RIGHTS_VECTOR(k);
      if(rights & vector)
        {
          if(!vector)
            printf(" <font color=\"#501515\">%s</font>", get_request_name(tmp1,k));
          else
          if(vector & RSBAC_READ_REQUEST_VECTOR)
            printf(" <font color=\"#008F00\">%s</font>", get_request_name(tmp1,k));
          else
          if(vector & RSBAC_READ_WRITE_REQUEST_VECTOR)
            printf(" <font color=\"#FF0000\">%s</font>", get_request_name(tmp1,k));
          else
          if(vector & RSBAC_EXECUTE_REQUEST_VECTOR)
            printf(" <font color=\"#FF8F00\">%s</font>", get_request_name(tmp1,k));
          else
          if(vector & RSBAC_SECURITY_REQUEST_VECTOR)
            printf(" <font color=\"#0000FF\">%s</font>", get_request_name(tmp1,k));
          else
          if(vector & RSBAC_SYSTEM_REQUEST_VECTOR)
            printf(" <font color=\"#800080\">%s</font>", get_request_name(tmp1,k));
          else
            printf(" %s", get_request_name(tmp1,k));
        }
    }
  for (k=RSBAC_RC_SPECIAL_RIGHT_BASE; k<RCR_NONE; k++)
    if(rights & RSBAC_RC_RIGHTS_VECTOR(k))
      printf(" <font color=\"#0000FF\">%s</font>", get_rc_special_right_name(tmp1,k));
}

char * print_html_bitstring(char * string, rsbac_rc_rights_vector_t rights)
{
  char tmp1[RSBAC_MAXNAMELEN];

  if(rights & (RSBAC_SECURITY_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR))
    sprintf(string, " <font color=\"#0000FF\">%s</font>", u64tostrrcr(tmp1,rights));
  else
  if(rights & (RSBAC_WRITE_REQUEST_VECTOR | RSBAC_READ_WRITE_OPEN_REQUEST_VECTOR))
    sprintf(string, " <font color=\"#FF0000\">%s</font>", u64tostrrcr(tmp1,rights));
  else
  if(rights & RSBAC_EXECUTE_REQUEST_VECTOR)
    sprintf(string, " <font color=\"#FF8F00\">%s</font>", u64tostrrcr(tmp1,rights));
  else
  if(rights & RSBAC_READ_REQUEST_VECTOR)
    sprintf(string, " <font color=\"#008F00\">%s</font>", u64tostrrcr(tmp1,rights));
  else
    sprintf(string, " %s", u64tostrrcr(tmp1,rights));

  return string;
}

void print_rights(rsbac_rc_rights_vector_t rights)
{
  int k;
  char tmp1[RSBAC_MAXNAMELEN];

  for (k=0; k<R_NONE; k++)
    if(rights & RSBAC_RC_RIGHTS_VECTOR(k))
      printf(" %s", get_request_name(tmp1,k));
  for (k=RSBAC_RC_SPECIAL_RIGHT_BASE; k<RCR_NONE; k++)
    if(rights & RSBAC_RC_RIGHTS_VECTOR(k))
      printf(" %s", get_rc_special_right_name(tmp1,k));
}

char * print_type_name(char * string,
                       rsbac_rc_type_id_t type,
                       enum rsbac_rc_item_t item)
{
  union rsbac_rc_target_id_t tid;
  union rsbac_rc_item_value_t value;
  int res;

  switch(type)
    {
      case RC_type_inherit_process:
        sprintf(string, "S: Inherit Process");
        break;

      case RC_type_inherit_parent:
        sprintf(string, "S: Inherit Parent");
        break;

      case RC_type_no_create:
        sprintf(string, "S: No Create");
        break;

      case RC_type_no_execute:
        sprintf(string, "S: No Execute");
        break;

      case RC_type_use_new_role_def_create:
        sprintf(string, "S: Use New Role Def Create");
        break;

      case RC_type_no_chown:
        sprintf(string, "S: No Chown");
        break;

      default:
        tid.type = type;
        res = rsbac_rc_get_item(ta_number, RT_TYPE, &tid, &tid, item, &value, NULL);
        if(!res)
          sprintf(string, "%s",
                  value.name);
        else
          sprintf(string, "*Unknown*");
    }

  return string;
}

int main(int argc, char ** argv)
{
  int res = 0;
  char tmp1[RSBAC_MAXNAMELEN],tmp2[RSBAC_MAXNAMELEN];
  enum rsbac_adf_request_t rc_right;
  int j,i;
  enum rsbac_rc_target_t target;
  union rsbac_rc_target_id_t tid;
  union rsbac_rc_target_id_t subtid;
  enum rsbac_rc_item_t item = RI_none;
  union rsbac_rc_item_value_t value;
  rsbac_time_t ttl;
  int role_entry_item_list[RSBAC_RC_NR_ROLE_ENTRY_ITEMS] = RSBAC_RC_ROLE_ENTRY_ITEM_LIST;
  int type_entry_item_list[RSBAC_RC_NR_TYPE_ENTRY_ITEMS] = RSBAC_RC_TYPE_ENTRY_ITEM_LIST;
  rsbac_boolean_t show_names = FALSE;
  int verbose=0;
  int printall=0;
  int remove=0;
  int norights=0;
  u_int stopflags = FALSE;

  progname = argv[0];
  locale_init();

  {
    char * env = getenv("RSBAC_TA");

    if(env)
      ta_number = strtoul(env,0,0);
  }

  while((argc > 1) && (argv[1][0] == '-') && !stopflags)
    {
      char * pos = argv[1];
      pos++;
      while(*pos)
        {
          switch(*pos)
            {
              case '-':
                stopflags = TRUE;
                break;
              case 'h':
                use();
                return 0;
              case 'v':
                verbose=1;
                break;
              case 'p':
                printall=1;
                break;
              case 'r':
                remove=1;
                break;
              case '0':
                norights=1;
                break;
              case 'i':
                if(   (argc > 2)
                   && ((item = get_rc_item_nr(argv[2])) != RI_none)
                  )
                  {
                    get_rc_item_name(tmp1, item);
                    get_rc_item_param(tmp2, item);
                    printf("%s \t%s\n",tmp1,tmp2);
                    exit(0);
                  }
                printf(gettext("- items and returned values = see following list:\n"));
                printf("- ROLE:\n");
                for (j=0;j<RSBAC_RC_NR_ROLE_ENTRY_ITEMS;j++)
                  {
                    get_rc_item_name(tmp1,role_entry_item_list[j]);
                    get_rc_item_param(tmp2,role_entry_item_list[j]);
                    printf("%s\t%s\n",tmp1,tmp2);
                  }
                printf("- TYPE:\n");
                for (j=0;j<RSBAC_RC_NR_TYPE_ENTRY_ITEMS;j++)
                  {
                    get_rc_item_name(tmp1,type_entry_item_list[j]);
                    get_rc_item_param(tmp2,type_entry_item_list[j]);
                    printf("%s\t%s\n",tmp1,tmp2);
                  }
                exit(0);
              case 'N':
                if(argc > 2)
                  {
                    ta_number = strtoul(argv[2], 0, 10);
                    argc--;
                    argv++;
                  }
                else
                  {
                    fprintf(stderr, gettext("%s: missing transaction number value for parameter %c\n"), progname, *pos);
                    exit(1);
                  }
                break;
              case 'H':
                if(argc > 2)
                  {
                    htmltitle = argv[2];
                    argc--;
                    argv++;
                  }
                else
                  {
                    fprintf(stderr, gettext("%s: missing string for parameter %c\n"), progname, *pos);
                    exit(1);
                  }
                break;

              default:
                fprintf(stderr, gettext("%s: unknown parameter %c\n"), progname, *pos);
                exit(1);
            }
          pos++;
        }
      argv++;
      argc--;
    }
  
  switch(argc)
    {
      case 2:
        if(   !strcmp(argv[1],"list_rights")
          )
          {
            char tmp[80];

            for(i=0; i<R_NONE; i++)
              printf("%s\n", get_request_name(tmp, i));
            for(i=RSBAC_RC_SPECIAL_RIGHT_BASE; i<RCR_NONE; i++)
              printf("%s\n", get_rc_special_right_name(tmp, i));
            exit(0);
          }
        if(   !strcmp(argv[1],"list_roles")
           || !strcmp(argv[1],"list_role_nr")
          )
          {
            __u32 * role_array;
            int nr_roles;

            target = RT_ROLE;
            tid.role = 0;
            item = RI_name;
            if(   !strcmp(argv[1],"list_roles")
              )
              show_names = TRUE;
            nr_roles = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_name, 0, NULL, NULL);
            error_exit(nr_roles);
            nr_roles += LISTROOM;
            role_array = malloc(nr_roles * sizeof(__u32));
            if(!role_array)
              {
                error_exit(-ENOMEM);
              }
            nr_roles = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_name, nr_roles, role_array, NULL);
            if(verbose)
              {
                printf(gettext("%u roles:\n"),
                       nr_roles);
              }
            if(nr_roles > 0) {
              qsort(role_array, nr_roles, sizeof(*role_array), rsbac_u32_void_compare);
              for(j=0; j<nr_roles; j++)
                {
                  if(show_names)
                    {
                      tid.role = role_array[j];
                      value.dummy = -1;
                      res = rsbac_rc_get_item(ta_number, target, &tid, &tid, item, &value, NULL);
                      if(!res)
                        {
                          for(i=0; i<strlen(value.name); i++)
                            if(value.name[i] == ' ')
                              value.name[i] = '_';
                          printf("%i %s\n", role_array[j], value.name);
                        }
                    }
                  else
                    printf("%u\n", role_array[j]);
                }
            }
            free(role_array);
            exit(0);
          }
        if(   !strcmp(argv[1],"list_fd_types")
           || !strcmp(argv[1],"list_fd_type_nr")
           || !strcmp(argv[1],"list_dev_types")
           || !strcmp(argv[1],"list_dev_type_nr")
           || !strcmp(argv[1],"list_user_types")
           || !strcmp(argv[1],"list_user_type_nr")
           || !strcmp(argv[1],"list_process_types")
           || !strcmp(argv[1],"list_process_type_nr")
           || !strcmp(argv[1],"list_ipc_types")
           || !strcmp(argv[1],"list_ipc_type_nr")
           || !strcmp(argv[1],"list_scd_types")
           || !strcmp(argv[1],"list_scd_type_nr")
           || !strcmp(argv[1],"list_group_types")
           || !strcmp(argv[1],"list_group_type_nr")
           || !strcmp(argv[1],"list_netdev_types")
           || !strcmp(argv[1],"list_netdev_type_nr")
           || !strcmp(argv[1],"list_nettemp_types")
           || !strcmp(argv[1],"list_nettemp_type_nr")
           || !strcmp(argv[1],"list_netobj_types")
           || !strcmp(argv[1],"list_netobj_type_nr")
          )
          {
            __u32 * type_array;
            int nr_types;

            if(   !strcmp(argv[1],"list_fd_types")
               || !strcmp(argv[1],"list_fd_type_nr")
              )
              item = RI_type_fd_name;
            else
            if(   !strcmp(argv[1],"list_dev_types")
               || !strcmp(argv[1],"list_dev_type_nr")
              )
              item = RI_type_dev_name;
            else
            if(   !strcmp(argv[1],"list_user_types")
               || !strcmp(argv[1],"list_user_type_nr")
              )
              item = RI_type_user_name;
            else
            if(   !strcmp(argv[1],"list_process_types")
               || !strcmp(argv[1],"list_process_type_nr")
              )
              item = RI_type_process_name;
            else
            if(   !strcmp(argv[1],"list_ipc_types")
               || !strcmp(argv[1],"list_ipc_type_nr")
              )
              item = RI_type_ipc_name;
            else
            if(   !strcmp(argv[1],"list_scd_types")
               || !strcmp(argv[1],"list_scd_type_nr")
              )
              item = RI_type_scd_name;
            else
            if(   !strcmp(argv[1],"list_group_types")
               || !strcmp(argv[1],"list_group_type_nr")
              )
              item = RI_type_group_name;
            else
            if(   !strcmp(argv[1],"list_netdev_types")
               || !strcmp(argv[1],"list_netdev_type_nr")
              )
              item = RI_type_netdev_name;
            else
            if(   !strcmp(argv[1],"list_nettemp_types")
               || !strcmp(argv[1],"list_nettemp_type_nr")
              )
              item = RI_type_nettemp_name;
            else
            if(   !strcmp(argv[1],"list_netobj_types")
               || !strcmp(argv[1],"list_netobj_type_nr")
              )
              item = RI_type_netobj_name;

            if(   !strcmp(argv[1],"list_fd_types")
               || !strcmp(argv[1],"list_dev_types")
               || !strcmp(argv[1],"list_user_types")
               || !strcmp(argv[1],"list_process_types")
               || !strcmp(argv[1],"list_ipc_types")
               || !strcmp(argv[1],"list_scd_types")
               || !strcmp(argv[1],"list_group_types")
               || !strcmp(argv[1],"list_netdev_types")
               || !strcmp(argv[1],"list_nettemp_types")
               || !strcmp(argv[1],"list_netobj_types")
              )
              show_names = TRUE;

            tid.type = 0;
            nr_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_types);
            type_array = malloc(nr_types * sizeof(__u32));
            if(!type_array)
              {
                error_exit(-ENOMEM);
              }
            nr_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_types, type_array, NULL);
            if(verbose)
              {
                printf(gettext("%u types:\n"),
                       nr_types);
              }
            if(nr_types > 0) {
              qsort(type_array, nr_types, sizeof(*type_array), rsbac_u32_void_compare);
              for(j=0; j<nr_types; j++)
                {
                  if(show_names)
                    {
                      tid.type = type_array[j];
                      value.dummy = -1;
                      res = rsbac_rc_get_item(ta_number, RT_TYPE, &tid, &tid, item, &value, NULL);
                      if(!res)
                        {
                          for(i=0; i<strlen(value.name); i++)
                            if(value.name[i] == ' ')
                              value.name[i] = '_';
                          printf("%i %s\n", type_array[j], value.name);
                        }
                    }
                  else
                    printf("%u\n", type_array[j]);
                }
            }
            free(type_array);
            exit(0);
          }
        if(   !strcmp(argv[1],"list_fd_rights")
           || !strcmp(argv[1],"list_dev_rights")
           || !strcmp(argv[1],"list_ipc_rights")
           || !strcmp(argv[1],"list_scd_rights")
           || !strcmp(argv[1],"list_user_rights")
           || !strcmp(argv[1],"list_process_rights")
           || !strcmp(argv[1],"list_group_rights")
           || !strcmp(argv[1],"list_netdev_rights")
           || !strcmp(argv[1],"list_nettemp_rights")
           || !strcmp(argv[1],"list_netobj_rights")
           || !strcmp(argv[1],"list_none_rights")
           || !strcmp(argv[1],"list_all_rights")
          )
          {
            rsbac_rc_rights_vector_t right_mask;

            switch(argv[1][5])
              {
                case 'f':
                  right_mask=RSBAC_FD_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
                  break;
                case 'd':
                  right_mask=RSBAC_DEV_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
                  break;
                case 'i':
                  right_mask=RSBAC_IPC_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
                  break;
                case 's':
                  right_mask=RSBAC_SCD_REQUEST_VECTOR | RSBAC_NONE_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
                  break;
                case 'u':
                  right_mask=RSBAC_USER_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
                  break;
                case 'p':
                  right_mask=RSBAC_RC_PROCESS_RIGHTS_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
                  break;
                case 'g':
                  right_mask=RSBAC_GROUP_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
                  break;
                case 'n':
                  switch(argv[1][8])
                    {
                      case 'd':
                        right_mask=RSBAC_NETDEV_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
                        break;
                      case 't':
                        right_mask=RSBAC_NETTEMP_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
                        break;
                      case 'o':
                        right_mask=RSBAC_NETOBJ_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
                        break;
                      default:
                        right_mask=RSBAC_NONE_REQUEST_VECTOR;
                        break;
                    }
                  break;
                case 'a':
                  right_mask=RSBAC_ALL_REQUEST_VECTOR | RSBAC_RC_SPECIAL_RIGHTS_VECTOR;
                  break;
                default:
                  fprintf(stderr, gettext("%s: Internal right list error, param %s!\n"),
                          progname,
                          argv[1]);
                  exit(1);
              }
            for(j=0; j<R_NONE; j++)
              {
                if(right_mask & ((rsbac_rc_rights_vector_t) 1 << j))
                  printf("%s\n", get_request_name(tmp1,j));
              }
            for(j=RSBAC_RC_SPECIAL_RIGHT_BASE; j<RCR_NONE; j++)
              {
                if(right_mask & ((rsbac_rc_rights_vector_t) 1 << j))
                  printf("%s\n", get_rc_special_right_name(tmp1,j));
              }
            exit(0);
          }
        if(!strcmp(argv[1],"list_unused_role_nr"))
          {
            item = RI_name;
            res = 0;
            j = -1;
            while(!res)
              {
                j++;
                tid.role = j;
                res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &tid, item, &value, NULL);
              }
            if(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
               || (res == -RSBAC_ENOTFOUND)
              )
              {
                printf("%u\n", j);
                res = 0;
              }
            error_exit(res);
            exit(0);
          }
        if(   !strcmp(argv[1],"list_unused_fd_type_nr")
           || !strcmp(argv[1],"list_unused_dev_type_nr")
           || !strcmp(argv[1],"list_unused_ipc_type_nr")
           || !strcmp(argv[1],"list_unused_user_type_nr")
           || !strcmp(argv[1],"list_unused_process_type_nr")
           || !strcmp(argv[1],"list_unused_group_type_nr")
           || !strcmp(argv[1],"list_unused_netdev_type_nr")
           || !strcmp(argv[1],"list_unused_nettemp_type_nr")
           || !strcmp(argv[1],"list_unused_netobj_type_nr")
          )
          {
            target = RT_TYPE;
            if(!strcmp(argv[1],"list_unused_fd_type_nr"))
              item = RI_type_fd_name;
            else
            if(!strcmp(argv[1],"list_unused_dev_type_nr"))
              item = RI_type_dev_name;
            else
            if(!strcmp(argv[1],"list_unused_ipc_type_nr"))
              item = RI_type_ipc_name;
            else
            if(!strcmp(argv[1],"list_unused_user_type_nr"))
              item = RI_type_user_name;
            else
            if(!strcmp(argv[1],"list_unused_process_type_nr"))
              item = RI_type_process_name;
            else
            if(!strcmp(argv[1],"list_unused_group_type_nr"))
              item = RI_type_group_name;
            else
            if(!strcmp(argv[1],"list_unused_netdev_type_nr"))
              item = RI_type_netdev_name;
            else
            if(!strcmp(argv[1],"list_unused_nettemp_type_nr"))
              item = RI_type_nettemp_name;
            else
            if(!strcmp(argv[1],"list_unused_netobj_type_nr"))
              item = RI_type_netobj_name;

            res = 0;
            j = -1;
            while(!res)
              {
                j++;
                tid.type = j;
                res = rsbac_rc_get_item(ta_number, RT_TYPE, &tid, &tid, item, &value, NULL);
              }
            if(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
               || (res == -RSBAC_ENOTFOUND)
              )
              {
                printf("%u\n", j);
                res = 0;
              }
            error_exit(res);
            exit(0);
          }
        if(   !strcmp(argv[1],"backup")
           || !strcmp(argv[1],"print")
           || !strcmp(argv[1],"htmlprint")
          )
          {
            __u32 * role_array;
            __u32 * sub_array;
            __u32 * type_array;
            rsbac_time_t * ttl_array;
            union rsbac_rc_item_value_t value2;
            int nr_roles;
            int nr_fd_types;
            int nr_dev_types;
            int nr_ipc_types;
            int nr_scd_types;
            int nr_user_types;
            int nr_process_types;
            int nr_group_types;
            int nr_netdev_types;
            int nr_nettemp_types;
            int nr_netobj_types;
            int sub_nr;
            rsbac_boolean_t doprint=FALSE;
            rsbac_boolean_t htmlprint=FALSE;

            if(!strcmp(argv[1],"print"))
              doprint=TRUE;
            else
            if(!strcmp(argv[1],"htmlprint"))
              htmlprint=TRUE;
            target = RT_TYPE;
            item = RI_type_fd_name;
            tid.role = 0;
            nr_fd_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_fd_types);
            nr_fd_types += LISTROOM;
            type_array = malloc(nr_fd_types * sizeof(__u32));
            if(!type_array)
              {
                error_exit(-ENOMEM);
              }
            nr_fd_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_fd_types, type_array, NULL);
            if(nr_fd_types > 0)
              qsort(type_array, nr_fd_types, sizeof(*type_array), rsbac_u32_void_compare);
            if(doprint)
              {
                printf("Types\n=====\n%u FD types defined:\n", nr_fd_types);
              }
            else
            if(htmlprint)
              {
                if(htmltitle)
                  printf("<html><title>RSBAC %s RC Configuration - %s</title></html>\n",
                         RSBAC_VERSION, htmltitle);
                else
                  printf("<html><title>RSBAC %s RC Configuration</title></html>\n", RSBAC_VERSION);
                printf("<body bgcolor=\"#ffffff\" text=\"#000000\" link=\"#303030\" vlink=\"#202020\">\n<a name=\"top\"></a>\n");
                if(htmltitle)
                  printf("<h1><font color=\"0000FF\"><a href=\"http://www.rsbac.org\">RSBAC</a> %s RC Configuration - %s</font></h1>\n", RSBAC_VERSION, htmltitle);
                else
                  printf("<h1><font color=\"0000FF\"><a href=\"http://www.rsbac.org\">RSBAC</a> %s RC Configuration</font></h1>\n", RSBAC_VERSION);
                printf("<hr/>\n<a name=\"types\"></a><h1><a href=\"#typerights\">Types</a></h1>\n<a name=\"typefd\"></a><h2>%u <a href=\"#typerightsfd\">FD</a> types defined</h2>\n", nr_fd_types);
                printf("<table border=1>\n<tr><th>Number</th><th>Name</th><th>Secdel</th></tr>\n");
              }
            for(j=0; j<nr_fd_types; j++)
              {
                tid.type = type_array[j];
                value.dummy = -1;
                res = rsbac_rc_get_item(ta_number, target, &tid, &tid, item, &value, NULL);
                if(!res)
                  {
                    if(value.name[0])
                      {
                        if(doprint)
                          printf("%u\t%s",
                                 type_array[j],
                                 value.name);
                        else
                        if(htmlprint)
                          printf("<a name=\"typefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a></td><td><a href=\"#typerightsfd%u\">%s</a></td>",
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 value.name);
                        else
                          printf("%s -V %u TYPE %u type_fd_name \"%s\"\n",
                                 SETPROG, RSBAC_API_VERSION_NR,
                                 type_array[j],
                                 value.name);
                      }
                  }
                res = rsbac_rc_get_item(ta_number, target, &tid, &tid, RI_type_fd_need_secdel, &value, NULL);
                if(!res)
                  {
                    if(value.need_secdel)
                      {
                        if(doprint)
                          printf(" (with secure delete)");
                        else
                        if(htmlprint)
                          printf("<td><font color=\"008F00\">yes</td></tr>\n");
                        else
                          printf("%s -V %u TYPE %u type_fd_need_secdel %u\n",
                                 SETPROG, RSBAC_API_VERSION_NR,
                                 type_array[j],
                                 value.need_secdel);
                      }
                    else
                      if(htmlprint)
                        printf("<td>no</td></tr>\n");
                  }
                if(doprint)
                  printf("\n");
              }
            free(type_array);
            if(htmlprint)
              printf("</table>\n");

            item = RI_type_dev_name;
            tid.role = 0;
            nr_dev_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_dev_types);
            nr_dev_types += LISTROOM;
            type_array = malloc(nr_dev_types * sizeof(__u32));
            if(!type_array)
              {
                error_exit(-ENOMEM);
              }
            nr_dev_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_dev_types, type_array, NULL);
            if(nr_dev_types > 0)
              qsort(type_array, nr_dev_types, sizeof(*type_array), rsbac_u32_void_compare);
            if(doprint)
              {
                printf("\n%u DEV types defined:\n", nr_dev_types);
              }
            else
            if(htmlprint)
              {
                printf("<a name=\"typedev\"></a><h2>%u <a href=\"#typerightsdev\">DEV</a> types defined</h2>\n", nr_dev_types);
                printf("<table border=1>\n<tr><th>Number</th><th>Name</th></tr>\n");
              }
            for(j=0; j<nr_dev_types; j++)
              {
                tid.type = type_array[j];
                value.dummy = -1;
                res = rsbac_rc_get_item(ta_number, target, &tid, &tid, item, &value, NULL);
                if(!res)
                  {
                    if(value.name[0])
                      {
                        if(doprint)
                          printf("%u\t%s\n",
                                 type_array[j],
                                 value.name);
                        else
                        if(htmlprint)
                          printf("<a name=\"typedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a></td><td><a href=\"#typerightsdev%u\">%s</a></td></tr>\n",
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 value.name);
                        else
                          printf("%s -V %u TYPE %u type_dev_name \"%s\"\n",
                                 SETPROG, RSBAC_API_VERSION_NR,
                                 type_array[j],
                                 value.name);
                      }
                  }
              }
            free(type_array);
            if(htmlprint)
              printf("</table>\n");

            item = RI_type_ipc_name;
            tid.role = 0;
            nr_ipc_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_ipc_types);
            nr_ipc_types += LISTROOM;
            type_array = malloc(nr_ipc_types * sizeof(__u32));
            if(!type_array)
              {
                error_exit(-ENOMEM);
              }
            nr_ipc_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_ipc_types, type_array, NULL);
            if(nr_dev_types > 0)
              qsort(type_array, nr_ipc_types, sizeof(*type_array), rsbac_u32_void_compare);
            if(doprint)
              {
                printf("\n%u IPC types defined:\n", nr_ipc_types);
              }
            else
            if(htmlprint)
              {
                printf("<a name=\"typeipc\"></a><h2>%u <a href=\"#typerightsipc\">IPC</a> types defined</h2>\n", nr_ipc_types);
                printf("<table border=1>\n<tr><th>Number</th><th>Name</th></tr>\n");
              }
            for(j=0; j<nr_ipc_types; j++)
              {
                tid.type = type_array[j];
                value.dummy = -1;
                res = rsbac_rc_get_item(ta_number, target, &tid, &tid, item, &value, NULL);
                if(!res)
                  {
                    if(value.name[0])
                      {
                        if(doprint)
                          printf("%u\t%s\n",
                                 type_array[j],
                                 value.name);
                        else
                        if(htmlprint)
                          printf("<a name=\"typeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a></td><td><a href=\"#typerightsipc%u\">%s</a></td></tr>\n",
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 value.name);
                        else
                          printf("%s -V %u TYPE %u type_ipc_name \"%s\"\n",
                                 SETPROG, RSBAC_API_VERSION_NR,
                                 type_array[j],
                                 value.name);
                      }
                  }
              }
            free(type_array);
            if(htmlprint)
              printf("</table>\n");

            item = RI_type_scd_name;
            tid.role = 0;
            nr_scd_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_scd_types);
            nr_scd_types += LISTROOM;
            type_array = malloc(nr_scd_types * sizeof(__u32));
            if(!type_array)
              {
                error_exit(-ENOMEM);
              }
            nr_scd_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_scd_types, type_array, NULL);
            if(nr_dev_types > 0)
              qsort(type_array, nr_scd_types, sizeof(*type_array), rsbac_u32_void_compare);
            if(doprint)
              {
                printf("\n%u SCD types defined:\n", nr_scd_types);
              }
            else
            if(htmlprint)
              {
                printf("<a name=\"typescd\"></a><h2>%u <a href=\"#typerightsscd\">SCD</a> types defined</h2>\n", nr_scd_types);
                printf("<table border=1>\n<tr><th>Number</th><th>Name</th></tr>\n");
              }
            for(j=0; j<nr_scd_types; j++)
              {
                tid.type = type_array[j];
                value.dummy = -1;
                res = rsbac_rc_get_item(ta_number, target, &tid, &tid, item, &value, NULL);
                if(!res)
                  {
                    if(value.name[0])
                      {
                        if(doprint)
                          printf("%u\t%s\n",
                                 type_array[j],
                                 value.name);
                        else
                        if(htmlprint)
                          printf("<a name=\"typescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a></td><td><a href=\"#typerightsscd%u\">%s</a></td></tr>\n",
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 value.name);
                      }
                  }
              }
            free(type_array);
            if(htmlprint)
              printf("</table>\n");

            item = RI_type_user_name;
            tid.role = 0;
            nr_user_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_user_types);
            nr_user_types += LISTROOM;
            type_array = malloc(nr_user_types * sizeof(__u32));
            if(!type_array)
              {
                error_exit(-ENOMEM);
              }
            nr_user_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_user_types, type_array, NULL);
            if(nr_user_types > 0)
              qsort(type_array, nr_user_types, sizeof(*type_array), rsbac_u32_void_compare);
            if(doprint)
              {
                printf("\n%u USER types defined:\n", nr_user_types);
              }
            else
            if(htmlprint)
              {
                printf("<a name=\"typeuser\"></a><h2>%u <a href=\"#typerightsuser\">USER</a> types defined</h2>\n", nr_user_types);
                printf("<table border=1>\n<tr><th>Number</th><th>Name</th></tr>\n");
              }
            for(j=0; j<nr_user_types; j++)
              {
                tid.type = type_array[j];
                value.dummy = -1;
                res = rsbac_rc_get_item(ta_number, target, &tid, &tid, item, &value, NULL);
                if(!res)
                  {
                    if(value.name[0])
                      {
                        if(doprint)
                          printf("%u\t%s\n",
                                 type_array[j],
                                 value.name);
                        else
                        if(htmlprint)
                          printf("<a name=\"typeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a></td><td><a href=\"#typerightsuser%u\">%s</a></td></tr>\n",
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 value.name);
                        else
                          printf("%s -V %u TYPE %u type_user_name \"%s\"\n",
                                 SETPROG, RSBAC_API_VERSION_NR,
                                 type_array[j],
                                 value.name);
                      }
                  }
              }
            free(type_array);
            if(htmlprint)
              printf("</table>\n");

            item = RI_type_process_name;
            tid.role = 0;
            nr_process_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_process_types);
            nr_process_types += LISTROOM;
            type_array = malloc(nr_process_types * sizeof(__u32));
            if(!type_array)
              {
                error_exit(-ENOMEM);
              }
            nr_process_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_process_types, type_array, NULL);
            if(nr_process_types > 0)
              qsort(type_array, nr_process_types, sizeof(*type_array), rsbac_u32_void_compare);
            if(doprint)
              {
                printf("\n%u PROCESS types defined:\n", nr_process_types);
              }
            else
            if(htmlprint)
              {
                printf("<a name=\"typeprocess\"></a><h2>%u <a href=\"#typerightsprocess\">PROCESS</a> types defined</h2>\n", nr_process_types);
                printf("<table border=1>\n<tr><th>Number</th><th>Name</th></tr>\n");
              }
            for(j=0; j<nr_process_types; j++)
              {
                tid.type = type_array[j];
                value.dummy = -1;
                res = rsbac_rc_get_item(ta_number, target, &tid, &tid, item, &value, NULL);
                if(!res)
                  {
                    if(value.name[0])
                      {
                        if(doprint)
                          printf("%u\t%s\n",
                                 type_array[j],
                                 value.name);
                        else
                        if(htmlprint)
                          printf("<a name=\"typeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a></td><td><a href=\"#typerightsprocess%u\">%s</a></td></tr>\n",
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 value.name);
                        else
                          printf("%s -V %u TYPE %u type_process_name \"%s\"\n",
                                 SETPROG, RSBAC_API_VERSION_NR,
                                 type_array[j],
                                 value.name);
                      }
                  }
              }
            free(type_array);
            if(htmlprint)
              printf("</table>\n");

            item = RI_type_group_name;
            tid.role = 0;
            nr_group_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_group_types);
            nr_group_types += LISTROOM;
            type_array = malloc(nr_group_types * sizeof(__u32));
            if(!type_array)
              {
                error_exit(-ENOMEM);
              }
            nr_group_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_group_types, type_array, NULL);
            if(nr_group_types > 0)
              qsort(type_array, nr_group_types, sizeof(*type_array), rsbac_u32_void_compare);
            if(doprint)
              {
                printf("\n%u GROUP types defined:\n", nr_group_types);
              }
            else
            if(htmlprint)
              {
                printf("<a name=\"typegroup\"></a><h2>%u <a href=\"#typerightsgroup\">GROUP</a> types defined</h2>\n", nr_group_types);
                printf("<table border=1>\n<tr><th>Number</th><th>Name</th></tr>\n");
              }
            for(j=0; j<nr_group_types; j++)
              {
                tid.type = type_array[j];
                value.dummy = -1;
                res = rsbac_rc_get_item(ta_number, target, &tid, &tid, item, &value, NULL);
                if(!res)
                  {
                    if(value.name[0])
                      {
                        if(doprint)
                          printf("%u\t%s\n",
                                 type_array[j],
                                 value.name);
                        else
                        if(htmlprint)
                          printf("<a name=\"typegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a></td><td><a href=\"#typerightsgroup%u\">%s</a></td></tr>\n",
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 value.name);
                        else
                          printf("%s -V %u TYPE %u type_group_name \"%s\"\n",
                                 SETPROG, RSBAC_API_VERSION_NR,
                                 type_array[j],
                                 value.name);
                      }
                  }
              }
            free(type_array);
            if(htmlprint)
              printf("</table>\n");

            item = RI_type_netdev_name;
            tid.role = 0;
            nr_netdev_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_netdev_types);
            nr_netdev_types += LISTROOM;
            type_array = malloc(nr_netdev_types * sizeof(__u32));
            if(!type_array)
              {
                error_exit(-ENOMEM);
              }
            nr_netdev_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_netdev_types, type_array, NULL);
            if(nr_netdev_types > 0)
              qsort(type_array, nr_netdev_types, sizeof(*type_array), rsbac_u32_void_compare);
            if(doprint)
              {
                printf("\n%u NETDEV types defined:\n", nr_netdev_types);
              }
            else
            if(htmlprint)
              {
                printf("<a name=\"typenetdev\"></a><h2>%u <a href=\"#typerightsnetdev\">NETDEV</a> types defined</h2>\n", nr_netdev_types);
                printf("<table border=1>\n<tr><th>Number</th><th>Name</th></tr>\n");
              }
            for(j=0; j<nr_netdev_types; j++)
              {
                tid.type = type_array[j];
                value.dummy = -1;
                res = rsbac_rc_get_item(ta_number, target, &tid, &tid, item, &value, NULL);
                if(!res)
                  {
                    if(value.name[0])
                      {
                        if(doprint)
                          printf("%u\t%s\n",
                                 type_array[j],
                                 value.name);
                        else
                        if(htmlprint)
                          printf("<a name=\"typenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a></td><td><a href=\"#typerightsnetdev%u\">%s</a></td></tr>\n",
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 value.name);
                        else
                          printf("%s -V %u TYPE %u type_netdev_name \"%s\"\n",
                                 SETPROG, RSBAC_API_VERSION_NR,
                                 type_array[j],
                                 value.name);
                      }
                  }
              }
            free(type_array);
            if(htmlprint)
              printf("</table>\n");

            item = RI_type_nettemp_name;
            tid.role = 0;
            nr_nettemp_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_nettemp_types);
            nr_nettemp_types += LISTROOM;
            type_array = malloc(nr_nettemp_types * sizeof(__u32));
            if(!type_array)
              {
                error_exit(-ENOMEM);
              }
            nr_nettemp_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_nettemp_types, type_array, NULL);
            if(nr_nettemp_types > 0)
              qsort(type_array, nr_nettemp_types, sizeof(*type_array), rsbac_u32_void_compare);
            if(doprint)
              {
                printf("\n%u NETTEMP types defined:\n", nr_nettemp_types);
              }
            else
            if(htmlprint)
              {
                printf("<a name=\"typenettemp\"></a><h2>%u <a href=\"#typerightsnettemp\">NETTEMP</a> types defined</h2>\n", nr_nettemp_types);
                printf("<table border=1>\n<tr><th>Number</th><th>Name</th></tr>\n");
              }
            for(j=0; j<nr_nettemp_types; j++)
              {
                tid.type = type_array[j];
                value.dummy = -1;
                res = rsbac_rc_get_item(ta_number, target, &tid, &tid, item, &value, NULL);
                if(!res)
                  {
                    if(value.name[0])
                      {
                        if(doprint)
                          printf("%u\t%s\n",
                                 type_array[j],
                                 value.name);
                        else
                        if(htmlprint)
                          printf("<a name=\"typenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a></td><td><a href=\"#typerightsnettemp%u\">%s</a></td></tr>\n",
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 value.name);
                        else
                          printf("%s -V %u TYPE %u type_nettemp_name \"%s\"\n",
                                 SETPROG, RSBAC_API_VERSION_NR,
                                 type_array[j],
                                 value.name);
                      }
                  }
              }
            free(type_array);
            if(htmlprint)
              printf("</table>\n");

            item = RI_type_netobj_name;
            tid.role = 0;
            nr_netobj_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_netobj_types);
            nr_netobj_types += LISTROOM;
            type_array = malloc(nr_netobj_types * sizeof(__u32));
            if(!type_array)
              {
                error_exit(-ENOMEM);
              }
            nr_netobj_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_netobj_types, type_array, NULL);
            if(nr_netobj_types > 0)
              qsort(type_array, nr_netobj_types, sizeof(*type_array), rsbac_u32_void_compare);
            if(doprint)
              {
                printf("\n%u NETOBJ types defined:\n", nr_netobj_types);
              }
            else
            if(htmlprint)
              {
                printf("<a name=\"typenetobj\"></a><a name=\"typenetobj\"></a><h2>%u <a href=\"#typerightsnetobj\">NETOBJ</a> types defined</h2>\n", nr_netobj_types);
                printf("<table border=1>\n<tr><th>Number</th><th>Name</th></tr>\n");
              }
            for(j=0; j<nr_netobj_types; j++)
              {
                tid.type = type_array[j];
                value.dummy = -1;
                res = rsbac_rc_get_item(ta_number, target, &tid, &tid, item, &value, NULL);
                if(!res)
                  {
                    if(value.name[0])
                      {
                        if(doprint)
                          printf("%u\t%s\n",
                                 type_array[j],
                                 value.name);
                        else
                        if(htmlprint)
                          printf("<a name=\"typenetobj%u\"></a><tr><td><a href=\"#typerightsnetobj%u\">%u</a></td><td><a href=\"#typerightsnetobj%u\">%s</a></td></tr>\n",
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 type_array[j],
                                 value.name);
                        else
                          printf("%s -V %u TYPE %u type_netobj_name \"%s\"\n",
                                 SETPROG, RSBAC_API_VERSION_NR,
                                 type_array[j],
                                 value.name);
                      }
                  }
              }
            free(type_array);
            if(htmlprint)
              printf("</table>\n<hr/>\n");


            target = RT_ROLE;
            item = RI_name;
            tid.role = 0;
            nr_roles = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_name, 0, NULL, NULL);
            error_exit(nr_roles);
            role_array = malloc(nr_roles * sizeof(__u32));
            if(!role_array)
              {
                error_exit(-ENOMEM);
              }
            nr_roles = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_name, nr_roles, role_array, NULL);
            if(nr_roles > 0)
              qsort(role_array, nr_roles, sizeof(*role_array), rsbac_u32_void_compare);
            if(doprint)
              {
                printf("\nRoles\n=====\n%u Roles defined:\n", nr_roles);
              }
            else
            if(htmlprint)
              {
                printf("<a name=\"roles\"></a><h1>%u Roles defined</h1>\n", nr_roles);
              }
            for(j=0; j<nr_roles; j++)
              {
                rsbac_time_t now = time(NULL);

                tid.role = role_array[j];
                value.dummy = -1;
                res = rsbac_rc_get_item(ta_number, target, &tid, &tid, RI_name, &value, NULL);
                if(!res)
                  {
                    if(value.name[0])
                      {
                        if(remove && !doprint)
                          printf("%s -V %u ROLE %u remove_role\n",
                                 SETPROG, RSBAC_API_VERSION_NR,
                                 role_array[j]);
                        if(doprint)
                          printf("\n\n%u\t%s\n",
                                 role_array[j],
                                 value.name);
                        else
                        if(htmlprint)
                          printf("<a name=\"role%u\"></a><h2><font color=\"0000FF\">Role %u: %s</font></h2>\n<table border=1>\n",
                                 role_array[j],
                                 role_array[j],
                                 value.name);
                        else
                          printf("%s -V %u ROLE %u name \"%s\"\n",
                                 SETPROG, RSBAC_API_VERSION_NR,
                                 role_array[j],
                                 value.name);
                        /* get role_comp number */
                        sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_role_comp, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            ttl_array = malloc(sub_nr * sizeof(*ttl_array));
                            if(!ttl_array)
                              error_exit(-ENOMEM);
                            /* get values */
                            sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_role_comp, sub_nr, sub_array, ttl_array);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("Compatible roles:   ");
                                else
                                if(htmlprint)
                                  printf("<tr><td>Compatible roles</td><td>");
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    if(ttl_array[i])
                                      if(doprint)
                                        printf(" %u(ttl %us)",
                                               sub_array[i],
                                               ttl_array[i]);
                                      else
                                      if(htmlprint)
                                        printf(" <a href=\"#role%u\">%u(ttl %us)</a>",
                                               sub_array[i],
                                               sub_array[i],
                                               ttl_array[i]);
                                      else
                                        printf("%s -V %u -T %u ROLE %u role_comp %u 1\n",
                                               SETPROG, RSBAC_API_VERSION_NR,
                                               now + ttl_array[i],
                                               role_array[j],
                                               sub_array[i]);
                                    else
                                      if(doprint)
                                        printf(" %u",
                                               sub_array[i]);
                                      else
                                      if(htmlprint)
                                        printf(" <a href=\"#role%u\">%u</a>",
                                               sub_array[i],
                                               sub_array[i]);
                                      else
                                        printf("%s -V %u ROLE %u role_comp %u 1\n",
                                               SETPROG, RSBAC_API_VERSION_NR,
                                               role_array[j],
                                               sub_array[i]);
                                  }
                                if(doprint)
                                  printf("\n");
                                else
                                if(htmlprint)
                                  printf("</td></tr>\n");
                              }
                            free(sub_array);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(sub_nr);

                        /* get admin_roles number */
                        sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_admin_roles, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            ttl_array = malloc(sub_nr * sizeof(*ttl_array));
                            if(!ttl_array)
                              error_exit(-ENOMEM);
                            /* get values */
                            sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_admin_roles, sub_nr, sub_array, ttl_array);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("Administrated roles:");
                                else
                                if(htmlprint)
                                  printf("<tr><td>Administrated roles</td><td>");
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    if(ttl_array[i])
                                      if(doprint)
                                        printf(" %u(ttl %us)",
                                               sub_array[i],
                                               ttl_array[i]);
                                      else
                                      if(htmlprint)
                                        printf(" <a href=\"#role%u\">%u(ttl %us)</a>",
                                               sub_array[i],
                                               sub_array[i],
                                               ttl_array[i]);
                                      else
                                        printf("%s -V %u -T %u ROLE %u admin_roles %u 1\n",
                                               SETPROG, RSBAC_API_VERSION_NR,
                                               now + ttl_array[i],
                                               role_array[j],
                                               sub_array[i]);
                                    else
                                      if(doprint)
                                        printf(" %u",
                                               sub_array[i]);
                                      else
                                      if(htmlprint)
                                        printf(" <a href=\"#role%u\">%u</a>",
                                               sub_array[i],
                                               sub_array[i]);
                                      else
                                        printf("%s -V %u ROLE %u admin_roles %u 1\n",
                                               SETPROG, RSBAC_API_VERSION_NR,
                                               role_array[j],
                                               sub_array[i]);
                                  }
                                if(doprint)
                                  printf("\n");
                                else
                                if(htmlprint)
                                  printf("</td></tr>\n");
                              }
                            free(sub_array);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(sub_nr);

                        /* get assign_roles number */
                        sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_assign_roles, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            ttl_array = malloc(sub_nr * sizeof(*ttl_array));
                            if(!ttl_array)
                              error_exit(-ENOMEM);
                            /* get values */
                            sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_assign_roles, sub_nr, sub_array, ttl_array);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("Assignable roles:   ");
                                else
                                if(htmlprint)
                                  printf("<tr><td>Assignable roles</td><td>");
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    if(ttl_array[i])
                                      if(doprint)
                                        printf(" %u(ttl %us)",
                                               sub_array[i],
                                               ttl_array[i]);
                                      else
                                      if(htmlprint)
                                        printf(" <a href=\"#role%u\">%u(ttl %us)</a>",
                                               sub_array[i],
                                               sub_array[i],
                                               ttl_array[i]);
                                      else
                                        printf("%s -V %u -T %u ROLE %u assign_roles %u 1\n",
                                               SETPROG, RSBAC_API_VERSION_NR,
                                               now + ttl_array[i],
                                               role_array[j],
                                               sub_array[i]);
                                    else
                                      if(doprint)
                                        printf(" %u",
                                               sub_array[i]);
                                      else
                                      if(htmlprint)
                                        printf(" <a href=\"#role%u\">%u</a>",
                                               sub_array[i],
                                               sub_array[i]);
                                      else
                                        printf("%s -V %u ROLE %u assign_roles %u 1\n",
                                               SETPROG, RSBAC_API_VERSION_NR,
                                               role_array[j],
                                               sub_array[i]);
                                  }
                                if(doprint)
                                  printf("\n");
                                else
                                if(htmlprint)
                                  printf("</td></tr>\n");
                              }
                            free(sub_array);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(sub_nr);

                        res = rsbac_rc_get_item(ta_number, target, &tid, &tid, RI_admin_type, &value, NULL);
                        if(!res)
                          {
                            if(   (role_array[j]<3)
                               || value.admin_type
                              ) {
                              if(doprint)
                                printf("admin_type: %u\n",
                                       value.admin_type);
                              else
                              if(htmlprint)
                                {
                                  if(value.admin_type == RC_role_admin)
                                    printf("<tr><td>Admin Type</td><td><font color=\"#0000FF\">Role Admin</font></td></tr>");
                                  else
                                  if(value.admin_type == RC_system_admin)
                                    printf("<tr><td>Admin Type</td><td><font color=\"#008F00\">System Admin</font></td></tr>");
                                  else
                                    printf("<tr><td>Admin Type</td><td>No Admin</td></tr>");
                                }
                              else
                                printf("%s -V %u ROLE %u admin_type %u\n",
                                       SETPROG, RSBAC_API_VERSION_NR,
                                       role_array[j],
                                       value.admin_type);
                            }
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(res);
                        res = rsbac_rc_get_item(ta_number, target, &tid, &tid, RI_def_fd_create_type, &value, NULL);
                        if(!res)
                          {
                            if(doprint)
                              printf("def_fd_create_type: %u\n",
                                     value.type_id);
                            else
                            if(htmlprint)
                              {
                                printf("<tr><td>Default FD Create Type</td><td><a href=\"#typerightsfd%u\">%u</a> (%s)</td></tr>",
                                       value.type_id,
                                       value.type_id,
                                       print_type_name(tmp1, value.type_id, RI_type_fd_name));
                              }
                            else
                              printf("%s -V %u ROLE %u def_fd_create_type %u\n",
                                     SETPROG, RSBAC_API_VERSION_NR,
                                     role_array[j],
                                     value.type_id);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(res);
                        /* get def_fd_ind_create_type number */
                        sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_def_fd_ind_create_type, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            ttl_array = malloc(sub_nr * sizeof(*ttl_array));
                            if(!ttl_array)
                              error_exit(-ENOMEM);
                            /* get values */
                            sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_def_fd_ind_create_type, sub_nr, sub_array, ttl_array);
                            if(sub_nr > 0)
                              {
                                union rsbac_rc_item_value_t value3;

                                if(doprint)
                                  {
                                    if(printall)
                                      printf("def_fd_ind_create_type:\n");
                                    else
                                      printf("def_fd_ind_create_type:");
                                  }
                                else
                                if(htmlprint)
                                  {
                                    if(printall)
                                      printf("<tr><td>Default FD Ind Create Type</td><td>Parent dir type</td><td>New object type</td></tr>\n");
                                    else
                                      printf("<tr><td>Default FD Ind Create Type</td><td>");
                                  }
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_def_fd_ind_create_type, &value, NULL);
                                    if(!res)
                                      {
                                          if(printall)
                                            {
                                              if(ttl_array[i])
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                    if(!res)
                                                      {
                                                        subtid.type = value.type_id;
                                                        res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value3, NULL);
                                                        if(!res)
                                                          printf("  %u (%s) (ttl %us): %u (%s)\n",
                                                                 sub_array[i],
                                                                 value2.name,
                                                                 ttl_array[i],
                                                                 value.type_id,
                                                                 value3.name);
                                                        else
                                                          printf("  %u (%s) (ttl %us): %u\n",
                                                                 sub_array[i],
                                                                 value2.name,
                                                                 ttl_array[i],
                                                                 value.type_id);
                                                      }
                                                    else
                                                      printf("  %u (ttl %us): %u\n",
                                                             sub_array[i],
                                                             ttl_array[i],
                                                             value.type_id);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                    if(!res)
                                                      {
                                                        subtid.type = value.type_id;
                                                        res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value3, NULL);
                                                        if(!res)
                                                          printf("<tr><td></td><td><a href=\"#typerightsfd%u\">%u (%s) (ttl %us)</a></td><td><a href=\"#typerightsfd%u\">%u (%s)</a></td></tr>\n",
                                                                 sub_array[i],
                                                                 sub_array[i],
                                                                 value2.name,
                                                                 ttl_array[i],
                                                                 value.type_id,
                                                                 value.type_id,
                                                                 value3.name);
                                                        else
                                                          printf("<tr><td></td><td><a href=\"#typerightsfd%u\">%u (%s) (ttl %us)</a></td><td>%u</td></tr>\n",
                                                                 sub_array[i],
                                                                 sub_array[i],
                                                                 value2.name,
                                                                 ttl_array[i],
                                                                 value.type_id);
                                                      }
                                                    else
                                                      printf("<tr><td></td><td>%u (ttl %us)</td><td>%u</td></tr>\n",
                                                             sub_array[i],
                                                             ttl_array[i],
                                                             value.type_id);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u def_fd_ind_create_type %u %u\n",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl_array[i],
                                                         role_array[j],
                                                         sub_array[i],
                                                         value.type_id);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                    if(!res)
                                                      {
                                                        subtid.type = value.type_id;
                                                        res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value3, NULL);
                                                        if(!res)
                                                          printf("  %u (%s): %u (%s)\n",
                                                                 sub_array[i],
                                                                 value2.name,
                                                                 value.type_id,
                                                                 value3.name);
                                                        else
                                                          printf("  %u (%s): %u\n",
                                                                 sub_array[i],
                                                                 value2.name,
                                                                 value.type_id);
                                                      }
                                                    else
                                                      printf("  %u: %u\n",
                                                             sub_array[i],
                                                             value.type_id);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                    if(!res)
                                                      {
                                                        subtid.type = value.type_id;
                                                        res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value3, NULL);
                                                        if(!res)
                                                          printf("<tr><td></td><td><a href=\"#typerightsfd%u\">%u (%s)</a></td><td><a href=\"#typerightsfd%u\">%u (%s)</a></td></tr>\n",
                                                                 sub_array[i],
                                                                 sub_array[i],
                                                                 value2.name,
                                                                 value.type_id,
                                                                 value.type_id,
                                                                 value3.name);
                                                        else
                                                          printf("<tr><td></td><td><a href=\"#typerightsfd%u\">%u (%s)</a></td><td>%u</td></tr>\n",
                                                                 sub_array[i],
                                                                 sub_array[i],
                                                                 value2.name,
                                                                 value.type_id);
                                                      }
                                                    else
                                                      printf("<tr><td></td>%u</td><td>%u</td></tr>\n",
                                                             sub_array[i],
                                                             value.type_id);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u def_fd_ind_create_type %u %u\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       value.type_id);
                                            }
                                          else
                                            if(ttl_array[i])
                                              if(doprint)
                                                printf(" %u(ttl %us):%u",
                                                       sub_array[i],
                                                       ttl_array[i],
                                                       value.type_id);
                                              else
                                              if(htmlprint)
                                                {
                                                  subtid.type = sub_array[i];
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value3, NULL);
                                                  printf(" <a href=\"#typerightsfd%u\">%u(ttl %us)</a>:<a href=\"#typerightsfd%u\">%u</a>",
                                                         sub_array[i],
                                                         sub_array[i],
                                                         ttl_array[i],
                                                         value.type_id,
                                                         value.type_id);
                                                }
                                              else
                                                printf("%s -V %u -T %u ROLE %u def_fd_ind_create_type %u %u\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl_array[i],
                                                       role_array[j],
                                                       sub_array[i],
                                                       value.type_id);
                                            else
                                              if(doprint)
                                                printf(" %u:%u",
                                                       sub_array[i],
                                                       value.type_id);
                                              else
                                              if(htmlprint)
                                                printf(" <a href=\"#typerightsfd%u\">%u</a>:<a href=\"#typerightsfd%u\">%u</a>",
                                                       sub_array[i],
                                                       sub_array[i],
                                                       value.type_id,
                                                       value.type_id);
                                              else
                                                printf("%s -V %u ROLE %u def_fd_ind_create_type %u %u\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       value.type_id);
                                      }
                                  }
                                if(!printall)
                                  {
                                    if(doprint)
                                      printf("\n");
                                    else
                                    if(htmlprint)
                                      printf("</td></tr>\n");
                                  }
                              }
                            free(sub_array);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(sub_nr);

                        res = rsbac_rc_get_item(ta_number, target, &tid, &tid, RI_def_user_create_type, &value, NULL);
                        if(!res)
                          {
                            if(doprint)
                              printf("def_user_create_type: %u\n",
                                     value.type_id);
                            else
                            if(htmlprint)
                              {
                                printf("<tr><td>Default User Create Type</td><td><a href=\"#typerightsuser%u\">%u</a> (%s)</td></tr>",
                                       value.type_id,
                                       value.type_id,
                                       print_type_name(tmp1, value.type_id, RI_type_user_name));
                              }
                            else
                              printf("%s -V %u ROLE %u def_user_create_type %u\n",
                                     SETPROG, RSBAC_API_VERSION_NR,
                                     role_array[j],
                                     value.type_id);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(res);
                        res = rsbac_rc_get_item(ta_number, target, &tid, &tid, RI_def_process_create_type, &value, NULL);
                        if(!res)
                          {
                            if(doprint)
                              printf("def_process_create_type: %u\n",
                                     value.type_id);
                            else
                            if(htmlprint)
                              {
                                printf("<tr><td>Default Process Create Type</td><td><a href=\"#typerightsprocess%u\">%u</a> (%s)</td></tr>",
                                       value.type_id,
                                       value.type_id,
                                       print_type_name(tmp1, value.type_id, RI_type_process_name));
                              }
                            else
                              printf("%s -V %u ROLE %u def_process_create_type %u\n",
                                     SETPROG, RSBAC_API_VERSION_NR,
                                     role_array[j],
                                     value.type_id);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(res);
                        res = rsbac_rc_get_item(ta_number, target, &tid, &tid, RI_def_process_chown_type, &value, NULL);
                        if(!res)
                          {
                            if(doprint)
                              printf("def_process_chown_type: %u\n",
                                     value.type_id);
                            else
                            if(htmlprint)
                              {
                                printf("<tr><td>Default Process Chown Type</td><td><a href=\"#typerightsprocess%u\">%u</a> (%s)</td></tr>",
                                       value.type_id,
                                       value.type_id,
                                       print_type_name(tmp1, value.type_id, RI_type_process_name));
                              }
                            else
                              printf("%s -V %u ROLE %u def_process_chown_type %u\n",
                                     SETPROG, RSBAC_API_VERSION_NR,
                                     role_array[j],
                                     value.type_id);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(res);
                        res = rsbac_rc_get_item(ta_number, target, &tid, &tid, RI_def_process_execute_type, &value, NULL);
                        if(!res)
                          {
                            if(doprint)
                              printf("def_process_execute_type: %u\n",
                                     value.type_id);
                            else
                            if(htmlprint)
                              {
                                printf("<tr><td>Default Process Execute Type</td><td><a href=\"#typerightsprocess%u\">%u</a> (%s)</td></tr>",
                                       value.type_id,
                                       value.type_id,
                                       print_type_name(tmp1, value.type_id, RI_type_process_name));
                              }
                            else
                              printf("%s -V %u ROLE %u def_process_execute_type %u\n",
                                     SETPROG, RSBAC_API_VERSION_NR,
                                     role_array[j],
                                     value.type_id);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(res);
                        res = rsbac_rc_get_item(ta_number, target, &tid, &tid, RI_def_ipc_create_type, &value, NULL);
                        if(!res)
                          {
                            if(doprint)
                              printf("def_ipc_create_type: %u\n",
                                     value.type_id);
                            else
                            if(htmlprint)
                              {
                                printf("<tr><td>Default IPC Create Type</td><td><a href=\"#typerightsipc%u\">%u</a> (%s)</td></tr>",
                                       value.type_id,
                                       value.type_id,
                                       print_type_name(tmp1, value.type_id, RI_type_ipc_name));
                              }
                            else
                              printf("%s -V %u ROLE %u def_ipc_create_type %u\n",
                                     SETPROG, RSBAC_API_VERSION_NR,
                                     role_array[j],
                                     value.type_id);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(res);
                        res = rsbac_rc_get_item(ta_number, target, &tid, &tid, RI_def_group_create_type, &value, NULL);
                        if(!res)
                          {
                            if(doprint)
                              printf("def_group_create_type: %u\n",
                                     value.type_id);
                            else
                            if(htmlprint)
                              {
                                printf("<tr><td>Default Group Create Type</td><td><a href=\"#typerightsgroup%u\">%u</a> (%s)</td></tr>",
                                       value.type_id,
                                       value.type_id,
                                       print_type_name(tmp1, value.type_id, RI_type_group_name));
                              }
                            else
                              printf("%s -V %u ROLE %u def_group_create_type %u\n",
                                     SETPROG, RSBAC_API_VERSION_NR,
                                     role_array[j],
                                     value.type_id);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(res);
                        res = rsbac_rc_get_item(ta_number, target, &tid, &tid, RI_def_unixsock_create_type, &value, NULL);
                        if(!res)
                          {
                            if(doprint)
                              printf("def_unixsock_create_type: %u\n",
                                     value.type_id);
                            else
                            if(htmlprint)
                              {
                                printf("<tr><td>Default Unixsock Create Type</td><td><a href=\"#typerightsgroup%u\">%u</a> (%s)</td></tr>",
                                       value.type_id,
                                       value.type_id,
                                       print_type_name(tmp1, value.type_id, RI_type_netobj_name));
                              }
                            else
                              printf("%s -V %u ROLE %u def_unixsock_create_type %u\n",
                                     SETPROG, RSBAC_API_VERSION_NR,
                                     role_array[j],
                                     value.type_id);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(res);
                        res = rsbac_rc_get_item(ta_number, target, &tid, &tid, RI_boot_role, &value, NULL);
                        if(!res)
                          {
                            if(doprint)
                              printf("boot_role: %u\n",
                                     value.boot_role);
                            else
                            if(htmlprint)
                              {
                                if(value.boot_role)
                                  printf("<tr><td>Boot Role</td><td><font color=\"#0000FF\">Yes</font></td></tr>");
                                else
                                  printf("<tr><td>Boot Role</td><td>No</td></tr>");
                              }
                            else
                              if(   (role_array[j]<3)
                                 || value.boot_role
                                )
                                printf("%s -V %u ROLE %u boot_role %u\n",
                                       SETPROG, RSBAC_API_VERSION_NR,
                                       role_array[j],
                                       value.boot_role);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(res);
                        res = rsbac_rc_get_item(ta_number, target, &tid, &tid, RI_req_reauth, &value, NULL);
                        if(!res)
                          {
                            if(doprint)
                              printf("req_reauth: %u\n",
                                     value.req_reauth);
                            else
                            if(htmlprint)
                              {
                                if(value.req_reauth)
                                  printf("<tr><td>Req Reauth</td><td><font color=\"#0000FF\">Yes</font></td></tr>");
                                else
                                  printf("<tr><td>Req Reauth</td><td>No</td></tr>");
                              }
                            else
                              if(   (role_array[j]<3)
                                 || value.req_reauth
                                )
                                printf("%s -V %u ROLE %u req_reauth %u\n",
                                       SETPROG, RSBAC_API_VERSION_NR,
                                       role_array[j],
                                       value.req_reauth);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(res);
                        if(htmlprint)
                          printf("</table>\n");

                        /* get type_comp_fd number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_fd_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_fd, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_fd_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_fd, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  {
                                    printf("\nFD Type Compatibilities:\n");
                                  }
                                else
                                if(htmlprint)
                                  printf("<h3>FD Type Compatibilities of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_fd, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_fd %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_fd %u",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_FD);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_fd %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_fd %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(sub_nr);

                        /* get type_comp_fd_log_always number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_fd_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_fd_log_always, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_fd_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_fd_log_always, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  {
                                    printf("\nFD Type log always:\n");
                                  }
                                else
                                if(htmlprint)
                                  printf("<h3>FD Type log always of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_fd_log_always, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_fd_log_always %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_fd_log_always %u",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_FD);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_fd_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_fd_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        /* get type_comp_fd_log_never number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_fd_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_fd_log_never, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_fd_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_fd_log_never, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  {
                                    printf("\nFD Type log never:\n");
                                  }
                                else
                                if(htmlprint)
                                  printf("<h3>FD Type log never of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_fd_log_never, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_fd_log_never %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_fd_log_never %u",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_FD);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_fd_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypefd%u\"></a><tr><td><a href=\"#typerightsfd%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_fd_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }

                        /* get type_comp_dev number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_dev_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_dev, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_dev_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_dev, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nDEV Type Compatibilities:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>DEV Type Compatibilities of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_dev, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_dev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_dev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_dev %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_dev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_dev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_dev %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_DEV);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_dev_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_dev %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_dev_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_dev %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(sub_nr);

                        /* get type_comp_dev_log_always number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_dev_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_dev_log_always, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_dev_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_dev_log_always, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nDEV Type log always:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>DEV Type log always of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_dev_log_always, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_dev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_dev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_dev_log_always %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_dev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_dev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_dev_log_always %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_DEV);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_dev_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_dev_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_dev_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_dev_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        /* get type_comp_dev_log_never number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_dev_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_dev_log_never, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_dev_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_dev_log_never, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nDEV Type log never:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>DEV Type log never of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_dev_log_never, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_dev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_dev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_dev_log_never %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_dev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_dev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_dev_log_never %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_DEV);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_dev_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_dev_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_dev_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypedev%u\"></a><tr><td><a href=\"#typerightsdev%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_dev_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }


                        /* get type_comp_ipc number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_ipc_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_ipc, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_ipc_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_ipc, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nIPC Type Compatibilities:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>IPC Type Compatibilities of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_ipc, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_ipc_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_ipc_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_ipc %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_ipc_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_ipc_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_ipc %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_IPC);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_ipc_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_ipc %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_ipc_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_ipc %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(sub_nr);

                        /* get type_comp_ipc_log_always number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_ipc_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_ipc_log_always, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_ipc_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_ipc_log_always, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nIPC Type log always:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>IPC Type log always of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_ipc_log_always, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_ipc_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_ipc_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_ipc_log_always %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_ipc_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_ipc_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_ipc_log_always %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_IPC);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_ipc_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_ipc_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_ipc_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_ipc_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        /* get type_comp_ipc_log_never number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_ipc_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_ipc_log_never, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_ipc_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_ipc_log_never, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nIPC Type log never:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>IPC Type log never of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_ipc_log_never, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_ipc_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_ipc_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_ipc_log_never %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_ipc_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_ipc_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_ipc_log_never %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_IPC);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_ipc_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_ipc_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_ipc_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypeipc%u\"></a><tr><td><a href=\"#typerightsipc%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_ipc_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }

                        /* get type_comp_scd number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_scd_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_scd, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_scd_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_scd, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nSCD Type Compatibilities:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>SCD Type Compatibilities of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_scd, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_scd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_scd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_scd %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_scd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_scd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_scd %u",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_SCD);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_scd_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_scd %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_scd_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_scd %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(sub_nr);
                        /* get type_comp_scd_log_always number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_scd_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_scd_log_always, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_scd_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_scd_log_always, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nSCD Type log always:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>SCD Type Compatibilities of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_scd_log_always, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_scd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_scd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_scd_log_always %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_scd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_scd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_scd_log_always %u",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_SCD);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_scd_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_scd_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_scd_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_scd_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        /* get type_comp_scd_log_never number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_scd_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_scd_log_never, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_scd_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_scd_log_never, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nSCD Type log never:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>SCD Type Compatibilities of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_scd_log_never, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_scd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_scd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_scd_log_never %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_scd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_scd_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_scd_log_never %u",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_SCD);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_scd_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_scd_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_scd_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypescd%u\"></a><tr><td><a href=\"#typerightsscd%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_scd_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }

                        /* get type_comp_user number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_user_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_user, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_user_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_user, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nUSER Type Compatibilities:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>USER Type Compatibilities of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_user, &value, NULL);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_user_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_user_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_user %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_user_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_user_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_user %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_USER);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_user_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_user %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_user_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_user %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(sub_nr);
                        /* get type_comp_user_log_always number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_user_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_user_log_always, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_user_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_user_log_always, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nUSER Type log always:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>USER Type log always of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_user_log_always, &value, NULL);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_user_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_user_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_user_log_always %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_user_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_user_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_user_log_always %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_USER);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_user_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_user_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_user_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_user_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        /* get type_comp_user_log_never number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_user_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_user_log_never, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_user_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_user_log_never, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nUSER Type log never:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>USER Type log never of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_user_log_never, &value, NULL);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_user_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_user_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_user_log_never %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_user_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_user_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_user_log_never %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_USER);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_user_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_user_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_user_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypeuser%u\"></a><tr><td><a href=\"#typerightsuser%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_user_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }

                        /* get type_comp_process number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_process_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_process, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_process_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_process, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nPROCESS Type Compatibilities:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>PROCESS Type Compatibilities of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_process, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_process_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_process_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_process %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_process_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_process_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_process %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_PROCESS);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_process_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_process %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_process_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_process %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(sub_nr);
                        /* get type_comp_process_log_always number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_process_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_process_log_always, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_process_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_process_log_always, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nPROCESS Type log always:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>PROCESS Type log always of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_process_log_always, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_process_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_process_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_process_log_always %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_process_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_process_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_process_log_always %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_PROCESS);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_process_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_process_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_process_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_process_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        /* get type_comp_process_log_never number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_process_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_process_log_never, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_process_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_process_log_never, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nPROCESS Type log never:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>PROCESS Type log never of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_process_log_never, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_process_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_process_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_process_log_never %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_process_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_process_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_process_log_never %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_PROCESS);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_process_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_process_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_process_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypeprocess%u\"></a><tr><td><a href=\"#typerightsprocess%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_process_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }

                        /* get type_comp_group number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_group_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_group, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_group_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_group, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nGROUP Type Compatibilities:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>GROUP Type Compatibilities of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_group, &value, NULL);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_group_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_group_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_group %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_group_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_group_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_group %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_GROUP);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_group_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_group %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_group_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_group %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(sub_nr);
                        /* get type_comp_group_log_always number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_group_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_group_log_always, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_group_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_group_log_always, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nGROUP Type log always:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>GROUP Type log always of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_group_log_always, &value, NULL);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_group_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_group_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_group_log_always %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_group_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_group_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_group_log_always %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_GROUP);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_group_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_group_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_group_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_group_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        /* get type_comp_group_log_never number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_group_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_group_log_never, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_group_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_group_log_never, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nGROUP Type log never:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>GROUP Type log never of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_group_log_never, &value, NULL);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_group_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_group_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_group_log_never %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_group_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_group_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_group_log_never %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_GROUP);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_group_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_group_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_group_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypegroup%u\"></a><tr><td><a href=\"#typerightsgroup%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_group_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }

                        /* get type_comp_netdev number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_netdev_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_netdev, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_netdev_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_netdev, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nNETDEV Type Compatibilities:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>NETDEV Type Compatibilities of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_netdev, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netdev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netdev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_netdev %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netdev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netdev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_netdev %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_NETDEV);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netdev_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_netdev %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netdev_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_netdev %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(sub_nr);
                        /* get type_comp_netdev_log_always number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_netdev_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_netdev_log_always, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_netdev_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_netdev_log_always, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nNETDEV Type log always:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>NETDEV Type log always of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_netdev_log_always, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netdev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netdev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_netdev_log_always %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netdev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netdev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_netdev_log_always %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_NETDEV);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netdev_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_netdev_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netdev_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_netdev_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        /* get type_comp_netdev_log_never number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_netdev_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_netdev_log_never, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_netdev_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_netdev_log_never, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nNETDEV Type log never:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>NETDEV Type log never of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_netdev_log_never, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netdev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netdev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_netdev_log_never %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netdev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netdev_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_netdev_log_never %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_NETDEV);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netdev_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_netdev_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netdev_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypenetdev%u\"></a><tr><td><a href=\"#typerightsnetdev%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_netdev_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }

                        /* get type_comp_nettemp number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_nettemp_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_nettemp, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_nettemp_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_nettemp, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nNETTEMP Type Compatibilities:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>NETTEMP Type Compatibilities of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_nettemp, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_nettemp_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_nettemp_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_nettemp %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_nettemp_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_nettemp_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_nettemp %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_NETTEMP);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_nettemp_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_nettemp %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_nettemp_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_nettemp %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(sub_nr);
                        /* get type_comp_nettemp_log_always number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_nettemp_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_nettemp_log_always, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_nettemp_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_nettemp_log_always, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nNETTEMP Type log always:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>NETTEMP Type log always of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_nettemp_log_always, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_nettemp_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_nettemp_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_nettemp_log_always %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_nettemp_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_nettemp_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_nettemp_log_always %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_NETTEMP);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_nettemp_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_nettemp_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_nettemp_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_nettemp_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        /* get type_comp_nettemp_log_never number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_nettemp_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_nettemp_log_never, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_nettemp_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_nettemp_log_never, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nNETTEMP Type log never:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>NETTEMP Type log never of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_nettemp_log_never, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_nettemp_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_nettemp_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_nettemp_log_never %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_nettemp_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_nettemp_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_nettemp_log_never %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_NETTEMP);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_nettemp_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_nettemp_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_nettemp_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypenettemp%u\"></a><tr><td><a href=\"#typerightsnettemp%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_nettemp_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }

                        /* get type_comp_netobj number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_netobj_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_netobj, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_netobj_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_netobj, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nNETOBJ Type Compatibilities:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>NETOBJ Type Compatibilities of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_netobj, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netobj_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netobj_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typerightsnetobj%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typerightsnetobj%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_netobj %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netobj_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netobj_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typerightsnetobj%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typerightsnetobj%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_netobj %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_NETOBJ);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netobj_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typerightsnetobj%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typerightsnetobj%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_netobj %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netobj_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typenetobj%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typenetobj%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_netobj %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        else
                          if(!(   ((res == -1) && (errno == RSBAC_ENOTFOUND))
                               || (res == -RSBAC_ENOTFOUND)
                              )
                            )
                            show_error(sub_nr);
                        /* get type_comp_netobj_log_always number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_netobj_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_netobj_log_always, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_netobj_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_netobj_log_always, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nNETOBJ Type log always:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>NETOBJ Type log always of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_netobj_log_always, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netobj_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netobj_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typerightsnetobj%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typerightsnetobj%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_netobj_log_always %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netobj_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netobj_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typerightsnetobj%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typerightsnetobj%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_netobj_log_always %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_NETOBJ);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netobj_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typerightsnetobj%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typerightsnetobj%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_netobj_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netobj_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typenetobj%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typenetobj%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_netobj_log_always %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                        /* get type_comp_netobj_log_never number */
                        if (norights)
                          sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_netobj_name, 0, NULL, NULL);
                        else
                          sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_netobj_log_never, 0, NULL, NULL);
                        if(sub_nr >= 0)
                          {
                            /* alloc with some extra room */
                            sub_nr += LISTROOM;
                            sub_array = malloc(sub_nr * sizeof(__u32));
                            if(!sub_array)
                              {
                                error_exit(-ENOMEM);
                              }
                            /* get values */
                            if (norights)
                              sub_nr = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, RI_type_netobj_name, sub_nr, sub_array, NULL);
                            else
                              sub_nr = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_type_comp_netobj_log_never, sub_nr, sub_array, NULL);
                            if(sub_nr > 0)
                              {
                                if(doprint)
                                  printf("\nNETOBJ Type log never:\n");
                                else
                                if(htmlprint)
                                  printf("<h3>NETOBJ Type log never of Role %u:</h3>\n<table border=\"1\">\n", role_array[j]);
                                qsort(sub_array, sub_nr, sizeof(*sub_array), rsbac_u32_void_compare);
                                for (i=0; i<sub_nr; i++)
                                  {
                                    subtid.type = sub_array[i];
                                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, RI_type_comp_netobj_log_never, &value, &ttl);
                                    if(!res)
                                      {
                                        if(   norights
                                           || (   (role_array[j] < 3)
                                               && (sub_array[i] < 3)
                                              )
                                           || (   value.rights
                                              && (role_array[j] != RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                           || (   (value.rights != RSBAC_RC_SPECIAL_RIGHTS_VECTOR)
                                               && (role_array[j] == RSBAC_RC_ROLE_ADMIN_ROLE)
                                              )
                                          ) {
                                          if(printall)
                                            {
                                              if(ttl)
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netobj_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s) (ttl %us):",
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  %u (ttl %us):",
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netobj_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typerightsnetobj%u\">%u</a> (%s) (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name,
                                                             ttl);
                                                    else
                                                      printf("  <a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typerightsnetobj%u\">%u</a> (ttl %us)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             ttl);
                                                  }
                                                else
                                                  printf("%s -V %u -T %u ROLE %u type_comp_netobj_log_never %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         now + ttl,
                                                         role_array[j],
                                                         sub_array[i]);
                                              else
                                                if(doprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netobj_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  %u (%s):",
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  %u:",
                                                             sub_array[i]);
                                                  }
                                                else
                                                if(htmlprint)
                                                  {
                                                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netobj_name, &value2, NULL);
                                                    if(!res)
                                                      printf("  <a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typerightsnetobj%u\">%u</a> (%s)</td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             value2.name);
                                                    else
                                                      printf("  <a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typerightsnetobj%u\">%u</a></td><td>",
                                                             role_array[j],
                                                             sub_array[i],
                                                             sub_array[i],
                                                             sub_array[i]);
                                                  }
                                                else
                                                  printf("%s -V %u ROLE %u type_comp_netobj_log_never %u",
                                                         SETPROG, RSBAC_API_VERSION_NR,
                                                         role_array[j],
                                                         sub_array[i]);
                                              if(htmlprint)
                                                {
                                                  print_html_rights(value.rights, T_NETOBJ);
                                                  printf("</td></tr>");
                                                }
                                              else
                                                print_rights(value.rights);
                                              printf("\n");
                                              if(doprint)
                                                printf("\n");
                                            }
                                          else
                                            if(ttl)
                                              if(doprint)
                                                printf("%8u(ttl %us): %s\n",
                                                       sub_array[i],
                                                       ttl,
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netobj_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typerightsnetobj%u\">%u</a> (%s) (ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typerightsnetobj%u\">%u</a>(ttl %us)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           ttl,
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -bT %u ROLE %u type_comp_netobj_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       now + ttl,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                            else
                                              if(doprint)
                                                printf("%8u: %s\n",
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                              else
                                              if(htmlprint)
                                                {
                                                  res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netobj_name, &value2, NULL);
                                                  if(!res)
                                                    printf("<a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typenetobj%u\">%u</a> (%s)</td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           value2.name,
                                                           print_html_bitstring(tmp1,value.rights));
                                                  else
                                                    printf("<a name=\"role%utypenetobj%u\"></a><tr><td><a href=\"#typenetobj%u\">%u</a></td><td>%s</td></tr>\n",
                                                             role_array[j],
                                                             sub_array[i],
                                                           sub_array[i],
                                                           sub_array[i],
                                                           print_html_bitstring(tmp1,value.rights));
                                                }
                                              else
                                                printf("%s -V %u -b ROLE %u type_comp_netobj_log_never %u %s\n",
                                                       SETPROG, RSBAC_API_VERSION_NR,
                                                       role_array[j],
                                                       sub_array[i],
                                                       u64tostrrcr(tmp1,value.rights));
                                        }
                                      }
                                  }
                                if(htmlprint)
                                  printf("</table>\n");
                              }
                            free(sub_array);
                          }
                      }
                  }
                else
                  break;
                if(htmlprint)
                  printf("<hr/>\n");
              }
            if(!doprint && !htmlprint)
              exit(0);

            /* Rights per type */
            if(doprint)
              printf("\n\nRights to Types\n");
            else
              printf("<hr/>\n<a name=\"typerights\"></a><h1>Rights to <a href=\"#types\">Types</a></h1>\n");
            nr_roles = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_name, 0, NULL, NULL);
            error_exit(nr_roles);
            nr_roles += LISTROOM;
            role_array = malloc(nr_roles * sizeof(__u32));
            if(!role_array)
              {
                error_exit(-ENOMEM);
              }
            nr_roles = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, RI_name, nr_roles, role_array, NULL);
            if (nr_roles <= 0)
              exit (0);

            qsort(role_array, nr_roles, sizeof(*role_array), rsbac_u32_void_compare);

            item = RI_type_fd_name;
            tid.role = 0;
            nr_fd_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_fd_types);
            nr_fd_types += LISTROOM;
            type_array = malloc(nr_fd_types * sizeof(__u32));
            if(!type_array)
              error_exit(-ENOMEM);
            nr_fd_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_fd_types, type_array, NULL);
            if(nr_fd_types > 0)
              {
                qsort(type_array, nr_fd_types, sizeof(*type_array), rsbac_u32_void_compare);
                if(doprint)
                  printf("\nRights to %u FD types:\n", nr_fd_types);
                else
                  printf("<a name=\"typerightsfd\"></a><h2>Rights to %u <a href=\"#typefd\">FD</a> types</h2>\n", nr_fd_types);
                for(i=0; i<nr_fd_types; i++)
                  {
                    subtid.type = type_array[i];
                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_fd_name, &value, NULL);
                    if(res)
                      value.name[0] = 0;
                    if(doprint)
                      printf("\nType %u %s:\n", i, value.name);
                    else
                      {
                        printf("<a name=\"typerightsfd%u\"></a><h3>FD Type <a href=\"#typefd%u\">%u %s</a></a></h3>\n", subtid.type, subtid.type, subtid.type, value.name);
                        printf("<table border=1>\n<tr><th>Role</th><th>Name</th><th>Rights</th></tr>\n");
                      }
                    for(j=0; j<nr_roles; j++)
                      {
                        tid.role = role_array[j];
                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &tid, RI_name, &value, NULL);
                        if(res)
                          value.name[0] = 0;

                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &subtid, RI_type_comp_fd, &value2, &ttl);
                        if(!res && (norights || value2.rights))
                          {
                            if(doprint)
                              {
                                if(ttl)
                                  printf("%u (%s)(ttl %us):", tid.role, value.name, ttl);
                                else
                                  printf("%u(%s):", tid.role, value.name);
                                if(printall)
                                  print_rights(value2.rights);
                                else
                                  printf(" %s", u64tostrrcr(tmp1,value2.rights));
                                printf("\n");
                              }
                            else
                              {
                                if(ttl)
                                  printf("<tr><td><a href=\"#role%utypefd%u\">%u (ttl %us)</a></td><td><a href=\"#role%utypefd%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, ttl, subtid.type, tid.role, value.name);
                                else
                                  printf("<tr><td><a href=\"#role%utypefd%u\">%u</a></td><td><a href=\"#role%utypefd%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, tid.role, subtid.type, value.name);
                                if(printall)
                                  print_html_rights(value2.rights, T_FD);
                                else
                                  printf("%s", print_html_bitstring(tmp1,value2.rights));
                                printf("</td></tr>\n");
                              }
                          }
                      }
                    if(htmlprint)
                      printf("</table>\n");
                  }
                free(type_array);
              }

            item = RI_type_dev_name;
            tid.role = 0;
            nr_dev_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_dev_types);
            nr_dev_types += LISTROOM;
            type_array = malloc(nr_dev_types * sizeof(__u32));
            if(!type_array)
              error_exit(-ENOMEM);
            nr_dev_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_dev_types, type_array, NULL);
            if(nr_dev_types > 0)
              {
                qsort(type_array, nr_dev_types, sizeof(*type_array), rsbac_u32_void_compare);
                if(doprint)
                  printf("\nRights to %u DEV types:\n", nr_dev_types);
                else
                  printf("<a name=\"typerightsdev\"></a><h2>Rights to %u <a href=\"#typedev\">DEV</a> types</h2>\n", nr_dev_types);
                for(i=0; i<nr_dev_types; i++)
                  {
                    subtid.type = type_array[i];
                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_dev_name, &value, NULL);
                    if(res)
                      value.name[0] = 0;
                    if(doprint)
                      printf("\nType %u %s:\n", i, value.name);
                    else
                      {
                        printf("<a name=\"typerightsdev%u\"></a><h3>DEV Type <a href=\"#typedev%u\">%u %s</a></a></h3>\n", subtid.type, subtid.type, subtid.type, value.name);
                        printf("<table border=1>\n<tr><th>Role</th><th>Name</th><th>Rights</th></tr>\n");
                      }
                    for(j=0; j<nr_roles; j++)
                      {
                        tid.role = role_array[j];
                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &tid, RI_name, &value, NULL);
                        if(res)
                          value.name[0] = 0;

                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &subtid, RI_type_comp_dev, &value2, &ttl);
                        if(!res && (norights || value2.rights))
                          {
                            if(doprint)
                              {
                                if(ttl)
                                  printf("%u (%s)(ttl %us):", tid.role, value.name, ttl);
                                else
                                  printf("%u(%s):", tid.role, value.name);
                                if(printall)
                                  print_rights(value2.rights);
                                else
                                  printf(" %s", u64tostrrcr(tmp1,value2.rights));
                                printf("\n");
                              }
                            else
                              {
                                if(ttl)
                                  printf("<tr><td><a href=\"#role%utypedev%u\">%u (ttl %us)</a></td><td><a href=\"#role%utypedev%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, ttl, subtid.type, tid.role, value.name);
                                else
                                  printf("<tr><td><a href=\"#role%utypedev%u\">%u</a></td><td><a href=\"#role%utypedev%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, tid.role, subtid.type, value.name);
                                if(printall)
                                  print_html_rights(value2.rights, T_DEV);
                                else
                                  printf("%s", print_html_bitstring(tmp1,value2.rights));
                                printf("</td></tr>\n");
                              }
                          }
                      }
                    if(htmlprint)
                      printf("</table>\n");
                  }
                free(type_array);
              }

            item = RI_type_ipc_name;
            tid.role = 0;
            nr_ipc_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_ipc_types);
            nr_ipc_types += LISTROOM;
            type_array = malloc(nr_ipc_types * sizeof(__u32));
            if(!type_array)
              error_exit(-ENOMEM);
            nr_ipc_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_ipc_types, type_array, NULL);
            if(nr_ipc_types > 0)
              {
                qsort(type_array, nr_ipc_types, sizeof(*type_array), rsbac_u32_void_compare);
                if(doprint)
                  printf("\nRights to %u IPC types:\n", nr_ipc_types);
                else
                  printf("<a name=\"typerightsipc\"></a><h2>Rights to %u <a href=\"#typeipc\">IPC</a> types</h2>\n", nr_ipc_types);
                for(i=0; i<nr_ipc_types; i++)
                  {
                    subtid.type = type_array[i];
                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_ipc_name, &value, NULL);
                    if(res)
                      value.name[0] = 0;
                    if(doprint)
                      printf("\nType %u %s:\n", i, value.name);
                    else
                      {
                        printf("<a name=\"typerightsipc%u\"></a><h3>IPC Type <a href=\"#typeipc%u\">%u %s</a></a></h3>\n", subtid.type, subtid.type, subtid.type, value.name);
                        printf("<table border=1>\n<tr><th>Role</th><th>Name</th><th>Rights</th></tr>\n");
                      }
                    for(j=0; j<nr_roles; j++)
                      {
                        tid.role = role_array[j];
                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &tid, RI_name, &value, NULL);
                        if(res)
                          value.name[0] = 0;

                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &subtid, RI_type_comp_ipc, &value2, &ttl);
                        if(!res && (norights || value2.rights))
                          {
                            if(doprint)
                              {
                                if(ttl)
                                  printf("%u (%s)(ttl %us):", tid.role, value.name, ttl);
                                else
                                  printf("%u(%s):", tid.role, value.name);
                                if(printall)
                                  print_rights(value2.rights);
                                else
                                  printf(" %s", u64tostrrcr(tmp1,value2.rights));
                                printf("\n");
                              }
                            else
                              {
                                if(ttl)
                                  printf("<tr><td><a href=\"#role%utypeipc%u\">%u (ttl %us)</a></td><td><a href=\"#role%utypeipc%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, ttl, subtid.type, tid.role, value.name);
                                else
                                  printf("<tr><td><a href=\"#role%utypeipc%u\">%u</a></td><td><a href=\"#role%utypeipc%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, tid.role, subtid.type, value.name);
                                if(printall)
                                  print_html_rights(value2.rights, T_IPC);
                                else
                                  printf("%s", print_html_bitstring(tmp1,value2.rights));
                                printf("</td></tr>\n");
                              }
                          }
                      }
                    if(htmlprint)
                      printf("</table>\n");
                  }
                free(type_array);
              }

            item = RI_type_scd_name;
            tid.role = 0;
            nr_scd_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_scd_types);
            nr_scd_types += LISTROOM;
            type_array = malloc(nr_scd_types * sizeof(__u32));
            if(!type_array)
              error_exit(-ENOMEM);
            nr_scd_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_scd_types, type_array, NULL);
            if(nr_scd_types > 0)
              {
                qsort(type_array, nr_scd_types, sizeof(*type_array), rsbac_u32_void_compare);
                if(doprint)
                  printf("\nRights to %u SCD types:\n", nr_scd_types);
                else
                  printf("<a name=\"typerightsscd\"></a><h2>Rights to %u <a href=\"#typescd\">SCD</a> types</h2>\n", nr_scd_types);
                for(i=0; i<nr_scd_types; i++)
                  {
                    subtid.type = type_array[i];
                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_scd_name, &value, NULL);
                    if(res)
                      value.name[0] = 0;
                    if(doprint)
                      printf("\nType %u %s:\n", i, value.name);
                    else
                      {
                        printf("<a name=\"typerightsscd%u\"></a><h3>SCD Type <a href=\"#typescd%u\">%u %s</a></a></h3>\n", subtid.type, subtid.type, subtid.type, value.name);
                        printf("<table border=1>\n<tr><th>Role</th><th>Name</th><th>Rights</th></tr>\n");
                      }
                    for(j=0; j<nr_roles; j++)
                      {
                        tid.role = role_array[j];
                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &tid, RI_name, &value, NULL);
                        if(res)
                          value.name[0] = 0;

                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &subtid, RI_type_comp_scd, &value2, &ttl);
                        if(!res && (norights || value2.rights))
                          {
                            if(doprint)
                              {
                                if(ttl)
                                  printf("%u (%s)(ttl %us):", tid.role, value.name, ttl);
                                else
                                  printf("%u(%s):", tid.role, value.name);
                                if(printall)
                                  print_rights(value2.rights);
                                else
                                  printf(" %s", u64tostrrcr(tmp1,value2.rights));
                                printf("\n");
                              }
                            else
                              {
                                if(ttl)
                                  printf("<tr><td><a href=\"#role%utypescd%u\">%u (ttl %us)</a></td><td><a href=\"#role%utypescd%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, ttl, subtid.type, tid.role, value.name);
                                else
                                  printf("<tr><td><a href=\"#role%utypescd%u\">%u</a></td><td><a href=\"#role%utypescd%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, tid.role, subtid.type, value.name);
                                if(printall)
                                  print_html_rights(value2.rights, T_SCD);
                                else
                                  printf("%s", print_html_bitstring(tmp1,value2.rights));
                                printf("</td></tr>\n");
                              }
                          }
                      }
                    if(htmlprint)
                      printf("</table>\n");
                  }
                free(type_array);
              }

            item = RI_type_user_name;
            tid.role = 0;
            nr_user_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_user_types);
            nr_user_types += LISTROOM;
            type_array = malloc(nr_user_types * sizeof(__u32));
            if(!type_array)
              error_exit(-ENOMEM);
            nr_user_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_user_types, type_array, NULL);
            if(nr_user_types > 0)
              {
                qsort(type_array, nr_user_types, sizeof(*type_array), rsbac_u32_void_compare);
                if(doprint)
                  printf("\nRights to %u USER types:\n", nr_user_types);
                else
                  printf("<a name=\"typerightsuser\"></a><h2>Rights to %u <a href=\"#typeuser\">USER</a> types</h2>\n", nr_user_types);
                for(i=0; i<nr_user_types; i++)
                  {
                    subtid.type = type_array[i];
                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_user_name, &value, NULL);
                    if(res)
                      value.name[0] = 0;
                    if(doprint)
                      printf("\nType %u %s:\n", i, value.name);
                    else
                      {
                        printf("<a name=\"typerightsuser%u\"></a><h3>USER Type <a href=\"#typeuser%u\">%u %s</a></a></h3>\n", subtid.type, subtid.type, subtid.type, value.name);
                        printf("<table border=1>\n<tr><th>Role</th><th>Name</th><th>Rights</th></tr>\n");
                      }
                    for(j=0; j<nr_roles; j++)
                      {
                        tid.role = role_array[j];
                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &tid, RI_name, &value, NULL);
                        if(res)
                          value.name[0] = 0;

                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &subtid, RI_type_comp_user, &value2, &ttl);
                        if(!res && (norights || value2.rights))
                          {
                            if(doprint)
                              {
                                if(ttl)
                                  printf("%u (%s)(ttl %us):", tid.role, value.name, ttl);
                                else
                                  printf("%u(%s):", tid.role, value.name);
                                if(printall)
                                  print_rights(value2.rights);
                                else
                                  printf(" %s", u64tostrrcr(tmp1,value2.rights));
                                printf("\n");
                              }
                            else
                              {
                                if(ttl)
                                  printf("<tr><td><a href=\"#role%utypeuser%u\">%u (ttl %us)</a></td><td><a href=\"#role%utypeuser%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, ttl, subtid.type, tid.role, value.name);
                                else
                                  printf("<tr><td><a href=\"#role%utypeuser%u\">%u</a></td><td><a href=\"#role%utypeuser%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, tid.role, subtid.type, value.name);
                                if(printall)
                                  print_html_rights(value2.rights, T_USER);
                                else
                                  printf("%s", print_html_bitstring(tmp1,value2.rights));
                                printf("</td></tr>\n");
                              }
                          }
                      }
                    if(htmlprint)
                      printf("</table>\n");
                  }
                free(type_array);
              }

            item = RI_type_process_name;
            tid.role = 0;
            nr_process_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_process_types);
            nr_process_types += LISTROOM;
            type_array = malloc(nr_process_types * sizeof(__u32));
            if(!type_array)
              error_exit(-ENOMEM);
            nr_process_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_process_types, type_array, NULL);
            if(nr_process_types > 0)
              {
                qsort(type_array, nr_process_types, sizeof(*type_array), rsbac_u32_void_compare);
                if(doprint)
                  printf("\nRights to %u PROCESS types:\n", nr_process_types);
                else
                  printf("<a name=\"typerightsprocess\"></a><h2>Rights to %u <a href=\"#typeprocess\">PROCESS</a> types</h2>\n", nr_process_types);
                for(i=0; i<nr_process_types; i++)
                  {
                    subtid.type = type_array[i];
                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_process_name, &value, NULL);
                    if(res)
                      value.name[0] = 0;
                    if(doprint)
                      printf("\nType %u %s:\n", i, value.name);
                    else
                      {
                        printf("<a name=\"typerightsprocess%u\"></a><h3>PROCESS Type <a href=\"#typeprocess%u\">%u %s</a></a></h3>\n", subtid.type, subtid.type, subtid.type, value.name);
                        printf("<table border=1>\n<tr><th>Role</th><th>Name</th><th>Rights</th></tr>\n");
                      }
                    for(j=0; j<nr_roles; j++)
                      {
                        tid.role = role_array[j];
                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &tid, RI_name, &value, NULL);
                        if(res)
                          value.name[0] = 0;

                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &subtid, RI_type_comp_process, &value2, &ttl);
                        if(!res && (norights || value2.rights))
                          {
                            if(doprint)
                              {
                                if(ttl)
                                  printf("%u (%s)(ttl %us):", tid.role, value.name, ttl);
                                else
                                  printf("%u(%s):", tid.role, value.name);
                                if(printall)
                                  print_rights(value2.rights);
                                else
                                  printf(" %s", u64tostrrcr(tmp1,value2.rights));
                                printf("\n");
                              }
                            else
                              {
                                if(ttl)
                                  printf("<tr><td><a href=\"#role%utypeprocess%u\">%u (ttl %us)</a></td><td><a href=\"#role%utypeprocess%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, ttl, subtid.type, tid.role, value.name);
                                else
                                  printf("<tr><td><a href=\"#role%utypeprocess%u\">%u</a></td><td><a href=\"#role%utypeprocess%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, tid.role, subtid.type, value.name);
                                if(printall)
                                  print_html_rights(value2.rights, T_PROCESS);
                                else
                                  printf("%s", print_html_bitstring(tmp1,value2.rights));
                                printf("</td></tr>\n");
                              }
                          }
                      }
                    if(htmlprint)
                      printf("</table>\n");
                  }
                free(type_array);
              }

            item = RI_type_group_name;
            tid.role = 0;
            nr_group_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_group_types);
            nr_group_types += LISTROOM;
            type_array = malloc(nr_group_types * sizeof(__u32));
            if(!type_array)
              error_exit(-ENOMEM);
            nr_group_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_group_types, type_array, NULL);
            if(nr_group_types > 0)
              {
                qsort(type_array, nr_group_types, sizeof(*type_array), rsbac_u32_void_compare);
                if(doprint)
                  printf("\nRights to %u GROUP types:\n", nr_group_types);
                else
                  printf("<a name=\"typerightsgroup\"></a><h2>Rights to %u <a href=\"#typegroup\">GROUP</a> types</h2>\n", nr_group_types);
                for(i=0; i<nr_group_types; i++)
                  {
                    subtid.type = type_array[i];
                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_group_name, &value, NULL);
                    if(res)
                      value.name[0] = 0;
                    if(doprint)
                      printf("\nType %u %s:\n", i, value.name);
                    else
                      {
                        printf("<a name=\"typerightsgroup%u\"></a><h3>GROUP Type <a href=\"#typegroup%u\">%u %s</a></a></h3>\n", subtid.type, subtid.type, subtid.type, value.name);
                        printf("<table border=1>\n<tr><th>Role</th><th>Name</th><th>Rights</th></tr>\n");
                      }
                    for(j=0; j<nr_roles; j++)
                      {
                        tid.role = role_array[j];
                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &tid, RI_name, &value, NULL);
                        if(res)
                          value.name[0] = 0;

                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &subtid, RI_type_comp_group, &value2, &ttl);
                        if(!res && (norights || value2.rights))
                          {
                            if(doprint)
                              {
                                if(ttl)
                                  printf("%u (%s)(ttl %us):", tid.role, value.name, ttl);
                                else
                                  printf("%u(%s):", tid.role, value.name);
                                if(printall)
                                  print_rights(value2.rights);
                                else
                                  printf(" %s", u64tostrrcr(tmp1,value2.rights));
                                printf("\n");
                              }
                            else
                              {
                                if(ttl)
                                  printf("<tr><td><a href=\"#role%utypegroup%u\">%u (ttl %us)</a></td><td><a href=\"#role%utypegroup%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, ttl, subtid.type, tid.role, value.name);
                                else
                                  printf("<tr><td><a href=\"#role%utypegroup%u\">%u</a></td><td><a href=\"#role%utypegroup%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, tid.role, subtid.type, value.name);
                                if(printall)
                                  print_html_rights(value2.rights, T_GROUP);
                                else
                                  printf("%s", print_html_bitstring(tmp1,value2.rights));
                                printf("</td></tr>\n");
                              }
                          }
                      }
                    if(htmlprint)
                      printf("</table>\n");
                  }
                free(type_array);
              }

            item = RI_type_netdev_name;
            tid.role = 0;
            nr_netdev_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_netdev_types);
            nr_netdev_types += LISTROOM;
            type_array = malloc(nr_netdev_types * sizeof(__u32));
            if(!type_array)
              error_exit(-ENOMEM);
            nr_netdev_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_netdev_types, type_array, NULL);
            if(nr_netdev_types > 0)
              {
                qsort(type_array, nr_netdev_types, sizeof(*type_array), rsbac_u32_void_compare);
                if(doprint)
                  printf("\nRights to %u NETDEV types:\n", nr_netdev_types);
                else
                  printf("<a name=\"typerightsnetdev\"></a><h2>Rights to %u <a href=\"#typenetdev\">NETDEV</a> types</h2>\n", nr_netdev_types);
                for(i=0; i<nr_netdev_types; i++)
                  {
                    subtid.type = type_array[i];
                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netdev_name, &value, NULL);
                    if(res)
                      value.name[0] = 0;
                    if(doprint)
                      printf("\nType %u %s:\n", i, value.name);
                    else
                      {
                        printf("<a name=\"typerightsnetdev%u\"></a><h3>NETDEV Type <a href=\"#typenetdev%u\">%u %s</a></a></h3>\n", subtid.type, subtid.type, subtid.type, value.name);
                        printf("<table border=1>\n<tr><th>Role</th><th>Name</th><th>Rights</th></tr>\n");
                      }
                    for(j=0; j<nr_roles; j++)
                      {
                        tid.role = role_array[j];
                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &tid, RI_name, &value, NULL);
                        if(res)
                          value.name[0] = 0;

                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &subtid, RI_type_comp_netdev, &value2, &ttl);
                        if(!res && (norights || value2.rights))
                          {
                            if(doprint)
                              {
                                if(ttl)
                                  printf("%u (%s)(ttl %us):", tid.role, value.name, ttl);
                                else
                                  printf("%u(%s):", tid.role, value.name);
                                if(printall)
                                  print_rights(value2.rights);
                                else
                                  printf(" %s", u64tostrrcr(tmp1,value2.rights));
                                printf("\n");
                              }
                            else
                              {
                                if(ttl)
                                  printf("<tr><td><a href=\"#role%utypenetdev%u\">%u (ttl %us)</a></td><td><a href=\"#role%utypenetdev%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, ttl, subtid.type, tid.role, value.name);
                                else
                                  printf("<tr><td><a href=\"#role%utypenetdev%u\">%u</a></td><td><a href=\"#role%utypenetdev%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, tid.role, subtid.type, value.name);
                                if(printall)
                                  print_html_rights(value2.rights, T_NETDEV);
                                else
                                  printf("%s", print_html_bitstring(tmp1,value2.rights));
                                printf("</td></tr>\n");
                              }
                          }
                      }
                    if(htmlprint)
                      printf("</table>\n");
                  }
                free(type_array);
              }

            item = RI_type_nettemp_name;
            tid.role = 0;
            nr_nettemp_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_nettemp_types);
            nr_nettemp_types += LISTROOM;
            type_array = malloc(nr_nettemp_types * sizeof(__u32));
            if(!type_array)
              error_exit(-ENOMEM);
            nr_nettemp_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_nettemp_types, type_array, NULL);
            if(nr_nettemp_types > 0)
              {
                qsort(type_array, nr_nettemp_types, sizeof(*type_array), rsbac_u32_void_compare);
                if(doprint)
                  printf("\nRights to %u NETTEMP types:\n", nr_nettemp_types);
                else
                  printf("<a name=\"typerightsnettemp\"></a><h2>Rights to %u <a href=\"#typenettemp\">NETTEMP</a> types</h2>\n", nr_nettemp_types);
                for(i=0; i<nr_nettemp_types; i++)
                  {
                    subtid.type = type_array[i];
                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_nettemp_name, &value, NULL);
                    if(res)
                      value.name[0] = 0;
                    if(doprint)
                      printf("\nType %u %s:\n", i, value.name);
                    else
                      {
                        printf("<a name=\"typerightsnettemp%u\"></a><h3>NETTEMP Type <a href=\"#typenettemp%u\">%u %s</a></a></h3>\n", subtid.type, subtid.type, subtid.type, value.name);
                        printf("<table border=1>\n<tr><th>Role</th><th>Name</th><th>Rights</th></tr>\n");
                      }
                    for(j=0; j<nr_roles; j++)
                      {
                        tid.role = role_array[j];
                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &tid, RI_name, &value, NULL);
                        if(res)
                          value.name[0] = 0;

                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &subtid, RI_type_comp_nettemp, &value2, &ttl);
                        if(!res && (norights || value2.rights))
                          {
                            if(doprint)
                              {
                                if(ttl)
                                  printf("%u (%s)(ttl %us):", tid.role, value.name, ttl);
                                else
                                  printf("%u(%s):", tid.role, value.name);
                                if(printall)
                                  print_rights(value2.rights);
                                else
                                  printf(" %s", u64tostrrcr(tmp1,value2.rights));
                                printf("\n");
                              }
                            else
                              {
                                if(ttl)
                                  printf("<tr><td><a href=\"#role%utypenettemp%u\">%u (ttl %us)</a></td><td><a href=\"#role%utypenettemp%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, ttl, subtid.type, tid.role, value.name);
                                else
                                  printf("<tr><td><a href=\"#role%utypenettemp%u\">%u</a></td><td><a href=\"#role%utypenettemp%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, tid.role, subtid.type, value.name);
                                if(printall)
                                  print_html_rights(value2.rights, T_NETTEMP);
                                else
                                  printf("%s", print_html_bitstring(tmp1,value2.rights));
                                printf("</td></tr>\n");
                              }
                          }
                      }
                    if(htmlprint)
                      printf("</table>\n");
                  }
                free(type_array);
              }

            item = RI_type_netobj_name;
            tid.role = 0;
            nr_netobj_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, 0, NULL, NULL);
            error_exit(nr_netobj_types);
            nr_netobj_types += LISTROOM;
            type_array = malloc(nr_netobj_types * sizeof(__u32));
            if(!type_array)
              error_exit(-ENOMEM);
            nr_netobj_types = rsbac_rc_get_list(ta_number, RT_TYPE, &tid, item, nr_netobj_types, type_array, NULL);
            if(nr_netobj_types > 0)
              {
                qsort(type_array, nr_netobj_types, sizeof(*type_array), rsbac_u32_void_compare);
                if(doprint)
                  printf("\nRights to %u NETOBJ types:\n", nr_netobj_types);
                else
                  printf("<a name=\"typerightsnetobj\"></a><h2>Rights to %u <a href=\"#typenetobj\">NETOBJ</a> types</h2>\n", nr_netobj_types);
                for(i=0; i<nr_netobj_types; i++)
                  {
                    subtid.type = type_array[i];
                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &subtid, &subtid, RI_type_netobj_name, &value, NULL);
                    if(res)
                      value.name[0] = 0;
                    if(doprint)
                      printf("\nType %u %s:\n", i, value.name);
                    else
                      {
                        printf("<a name=\"typerightsnetobj%u\"></a><h3>NETOBJ Type <a href=\"#typenetobj%u\">%u %s</a></a></h3>\n", subtid.type, subtid.type, subtid.type, value.name);
                        printf("<table border=1>\n<tr><th>Role</th><th>Name</th><th>Rights</th></tr>\n");
                      }
                    for(j=0; j<nr_roles; j++)
                      {
                        tid.role = role_array[j];
                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &tid, RI_name, &value, NULL);
                        if(res)
                          value.name[0] = 0;

                        res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &subtid, RI_type_comp_netobj, &value2, &ttl);
                        if(!res && (norights || value2.rights))
                          {
                            if(doprint)
                              {
                                if(ttl)
                                  printf("%u (%s)(ttl %us):", tid.role, value.name, ttl);
                                else
                                  printf("%u(%s):", tid.role, value.name);
                                if(printall)
                                  print_rights(value2.rights);
                                else
                                  printf(" %s", u64tostrrcr(tmp1,value2.rights));
                                printf("\n");
                              }
                            else
                              {
                                if(ttl)
                                  printf("<tr><td><a href=\"#role%utypenetobj%u\">%u (ttl %us)</a></td><td><a href=\"#role%utypenetobj%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, ttl, subtid.type, tid.role, value.name);
                                else
                                  printf("<tr><td><a href=\"#role%utypenetobj%u\">%u</a></td><td><a href=\"#role%utypenetobj%u\">%s</a></td><td>", tid.role, subtid.type, tid.role, tid.role, subtid.type, value.name);
                                if(printall)
                                  print_html_rights(value2.rights, T_NETOBJ);
                                else
                                  printf("%s", print_html_bitstring(tmp1,value2.rights));
                                printf("</td></tr>\n");
                              }
                          }
                      }
                    if(htmlprint)
                      printf("</table>\n");
                  }
                free(type_array);
              }

            /* Finally, free role_array */
            free(role_array);

            if(htmlprint)
              {
                printf("<hr/>\n<a name=\"bottom\"></a><a href=\"http://www.rsbac.org\">RSBAC</a> %s RC Configuration - <a href=\"#top\">Jump to top</a>\n", RSBAC_VERSION);
                printf("</body>\n</html>\n");
              }
            exit(0);
          }
        fprintf(stderr, gettext("Invalid parameter %s\n"), argv[1]);
        exit(1);
        break;

      case 3:
        if(   !strcmp(argv[1],"list_def_fd_ind_create_types")
           || !strcmp(argv[1],"list_def_fd_ind_create_type_nr")
           || !strcmp(argv[1],"list_def_fd_ind_create_type_values")
          )
          {
            __u32 * type_array;
            int nr_types;
            int show_values = FALSE;

            target = RT_ROLE;
            tid.role = strtoul(argv[2],0,10);
            item = RI_def_fd_ind_create_type;
            if(   !strcmp(argv[1],"list_def_fd_ind_create_types")
              )
              show_names = TRUE;
            else
            if(   !strcmp(argv[1],"list_def_fd_ind_create_type_values")
              )
              show_values = TRUE;
            nr_types = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, item, 0, NULL, NULL);
            if(nr_types < 0)
              {
                if(   ((nr_types == -1) && (errno == RSBAC_ENOTFOUND))
                   || (nr_types == -RSBAC_ENOTFOUND)
                  )
                  exit(0);
                error_exit(nr_types);
              }
            nr_types += LISTROOM;
            type_array = malloc(nr_types * sizeof(__u32));
            if(!type_array)
              {
                error_exit(-ENOMEM);
              }
            nr_types = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, item, nr_types, type_array, NULL);
            if(verbose)
              {
                printf(gettext("%u types:\n"),
                       nr_types);
              }
            for(j=0; j<nr_types; j++)
              {
                if(show_names)
                  {
                    tid.type = type_array[j];
                    value.dummy = -1;
                    res = rsbac_rc_get_item(ta_number, RT_TYPE, &tid, &tid, RI_type_fd_name, &value, NULL);
                    if(!res)
                      {
                        for(i=0; i<strlen(value.name); i++)
                          if(value.name[i] == ' ')
                            value.name[i] = '_';
                        printf("%u %s\n", type_array[j], value.name);
                      }
                  }
                else
                if(show_values)
                  {
                    subtid.type = type_array[j];
                    value.dummy = -1;
                    res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, item, &value, NULL);
                    if(!res)
                      {
                        printf("%u:%u\n", type_array[j], value.type_id);
                      }
                  }
                else
                  printf("%u\n", type_array[j]);
              }
            free(type_array);
            exit(0);
          }
        break;

      case 4:
        target = get_rc_target_nr(argv[1]);
        switch(target)
          {
            case RT_ROLE:
              tid.role = strtol(argv[2],0,10);
              break;
            case RT_TYPE:
              tid.type = strtol(argv[2],0,10);
              break;
            default:
              fprintf(stderr, gettext("Invalid target %s\n"), argv[1]);
              exit(1);
           }
        if(   !strcmp(argv[3],"list_role_comp")
           || !strcmp(argv[3],"list_role_comp_nr")
           || !strcmp(argv[3],"list_admin_roles")
           || !strcmp(argv[3],"list_admin_role_nr")
           || !strcmp(argv[3],"list_assign_roles")
           || !strcmp(argv[3],"list_assign_role_nr")
          )
          {
            __u32 * role_array;
            rsbac_time_t * ttl_array;
            int nr_roles;

            if(   !strcmp(argv[3],"list_role_comp")
               || !strcmp(argv[3],"list_role_comp_nr")
              )
              item = RI_role_comp;
            else
            if(   !strcmp(argv[3],"list_admin_roles")
               || !strcmp(argv[3],"list_admin_role_nr")
              )
              item = RI_admin_roles;
            else
            if(   !strcmp(argv[3],"list_assign_roles")
               || !strcmp(argv[3],"list_assign_role_nr")
              )
              item = RI_assign_roles;

            if(   !strcmp(argv[3],"list_role_comp")
               || !strcmp(argv[3],"list_admin_roles")
               || !strcmp(argv[3],"list_assign_roles")
              )
              show_names = TRUE;

            nr_roles = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, item, 0, NULL, NULL);
            error_exit(nr_roles);
            nr_roles += LISTROOM;
            role_array = malloc(nr_roles * sizeof(__u32));
            if(!role_array)
              {
                error_exit(-ENOMEM);
              }
            ttl_array = malloc(nr_roles * sizeof(*ttl_array));
            if(!ttl_array)
              {
                error_exit(-ENOMEM);
              }
            nr_roles = rsbac_rc_get_list(ta_number, RT_ROLE, &tid, item, nr_roles, role_array, ttl_array);
            if(verbose)
              {
                printf(gettext("%u roles:\n"),
                       nr_roles);
              }
            for(j=0; j<nr_roles; j++)
              {
                if(show_names)
                  {
                    tid.role = role_array[j];
                    value.dummy = -1;
                    res = rsbac_rc_get_item(ta_number, RT_ROLE, &tid, &tid, RI_name, &value, NULL);
                    if(!res)
                      {
                        for(i=0; i<strlen(value.name); i++)
                          if(value.name[i] == ' ')
                            value.name[i] = '_';
                        if(ttl_array[j])
                          printf("%u(ttl:%us) %s\n", role_array[j], ttl_array[j], value.name);
                        else
                          printf("%u %s\n", role_array[j], value.name);
                      }
                  }
                else
                  {
                    if(ttl_array[j])
                      printf("%u(ttl:%us)\n", role_array[j], ttl_array[j]);
                    else
                      printf("%u\n", role_array[j]);
                  }
              }
            free(role_array);
            exit(0);
          }
        item = get_rc_item_nr(argv[3]);
        switch (item)
          {
            case RI_name:
            case RI_type_fd_name:
            case RI_type_fd_need_secdel:
            case RI_type_dev_name:
            case RI_type_ipc_name:
            case RI_type_user_name:
            case RI_type_process_name:
            case RI_type_group_name:
            case RI_type_netdev_name:
            case RI_type_nettemp_name:
            case RI_type_netobj_name:
            case RI_type_scd_name:
            case RI_admin_type:
            case RI_boot_role:
	    case RI_req_reauth:
            case RI_def_fd_create_type:
            case RI_def_user_create_type:
            case RI_def_process_create_type:
            case RI_def_process_chown_type:
            case RI_def_process_execute_type:
            case RI_def_ipc_create_type:
            case RI_def_group_create_type:
            case RI_def_unixsock_create_type:
              break;
            default:
              fprintf(stderr, gettext("Invalid item %s or too few arguments\n"), argv[3]);
              exit(1);
          }
        value.dummy = -1;
        res = rsbac_rc_get_item(ta_number, target, &tid, &tid, item, &value, &ttl);
        error_exit(res);
        switch (item)
          {
            case RI_name:
            case RI_type_fd_name:
            case RI_type_dev_name:
            case RI_type_ipc_name:
            case RI_type_user_name:
            case RI_type_process_name:
            case RI_type_group_name:
            case RI_type_netdev_name:
            case RI_type_nettemp_name:
            case RI_type_netobj_name:
            case RI_type_scd_name:
              printf("%s\n",value.name);
              break;
            case RI_admin_type:
              printf("%u\n",value.admin_type);
              break;
            case RI_boot_role:
              printf("%u\n",value.boot_role);
	    case RI_req_reauth:
	      printf("%u\n",value.req_reauth);
              break;
            default:
              printf("%u\n",value.type_id);
          }
        exit(0);
        break;

      case 5:
        target = get_rc_target_nr(argv[1]);
        switch(target)
          {
            case RT_ROLE:
              tid.role = strtol(argv[2],0,10);
              break;
            case RT_TYPE:
              tid.type = strtol(argv[2],0,10);
              break;
            default:
              fprintf(stderr, gettext("Invalid target %s\n"), argv[1]);
              exit(1);
           }
        item = get_rc_item_nr(argv[3]);
        switch (item)
          {
            case RI_role_comp:
            case RI_admin_roles:
            case RI_assign_roles:
            case RI_def_fd_ind_create_type:
            case RI_type_comp_fd:
            case RI_type_comp_dev:
            case RI_type_comp_user:
            case RI_type_comp_process:
            case RI_type_comp_ipc:
            case RI_type_comp_scd:
            case RI_type_comp_group:
            case RI_type_comp_netdev:
            case RI_type_comp_nettemp:
            case RI_type_comp_netobj:
            case RI_type_comp_fd_log_always:
            case RI_type_comp_dev_log_always:
            case RI_type_comp_user_log_always:
            case RI_type_comp_process_log_always:
            case RI_type_comp_ipc_log_always:
            case RI_type_comp_scd_log_always:
            case RI_type_comp_group_log_always:
            case RI_type_comp_netdev_log_always:
            case RI_type_comp_nettemp_log_always:
            case RI_type_comp_netobj_log_always:
            case RI_type_comp_fd_log_never:
            case RI_type_comp_dev_log_never:
            case RI_type_comp_user_log_never:
            case RI_type_comp_process_log_never:
            case RI_type_comp_ipc_log_never:
            case RI_type_comp_scd_log_never:
            case RI_type_comp_group_log_never:
            case RI_type_comp_netdev_log_never:
            case RI_type_comp_nettemp_log_never:
            case RI_type_comp_netobj_log_never:
              break;
            default:
              fprintf(stderr, gettext("Invalid item %s or invalid number of arguments\n"), argv[3]);
              exit(1);
          }
        if(target == RT_ROLE)
          {
            subtid.role = strtoul(argv[4],0,10);
#if 0
            if(subtid.role > RC_role_max_value)
              {
                fprintf(stderr, gettext("Invalid subrole %s\n"), argv[4]);
                exit(1);
              }
#endif
          }
        else
          {
            subtid.type = strtoul(argv[4],0,10);
            if(subtid.type > RC_type_max_value)
              {
                fprintf(stderr, gettext("Invalid subtype %s\n"), argv[4]);
                exit(1);
              }
          }
        value.dummy = -1;
        res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, item, &value, &ttl);
        error_exit(res);
        switch (item)
          {
            case RI_role_comp:
            case RI_admin_roles:
            case RI_assign_roles:
              if(verbose)
                {
                  printf(gettext("Getting %s for ROLE %u to ROLE %u\n"),
                         get_rc_item_name(tmp1, item), tid.role, subtid.role);
                }
              if(ttl)
                printf("%u (ttl: %us)\n", value.comp, ttl);
              else
                printf("%u\n", value.comp);
              break;
            case RI_def_fd_ind_create_type:
              if(verbose)
                {
                  printf(gettext("Getting def_fd_ind_create_type for ROLE %u to TYPE %u\n"),
                         tid.role, subtid.type);
                }
              if(ttl)
                printf("%u (ttl: %us)\n",value.type_id, ttl);
              else
                printf("%u\n",value.type_id);
              break;

            case RI_type_comp_fd:
            case RI_type_comp_dev:
            case RI_type_comp_user:
            case RI_type_comp_process:
            case RI_type_comp_ipc:
            case RI_type_comp_scd:
            case RI_type_comp_group:
            case RI_type_comp_netdev:
            case RI_type_comp_nettemp:
            case RI_type_comp_netobj:
            case RI_type_comp_fd_log_always:
            case RI_type_comp_dev_log_always:
            case RI_type_comp_user_log_always:
            case RI_type_comp_process_log_always:
            case RI_type_comp_ipc_log_always:
            case RI_type_comp_scd_log_always:
            case RI_type_comp_group_log_always:
            case RI_type_comp_netdev_log_always:
            case RI_type_comp_nettemp_log_always:
            case RI_type_comp_netobj_log_always:
            case RI_type_comp_fd_log_never:
            case RI_type_comp_dev_log_never:
            case RI_type_comp_user_log_never:
            case RI_type_comp_process_log_never:
            case RI_type_comp_ipc_log_never:
            case RI_type_comp_scd_log_never:
            case RI_type_comp_group_log_never:
            case RI_type_comp_netdev_log_never:
            case RI_type_comp_nettemp_log_never:
            case RI_type_comp_netobj_log_never:
              if(verbose)
                {
                  printf(gettext("Getting %s rights for ROLE %u to TYPE %u\n"),
                         get_rc_item_name(tmp1, item), tid.role, subtid.type);
                }
              if(ttl)
                printf("%s (ttl: %us)\n",u64tostrrcr(tmp1,value.rights), ttl);
              else
                printf("%s\n",u64tostrrcr(tmp1,value.rights));
              if(printall)
                {
                  for (i=0; i<R_NONE; i++)
                    if(value.rights & RSBAC_RC_RIGHTS_VECTOR(i))
                      printf("  %s\n", get_request_name(tmp1,i));
                  for (i=RSBAC_RC_SPECIAL_RIGHT_BASE; i<RCR_NONE; i++)
                    if(value.rights & RSBAC_RC_RIGHTS_VECTOR(i))
                      printf("  %s\n", get_rc_special_right_name(tmp1,i));
                }
              break;

            default:
              printf("%u\n",value.type_id);
          }
        exit(0);
        break;

      case 6:
        target = get_rc_target_nr(argv[1]);
        switch(target)
          {
            case RT_ROLE:
              tid.role = strtol(argv[2],0,10);
              break;
            case RT_TYPE:
              tid.type = strtol(argv[2],0,10);
              break;
            default:
              fprintf(stderr, gettext("Invalid target %s\n"), argv[1]);
              exit(1);
          }
        item = get_rc_item_nr(argv[3]);
        switch (item)
          {
            case RI_type_comp_fd:
            case RI_type_comp_dev:
            case RI_type_comp_user:
            case RI_type_comp_process:
            case RI_type_comp_ipc:
            case RI_type_comp_scd:
            case RI_type_comp_group:
            case RI_type_comp_netdev:
            case RI_type_comp_nettemp:
            case RI_type_comp_netobj:
            case RI_type_comp_fd_log_always:
            case RI_type_comp_dev_log_always:
            case RI_type_comp_user_log_always:
            case RI_type_comp_process_log_always:
            case RI_type_comp_ipc_log_always:
            case RI_type_comp_scd_log_always:
            case RI_type_comp_group_log_always:
            case RI_type_comp_netdev_log_always:
            case RI_type_comp_nettemp_log_always:
            case RI_type_comp_netobj_log_always:
            case RI_type_comp_fd_log_never:
            case RI_type_comp_dev_log_never:
            case RI_type_comp_user_log_never:
            case RI_type_comp_process_log_never:
            case RI_type_comp_ipc_log_never:
            case RI_type_comp_scd_log_never:
            case RI_type_comp_group_log_never:
            case RI_type_comp_netdev_log_never:
            case RI_type_comp_nettemp_log_never:
            case RI_type_comp_netobj_log_never:
              break;
            default:
              fprintf(stderr, gettext("Invalid item-position combination %s\n"), argv[3]);
              exit(1);
          }
        subtid.type = strtol(argv[4],0,10);
        if(   (subtid.type < 0)
           || (subtid.type > RC_type_max_value)
          )
          {
            fprintf(stderr, gettext("Invalid comp_type %s\n"), argv[4]);
            exit(1);
          }
        rc_right = get_rc_special_right_nr(argv[5]);
        if(   (rc_right == R_NONE)
           || (((enum rsbac_rc_special_rights_t) rc_right) == RCR_NONE)
          )
          {
            rc_right = strtol(argv[5],0,10);
            if(   (((enum rsbac_rc_special_rights_t) rc_right) >= RCR_NONE)
               || (   (rc_right == 0)
                   && strcmp(argv[5],"0")
                  )
              )
              {
                fprintf(stderr, gettext("Invalid right %s\n"), argv[4]);
                exit(1);
              }
          }

        value.dummy = -1;
        res = rsbac_rc_get_item(ta_number, target, &tid, &subtid, item, &value, &ttl);
        error_exit(res);
        switch (item)
          {
            case RI_type_comp_fd:
            case RI_type_comp_dev:
            case RI_type_comp_user:
            case RI_type_comp_process:
            case RI_type_comp_ipc:
            case RI_type_comp_scd:
            case RI_type_comp_group:
            case RI_type_comp_netdev:
            case RI_type_comp_nettemp:
            case RI_type_comp_netobj:
            case RI_type_comp_fd_log_always:
            case RI_type_comp_dev_log_always:
            case RI_type_comp_user_log_always:
            case RI_type_comp_process_log_always:
            case RI_type_comp_ipc_log_always:
            case RI_type_comp_scd_log_always:
            case RI_type_comp_group_log_always:
            case RI_type_comp_netdev_log_always:
            case RI_type_comp_nettemp_log_always:
            case RI_type_comp_netobj_log_always:
            case RI_type_comp_fd_log_never:
            case RI_type_comp_dev_log_never:
            case RI_type_comp_user_log_never:
            case RI_type_comp_process_log_never:
            case RI_type_comp_ipc_log_never:
            case RI_type_comp_scd_log_never:
            case RI_type_comp_group_log_never:
            case RI_type_comp_netdev_log_never:
            case RI_type_comp_nettemp_log_never:
            case RI_type_comp_netobj_log_never:
              if(ttl)
                {
                  if(value.rights & RSBAC_RC_RIGHTS_VECTOR(rc_right))
                    printf("1 (ttl: %us)\n", ttl);
                  else
                    printf("0 (ttl: %us)\n", ttl);
                }
              else
                {
                  if(value.rights & RSBAC_RC_RIGHTS_VECTOR(rc_right))
                    printf("1\n");
                  else
                    printf("0\n");
                }
              break;
            default:
              printf("ERROR!");
              exit(1);
          }
        exit(0);
        break;

      default:
        use();
        return 1;
    }
  exit(1);
}
