/*************************************************** */
/* Rule Set Based Access Control                     */
/*                                                   */
/* Author and (c) 1999-2010: Amon Ott <ao@rsbac.org> */
/*                                                   */
/* Last modified: 25/Jan/2010                        */
/*************************************************** */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <rsbac/types.h>
#include <rsbac/syscalls.h>
#include <rsbac/error.h>
#include "nls.h"
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

char * progname;
int verbose = 0;
rsbac_list_ta_number_t ta_number = 0;
rsbac_um_set_t vset = RSBAC_UM_VIRTUAL_KEEP;

void use(void)
    {
      printf(gettext("%s (RSBAC %s)\n***\n"), progname, VERSION);
      printf(gettext("Use: %s [flags] group [group2 ...]\n"), progname);
      printf(gettext(" -h = this help, -- = no more flags,\n"));
      printf(gettext(" -v = verbose,\n"));
      printf(gettext(" -S n = virtual user set n\n"));
      printf(gettext(" -N ta = transaction number (default = value of RSBAC_TA, if set, or 0)\n"));
    }

int process(char * name)
  {
    int res = 0;
    rsbac_gid_t group = RSBAC_GEN_GID(vset, RSBAC_NO_GROUP);

    if((res = rsbac_um_get_gid(ta_number, name, &group)))
      {
          char * p = name;
          rsbac_um_set_t tmp_vset = vset;

          while (*p && (*p != '/'))
            p++;
          if (*p) {
                    *p = 0;
                    tmp_vset = strtoul(name, NULL, 0);
                    *p = '/';
                    p++;
                    name = p;
          }
          group = strtoul(name, NULL, 0);
          if(!group && strcmp(name,"0"))
            {
              fprintf(stderr, gettext("%s: Unknown group %s\n"), progname, name);
              return 1;
            }
          group = RSBAC_GEN_GID(tmp_vset, group);
      }
    if(verbose) {
      if (vset != RSBAC_UM_VIRTUAL_KEEP)
        printf("Deleting group %s, gid %u/%u\n",
               name, RSBAC_GID_SET(group), RSBAC_GID_NUM(group));
      else
        printf("Deleting group %s, gid %u\n", name, RSBAC_GID_NUM(group));
    }
    res = rsbac_um_remove_group(ta_number, group);
    if(res)
      {
        if (vset != RSBAC_UM_VIRTUAL_KEEP)
          fprintf(stderr, "%u/%s: ", RSBAC_GID_SET(group),name);
        else
          fprintf(stderr, "%s: ", name);
        show_error(res);
        return res;
      }
    return 0;
  }

int main(int argc, char ** argv)
{
  int res = 0;
  int verbose = 0;
  u_int stopflags = FALSE;

  locale_init();

  progname = argv[0];
  {
    char * env = getenv("RSBAC_TA");

    if(env)
      ta_number = strtoul(env,0,0);
  }
  while((argc > 1) && (argv[1][0] == '-') && !stopflags)
    {
      char * pos = argv[1];
      pos++;
      while(*pos)
        {
          switch(*pos)
            {
              case '-':
                stopflags = TRUE;
                break;
              case 'h':
                use();
                return 0;
              case 'v':
                verbose++;
                break;
              case 'N':
                if(argc > 2)
                  {
                    ta_number = strtoul(argv[2], 0, 10);
                    argc--;
                    argv++;
                  }
                else
                  {
                    fprintf(stderr, gettext("%s: missing transaction number value for parameter %c\n"), progname, *pos);
                    exit(1);
                  }
                break;
              case 'S':
                if(argc > 2)
                  {
                    if (rsbac_get_vset_num(argv[2], &vset))
                      {
                        fprintf(stderr, gettext("%s: invalid virtual set number for parameter %c\n"), progname, *pos);
                        exit(1);
                      }
                    argc--;
                    argv++;
                  }
                else
                  {
                    fprintf(stderr, gettext("%s: missing virtual set number for parameter %c\n"), progname, *pos);
                    exit(1);
                  }
                break;

              default:
                fprintf(stderr, gettext("%s: unknown parameter %c\n"), progname, *pos);
                exit(1);
            }
          pos++;
        }
      argv++;
      argc--;
    }

  if (argc > 1)
    {
      int i;

      for(i=1; i< argc; i++)
        process(argv[i]);
      exit(0);
    }
  else
    {
      use();
      return 1;
    }
  return (res);
}
